#include "StdAfx.h"
#include "PTFSUtil.h"
#include "PTFSCtl.h"
#include "util/util.h"
#include "List/List.h"

#define DOS_STAR                        (L'<')
#define DOS_QM                          (L'>')
#define DOS_DOT                         (L'"')


CRITICAL_SECTION	CPTFSUtil::m_InstanceCriticalSection;
LIST_ENTRY	CPTFSUtil::m_InstanceList;	

CPTFSUtil::CPTFSUtil(void)
{
}

CPTFSUtil::~CPTFSUtil(void)
{
}

void CPTFSUtil::InitializeResource()
{
#if _MSC_VER < 1300
	InitializeCriticalSection(&m_InstanceCriticalSection);
#else
	InitializeCriticalSectionAndSpinCount(&m_InstanceCriticalSection, 0x80000400);
#endif			
	InitializeListHead(&m_InstanceList);
}

void CPTFSUtil::DeleteResource()
{
	EnterCriticalSection(&m_InstanceCriticalSection);

	while(!IsListEmpty(&m_InstanceList)) {
		PLIST_ENTRY pEntry = RemoveHeadList(&m_InstanceList);
		PPTFS_INSTANCE pInstance =CONTAINING_RECORD(pEntry, PTFS_INSTANCE, ListEntry);	
		CPTFSCtl::RemoveMountPoint(pInstance->MountPoint, TRUE);
		free(pInstance);
	}

	LeaveCriticalSection(&m_InstanceCriticalSection);
	DeleteCriticalSection(&m_InstanceCriticalSection);
}

PPTFS_INSTANCE CPTFSUtil::NewPTFSInstance()
{
	PPTFS_INSTANCE pInstance = (PPTFS_INSTANCE)malloc(sizeof(PTFS_INSTANCE));
	ZeroMemory(pInstance, sizeof(PTFS_INSTANCE));

#if _MSC_VER < 1300
	InitializeCriticalSection(&pInstance->CriticalSection);
#else
	InitializeCriticalSectionAndSpinCount( &pInstance->CriticalSection, 0x80000400);
#endif

	InitializeListHead(&pInstance->ListEntry);

	EnterCriticalSection(&m_InstanceCriticalSection);
	InsertTailList(&m_InstanceList, &pInstance->ListEntry);
	LeaveCriticalSection(&m_InstanceCriticalSection);

	return pInstance;
}

VOID CPTFSUtil::DeletePTFSInstance(PPTFS_INSTANCE pInstance)
{
	DeleteCriticalSection(&pInstance->CriticalSection);

	EnterCriticalSection(&m_InstanceCriticalSection);
	RemoveEntryList(&pInstance->ListEntry);
	LeaveCriticalSection(&m_InstanceCriticalSection);

	free(pInstance);
}

BOOL CPTFSUtil::IsValidDriveLetter(WCHAR wDriveLetter)
{
	return (L'd' <= wDriveLetter && wDriveLetter <= L'z') ||
		(L'D' <= wDriveLetter && wDriveLetter <= L'Z');
}

int CPTFSUtil::CheckMountPoint(LPCWSTR lpcwszMountPoint)
{
	ULONG ulLength = (ULONG)wcslen(lpcwszMountPoint);

	if ((ulLength == 1) ||
		(ulLength == 2 && lpcwszMountPoint[1] == L':') ||
		(ulLength == 3 && lpcwszMountPoint[1] == L':' && lpcwszMountPoint[2] == L'\\')) {

		WCHAR wDriveLetter = lpcwszMountPoint[0];
		
		if (IsValidDriveLetter(wDriveLetter)) {
			return PTFS_SUCCESS;
		} else {
			CLogUtil::DebugLogString(DLEVEL_ERROR, _T("PTFS Error: bad drive letter %s\n"), lpcwszMountPoint);
			return PTFS_DRIVE_LETTER_ERROR;
		}
	} else if (ulLength > 3) {
		HANDLE hDevice = CreateFile( lpcwszMountPoint, GENERIC_WRITE, 0, NULL, OPEN_EXISTING,
						FILE_FLAG_OPEN_REPARSE_POINT | FILE_FLAG_BACKUP_SEMANTICS, NULL);
		if (hDevice == INVALID_HANDLE_VALUE) {
			CLogUtil::DebugLogString(DLEVEL_ERROR, _T("PTFS Error: bad mount point %s\n"), lpcwszMountPoint);
			return PTFS_MOUNT_POINT_ERROR;
		}
		CloseHandle(hDevice);
		return PTFS_SUCCESS;
	}
	return PTFS_MOUNT_POINT_ERROR;
}


BOOL CPTFSUtil::IsNameInExpression( LPCWSTR lpcwszExpression, LPCWSTR	lpcwszName, BOOL	bIgnoreCase)
{
	ULONG ei = 0;
	ULONG ni = 0;

	while (lpcwszExpression[ei] != '\0') {

		if (lpcwszExpression[ei] == L'*') {
			ei++;
			if (lpcwszExpression[ei] == '\0')
				return TRUE;

			while (lpcwszName[ni] != '\0') {
				if (IsNameInExpression(&lpcwszExpression[ei], &lpcwszName[ni], bIgnoreCase))
					return TRUE;
				ni++;
			}

		} else if (lpcwszExpression[ei] == DOS_STAR) {

			ULONG p = ni;
			ULONG lastDot = 0;
			ei++;
			
			while (lpcwszName[p] != '\0') {
				if (lpcwszName[p] == L'.')
					lastDot = p;
				p++;
			}
			

			while (TRUE) {
				if (lpcwszName[ni] == '\0' || ni == lastDot)
					break;

				if (IsNameInExpression(&lpcwszExpression[ei], &lpcwszName[ni], bIgnoreCase))
					return TRUE;
				ni++;
			}
			
		} else if (lpcwszExpression[ei] == DOS_QM)  {
			
			ei++;
			if (lpcwszName[ni] != L'.') {
				ni++;
			} else {

				ULONG p = ni + 1;
				while (lpcwszName[p] != '\0') {
					if (lpcwszName[p] == L'.')
						break;
					p++;
				}

				if (lpcwszName[p] == L'.')
					ni++;
			}

		} else if (lpcwszExpression[ei] == DOS_DOT) {
			ei++;

			if (lpcwszName[ni] == L'.')
				ni++;

		} else {
			if (lpcwszExpression[ei] == L'?') {
				ei++; ni++;
			} else if(bIgnoreCase && towupper(lpcwszExpression[ei]) == towupper(lpcwszName[ni])) {
				ei++; ni++;
			} else if(!bIgnoreCase && lpcwszExpression[ei] == lpcwszName[ni]) {
				ei++; ni++;
			} else {
				return FALSE;
			}
		}
	}

	if (ei == wcslen(lpcwszExpression) && ni == wcslen(lpcwszName))
		return TRUE;
	

	return FALSE;

}

LPCWSTR CPTFSUtil::GetRawDeviceName(LPCWSTR	lpcwszDeviceName)
{
	static WCHAR wszRawDeviceName[MAX_PATH];
	wcscpy_s(wszRawDeviceName, MAX_PATH, L"\\\\.");
	wcscat_s(wszRawDeviceName, MAX_PATH, lpcwszDeviceName);
	return wszRawDeviceName;
}

VOID CPTFSUtil::CheckFileName(LPWSTR	lpwszFileName)
{
	if (lpwszFileName[0] == L'\\' && lpwszFileName[1] == L'\\') {
		int i;
		for (i = 0; lpwszFileName[i+1] != L'\0'; ++i) {
			lpwszFileName[i] = lpwszFileName[i+1];
		}
		lpwszFileName[i] = L'\0';
	}
}

ULONG CPTFSUtil::GetNTStatus(DWORD dwErrorCode)
{
	switch (dwErrorCode) {
	case ERROR_DIR_NOT_EMPTY:
		return STATUS_DIRECTORY_NOT_EMPTY;
	case ERROR_ACCESS_DENIED:
		return STATUS_ACCESS_DENIED;
	case ERROR_SHARING_VIOLATION:
		return STATUS_SHARING_VIOLATION;
	case ERROR_INVALID_NAME:
		return STATUS_OBJECT_NAME_NOT_FOUND;
	case ERROR_ALREADY_EXISTS:
		return STATUS_OBJECT_NAME_COLLISION;
	case ERROR_DISK_FULL:
		return STATUS_DISK_FULL;
	default:
		return STATUS_OBJECT_NAME_NOT_FOUND;
	}
}

///////////////////////////////////////////////////////////////////////
VOID CPTFSUtil::FillDirInfo(PFILE_DIRECTORY_INFORMATION pBuffer, PWIN32_FIND_DATAW pFindData, ULONG ulIndex)
{
	ULONG ulNameBytes = (ULONG)wcslen(pFindData->cFileName) * sizeof(WCHAR);

	pBuffer->FileIndex = ulIndex;
	pBuffer->FileAttributes = pFindData->dwFileAttributes;
	pBuffer->FileNameLength = ulNameBytes;

	pBuffer->EndOfFile.HighPart = pFindData->nFileSizeHigh;
	pBuffer->EndOfFile.LowPart   = pFindData->nFileSizeLow;
	pBuffer->AllocationSize.HighPart = pFindData->nFileSizeHigh;
	pBuffer->AllocationSize.LowPart = pFindData->nFileSizeLow;

	pBuffer->CreationTime.HighPart = pFindData->ftCreationTime.dwHighDateTime;
	pBuffer->CreationTime.LowPart  = pFindData->ftCreationTime.dwLowDateTime;

	pBuffer->LastAccessTime.HighPart = pFindData->ftLastAccessTime.dwHighDateTime;
	pBuffer->LastAccessTime.LowPart  = pFindData->ftLastAccessTime.dwLowDateTime;

	pBuffer->LastWriteTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->LastWriteTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	pBuffer->ChangeTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->ChangeTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	RtlCopyMemory(pBuffer->FileName, pFindData->cFileName, ulNameBytes);
}

VOID CPTFSUtil::FillFullDirInfo(PFILE_FULL_DIR_INFORMATION pBuffer, PWIN32_FIND_DATAW pFindData, ULONG ulIndex)
{
	ULONG ulNameBytes = (ULONG)wcslen(pFindData->cFileName) * sizeof(WCHAR);

	pBuffer->FileIndex = ulIndex;
	pBuffer->FileAttributes = pFindData->dwFileAttributes;
	pBuffer->FileNameLength = ulNameBytes;

	pBuffer->EndOfFile.HighPart = pFindData->nFileSizeHigh;
	pBuffer->EndOfFile.LowPart   = pFindData->nFileSizeLow;
	pBuffer->AllocationSize.HighPart = pFindData->nFileSizeHigh;
	pBuffer->AllocationSize.LowPart = pFindData->nFileSizeLow;

	pBuffer->CreationTime.HighPart = pFindData->ftCreationTime.dwHighDateTime;
	pBuffer->CreationTime.LowPart  = pFindData->ftCreationTime.dwLowDateTime;

	pBuffer->LastAccessTime.HighPart = pFindData->ftLastAccessTime.dwHighDateTime;
	pBuffer->LastAccessTime.LowPart  = pFindData->ftLastAccessTime.dwLowDateTime;

	pBuffer->LastWriteTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->LastWriteTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	pBuffer->ChangeTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->ChangeTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	pBuffer->EaSize = 0;

	RtlCopyMemory(pBuffer->FileName, pFindData->cFileName, ulNameBytes);
}

VOID CPTFSUtil::FillIdFullDirInfo(PFILE_ID_FULL_DIR_INFORMATION pBuffer, PWIN32_FIND_DATAW pFindData, ULONG ulIndex)
{
	ULONG ulNameBytes = (ULONG)wcslen(pFindData->cFileName) * sizeof(WCHAR);

	pBuffer->FileIndex = ulIndex;
	pBuffer->FileAttributes = pFindData->dwFileAttributes;
	pBuffer->FileNameLength = ulNameBytes;

	pBuffer->EndOfFile.HighPart = pFindData->nFileSizeHigh;
	pBuffer->EndOfFile.LowPart   = pFindData->nFileSizeLow;
	pBuffer->AllocationSize.HighPart = pFindData->nFileSizeHigh;
	pBuffer->AllocationSize.LowPart = pFindData->nFileSizeLow;

	pBuffer->CreationTime.HighPart = pFindData->ftCreationTime.dwHighDateTime;
	pBuffer->CreationTime.LowPart  = pFindData->ftCreationTime.dwLowDateTime;

	pBuffer->LastAccessTime.HighPart = pFindData->ftLastAccessTime.dwHighDateTime;
	pBuffer->LastAccessTime.LowPart  = pFindData->ftLastAccessTime.dwLowDateTime;

	pBuffer->LastWriteTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->LastWriteTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	pBuffer->ChangeTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->ChangeTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	pBuffer->EaSize = 0;
	pBuffer->FileId.QuadPart = 0;

	RtlCopyMemory(pBuffer->FileName, pFindData->cFileName, ulNameBytes);
}

VOID CPTFSUtil::FillIdBothDirInfo(PFILE_ID_BOTH_DIR_INFORMATION pBuffer, PWIN32_FIND_DATAW pFindData, ULONG ulIndex)
{
	ULONG ulNameBytes = (ULONG)wcslen(pFindData->cFileName) * sizeof(WCHAR);

	pBuffer->FileIndex = ulIndex;
	pBuffer->FileAttributes = pFindData->dwFileAttributes;
	pBuffer->FileNameLength = ulNameBytes;
	pBuffer->ShortNameLength = 0;

	pBuffer->EndOfFile.HighPart = pFindData->nFileSizeHigh;
	pBuffer->EndOfFile.LowPart   = pFindData->nFileSizeLow;
	pBuffer->AllocationSize.HighPart = pFindData->nFileSizeHigh;
	pBuffer->AllocationSize.LowPart = pFindData->nFileSizeLow;

	pBuffer->CreationTime.HighPart = pFindData->ftCreationTime.dwHighDateTime;
	pBuffer->CreationTime.LowPart  = pFindData->ftCreationTime.dwLowDateTime;

	pBuffer->LastAccessTime.HighPart = pFindData->ftLastAccessTime.dwHighDateTime;
	pBuffer->LastAccessTime.LowPart  = pFindData->ftLastAccessTime.dwLowDateTime;

	pBuffer->LastWriteTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->LastWriteTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	pBuffer->ChangeTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->ChangeTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	pBuffer->EaSize = 0;
	pBuffer->FileId.QuadPart = 0;

	RtlCopyMemory(pBuffer->FileName, pFindData->cFileName, ulNameBytes);
}

VOID CPTFSUtil::FillIdExtBothDirInfo(PFILE_ID_EXTD_BOTH_DIR_INFORMATION Buffer, PWIN32_FIND_DATAW FindData, ULONG Index,
	PDOKAN_INSTANCE DokanInstance) {
	ULONG nameBytes = (ULONG)wcslen(FindData->cFileName) * sizeof(WCHAR);

	Buffer->FileIndex = Index;
	Buffer->FileAttributes = FindData->dwFileAttributes;
	Buffer->FileNameLength = nameBytes;
	Buffer->ShortNameLength = 0;

	Buffer->EndOfFile.HighPart = FindData->nFileSizeHigh;
	Buffer->EndOfFile.LowPart = FindData->nFileSizeLow;
	Buffer->AllocationSize.HighPart = FindData->nFileSizeHigh;
	Buffer->AllocationSize.LowPart = FindData->nFileSizeLow;
	ALIGN_ALLOCATION_SIZE(&Buffer->AllocationSize, DokanInstance->DokanOptions);

	Buffer->CreationTime.HighPart = FindData->ftCreationTime.dwHighDateTime;
	Buffer->CreationTime.LowPart = FindData->ftCreationTime.dwLowDateTime;

	Buffer->LastAccessTime.HighPart = FindData->ftLastAccessTime.dwHighDateTime;
	Buffer->LastAccessTime.LowPart = FindData->ftLastAccessTime.dwLowDateTime;

	Buffer->LastWriteTime.HighPart = FindData->ftLastWriteTime.dwHighDateTime;
	Buffer->LastWriteTime.LowPart = FindData->ftLastWriteTime.dwLowDateTime;

	Buffer->ChangeTime.HighPart = FindData->ftLastWriteTime.dwHighDateTime;
	Buffer->ChangeTime.LowPart = FindData->ftLastWriteTime.dwLowDateTime;

	Buffer->EaSize = 0;
	Buffer->ReparsePointTag = 0;
	RtlFillMemory(&Buffer->FileId.Identifier, sizeof Buffer->FileId.Identifier, 0);

	RtlCopyMemory(Buffer->FileName, FindData->cFileName, nameBytes);
}
VOID CPTFSUtil::FillBothDirInfo(PFILE_BOTH_DIR_INFORMATION pBuffer, PWIN32_FIND_DATAW pFindData, ULONG ulIndex)
{
	ULONG ulNameBytes = (ULONG)wcslen(pFindData->cFileName) * sizeof(WCHAR);

	pBuffer->FileIndex = ulIndex;
	pBuffer->FileAttributes = pFindData->dwFileAttributes;
	pBuffer->FileNameLength = ulNameBytes;
	pBuffer->ShortNameLength = 0;

	pBuffer->EndOfFile.HighPart = pFindData->nFileSizeHigh;
	pBuffer->EndOfFile.LowPart   = pFindData->nFileSizeLow;
	pBuffer->AllocationSize.HighPart = pFindData->nFileSizeHigh;
	pBuffer->AllocationSize.LowPart = pFindData->nFileSizeLow;

	pBuffer->CreationTime.HighPart = pFindData->ftCreationTime.dwHighDateTime;
	pBuffer->CreationTime.LowPart  = pFindData->ftCreationTime.dwLowDateTime;

	pBuffer->LastAccessTime.HighPart = pFindData->ftLastAccessTime.dwHighDateTime;
	pBuffer->LastAccessTime.LowPart  = pFindData->ftLastAccessTime.dwLowDateTime;

	pBuffer->LastWriteTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->LastWriteTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	pBuffer->ChangeTime.HighPart = pFindData->ftLastWriteTime.dwHighDateTime;
	pBuffer->ChangeTime.LowPart  = pFindData->ftLastWriteTime.dwLowDateTime;

	pBuffer->EaSize = 0;

	RtlCopyMemory(pBuffer->FileName, pFindData->cFileName, ulNameBytes);
}

VOID CPTFSUtil::FillNamesInfo(PFILE_NAMES_INFORMATION	pBuffer, PWIN32_FIND_DATAW pFindData, ULONG ulIndex)
{
	ULONG ulNameBytes = (ULONG)wcslen(pFindData->cFileName) * sizeof(WCHAR);

	pBuffer->FileIndex = ulIndex;
	pBuffer->FileNameLength = ulNameBytes;

	RtlCopyMemory(pBuffer->FileName, pFindData->cFileName, ulNameBytes);
}

ULONG CPTFSUtil::FillDirectoryInformation(FILE_INFORMATION_CLASS DirectoryInfo, PVOID pBuffer, PULONG pulLengthRemaining,
	PWIN32_FIND_DATAW pFindData, ULONG ulIndex)
{
	ULONG	ulNameBytes;
	ULONG	ulThisEntrySize;
	
	ulNameBytes = (ULONG)wcslen(pFindData->cFileName) * sizeof(WCHAR);

	ulThisEntrySize = ulNameBytes;

	switch (DirectoryInfo) {
	case FileDirectoryInformation:
		ulThisEntrySize += sizeof(FILE_DIRECTORY_INFORMATION);
		break;
	case FileFullDirectoryInformation:
		ulThisEntrySize += sizeof(FILE_FULL_DIR_INFORMATION);
		break;
	case FileNamesInformation:
		ulThisEntrySize += sizeof(FILE_NAMES_INFORMATION);
		break;
	case FileBothDirectoryInformation:
		ulThisEntrySize += sizeof(FILE_BOTH_DIR_INFORMATION);
		break;
	case FileIdBothDirectoryInformation:
		ulThisEntrySize += sizeof(FILE_ID_BOTH_DIR_INFORMATION);
		break;
	default:
		break;
	}

	// Must be align on a 8-byte boundary.
	ulThisEntrySize = QuadAlign(ulThisEntrySize);

	// no more memory, don't fill any more
	if (*pulLengthRemaining < ulThisEntrySize) {
		CLogUtil::DebugLogString(DLEVEL_INFO, _T("no memory\n"));
		return 0;
	}

	RtlZeroMemory(pBuffer, ulThisEntrySize);

	switch (DirectoryInfo) {
	case FileDirectoryInformation:
		FillDirInfo((PFILE_DIRECTORY_INFORMATION)pBuffer, pFindData, ulIndex);
		break;
	case FileFullDirectoryInformation:
		FillFullDirInfo((PFILE_FULL_DIR_INFORMATION)pBuffer, pFindData, ulIndex);
		break;
	case FileNamesInformation:
		FillNamesInfo((PFILE_NAMES_INFORMATION)pBuffer, pFindData, ulIndex);
		break;
	case FileBothDirectoryInformation:
		FillBothDirInfo((PFILE_BOTH_DIR_INFORMATION)pBuffer, pFindData, ulIndex);
		break;
	case FileIdBothDirectoryInformation:
		FillIdBothDirInfo((PFILE_ID_BOTH_DIR_INFORMATION)pBuffer, pFindData, ulIndex);
		break;
	default:
		break;
	}

	*pulLengthRemaining -= ulThisEntrySize;

	return ulThisEntrySize;
}

VOID CPTFSUtil::ClearFindData(PLIST_ENTRY pListHead)
{
	// free all list entries
	while(!IsListEmpty(pListHead)) {
		PLIST_ENTRY pEntry = RemoveHeadList(pListHead);
		PPTFS_FIND_DATA pFind = CONTAINING_RECORD(pEntry, PTFS_FIND_DATA, ListEntry);
		free(pFind);
	}
}

VOID WINAPI CPTFSUtil::ClearFindData2(PPTFS_FILE_INFO FileInfo)
{
	PLIST_ENTRY pListHead = ((PPTFS_OPEN_INFO)FileInfo->PTFSContext)->DirListHead;

	// free all list entries
	while(!IsListEmpty(pListHead)) {
		PLIST_ENTRY pEntry = RemoveHeadList(pListHead);
		PPTFS_FIND_DATA pFind = CONTAINING_RECORD(pEntry, PTFS_FIND_DATA, ListEntry);
		free(pFind);
	}
}

LONG CPTFSUtil::MatchFiles(PEVENT_CONTEXT pEventContext, PEVENT_INFORMATION pEventInfo, PLIST_ENTRY pFindDataList, BOOLEAN bPatternCheck)
{
	PLIST_ENTRY	pThisEntry, pListHead, pNextEntry;
	ULONG	ulLengthRemaining = pEventInfo->BufferLength;
	PVOID		pCurrentBuffer	= pEventInfo->Buffer;
	PVOID		pLastBuffer		= pCurrentBuffer;
	ULONG	ulIndex = 0;
	PWCHAR pwszPattern = NULL;
	
	// search patten is specified
	if (bPatternCheck && pEventContext->Directory.SearchPatternLength != 0) {
		pwszPattern = (PWCHAR)((SIZE_T)&pEventContext->Directory.SearchPatternBase[0] + (SIZE_T)pEventContext->Directory.SearchPatternOffset);
	}

	pListHead = pFindDataList;

    for(pThisEntry = pListHead->Flink; pThisEntry != pListHead; pThisEntry = pNextEntry) {
        
		PPTFS_FIND_DATA	pFind;
		pNextEntry = pThisEntry->Flink;
		pFind = CONTAINING_RECORD(pThisEntry, PTFS_FIND_DATA, ListEntry);

		CLogUtil::DebugLogString(DLEVEL_INFO,_T("FileMatch? : %s (%s,%d,%d)\n"), pFind->FindData.cFileName, (pwszPattern ? pwszPattern : L"null"), pEventContext->Directory.FileIndex, ulIndex);

		// pattern is not specified or pattern match is ignore cases
		if (!pwszPattern || IsNameInExpression(pwszPattern, pFind->FindData.cFileName, TRUE)) {
			
			if(pEventContext->Directory.FileIndex <= ulIndex) {
				// ulIndex+1 is very important, should use next entry ulIndex
				ULONG ulEntrySize = FillDirectoryInformation((FILE_INFORMATION_CLASS)pEventContext->Directory.FileInformationClass, 
					pCurrentBuffer, &ulLengthRemaining, &pFind->FindData, ulIndex+1);
				// buffer is full
				if (ulEntrySize == 0)
					break;
			
				// pointer of the current last entry
				pLastBuffer = pCurrentBuffer;

				// end if needs to return single entry
				if (pEventContext->Flags & SL_RETURN_SINGLE_ENTRY) {
					CLogUtil::DebugLogString(DLEVEL_INFO, _T("=>return single entry\n"));
					ulIndex++;
					break;
				}

				CLogUtil::DebugLogString(DLEVEL_INFO,_T("=>return\n"));

				// the offset of next entry
				((PFILE_BOTH_DIR_INFORMATION)pCurrentBuffer)->NextEntryOffset = ulEntrySize;

				// next buffer position
				pCurrentBuffer = (PCHAR)pCurrentBuffer + ulEntrySize;
			}
			ulIndex++;
		}
	}

	// Since next of the last entry doesn't exist, clear next offset
	((PFILE_BOTH_DIR_INFORMATION)pLastBuffer)->NextEntryOffset = 0;

	// acctualy used length of buffer
	pEventInfo->BufferLength = pEventContext->Directory.BufferLength - ulLengthRemaining;

	// NO_MORE_FILES
	if (ulIndex <= pEventContext->Directory.FileIndex)
		return -1;

	return ulIndex;
}


PPTFS_OPEN_INFO CPTFSUtil::GetPTFSOpenInfo(PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PPTFS_OPEN_INFO pOpenInfo;

	EnterCriticalSection(&pPTFSInstance->CriticalSection);
	pOpenInfo = (PPTFS_OPEN_INFO)pEventContext->Context;
	if (pOpenInfo != NULL) {
		pOpenInfo->OpenCount++;
		pOpenInfo->EventContext = pEventContext;
		pOpenInfo->PTFSInstance = pPTFSInstance;
	}
	LeaveCriticalSection(&pPTFSInstance->CriticalSection);
	
	return pOpenInfo;
}

VOID CPTFSUtil::ReleasePTFSOpenInfo(PEVENT_INFORMATION pEventInformation, PPTFS_INSTANCE pPTFSInstance)
{
	PPTFS_OPEN_INFO pOpenInfo;
	EnterCriticalSection(&pPTFSInstance->CriticalSection);

	pOpenInfo = (PPTFS_OPEN_INFO)pEventInformation->Context;
	if (pOpenInfo != NULL) {
		pOpenInfo->OpenCount--;
		if (pOpenInfo->OpenCount < 1) {
			if (pOpenInfo->DirListHead != NULL) {
				ClearFindData(pOpenInfo->DirListHead);
				free(pOpenInfo->DirListHead);
				pOpenInfo->DirListHead = NULL;
			}
			free(pOpenInfo);
			pEventInformation->Context = 0;
		}
	}
	LeaveCriticalSection(&pPTFSInstance->CriticalSection);
}


ULONG CPTFSUtil::FillFileBasicInfo(PFILE_BASIC_INFORMATION pBasicInfo, PBY_HANDLE_FILE_INFORMATION pFileInfo, PULONG pulRemainingLength)
{
	if (*pulRemainingLength < sizeof(FILE_BASIC_INFORMATION)) {
		return STATUS_BUFFER_OVERFLOW;
	}

	pBasicInfo->CreationTime.LowPart   = pFileInfo->ftCreationTime.dwLowDateTime;
	pBasicInfo->CreationTime.HighPart  = pFileInfo->ftCreationTime.dwHighDateTime;
	pBasicInfo->LastAccessTime.LowPart = pFileInfo->ftLastAccessTime.dwLowDateTime;
	pBasicInfo->LastAccessTime.HighPart= pFileInfo->ftLastAccessTime.dwHighDateTime;
	pBasicInfo->LastWriteTime.LowPart  = pFileInfo->ftLastWriteTime.dwLowDateTime;
	pBasicInfo->LastWriteTime.HighPart = pFileInfo->ftLastWriteTime.dwHighDateTime;
	pBasicInfo->ChangeTime.LowPart     = pFileInfo->ftLastWriteTime.dwLowDateTime;
	pBasicInfo->ChangeTime.HighPart    = pFileInfo->ftLastWriteTime.dwHighDateTime;
	pBasicInfo->FileAttributes         = pFileInfo->dwFileAttributes;

	*pulRemainingLength -= sizeof(FILE_BASIC_INFORMATION);
	
	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FillFileStandardInfo(PFILE_STANDARD_INFORMATION pStandardInfo, PBY_HANDLE_FILE_INFORMATION pFileInfo, PULONG pulRemainingLength)
{
	if (*pulRemainingLength < sizeof(FILE_STANDARD_INFORMATION)) {
		return STATUS_BUFFER_OVERFLOW;
	}

	pStandardInfo->AllocationSize.HighPart = pFileInfo->nFileSizeHigh;
	pStandardInfo->AllocationSize.LowPart  = pFileInfo->nFileSizeLow;
	pStandardInfo->EndOfFile.HighPart      = pFileInfo->nFileSizeHigh;
	pStandardInfo->EndOfFile.LowPart       = pFileInfo->nFileSizeLow;
	pStandardInfo->NumberOfLinks           = pFileInfo->nNumberOfLinks;
	pStandardInfo->DeletePending           = FALSE;
	pStandardInfo->Directory               = FALSE;

	if (pFileInfo->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
		pStandardInfo->Directory = TRUE;
	}

	*pulRemainingLength -= sizeof(FILE_STANDARD_INFORMATION);
	
	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FillFilePositionInfo(PFILE_POSITION_INFORMATION pPosInfo, PBY_HANDLE_FILE_INFORMATION pFileInfo, PULONG pulRemainingLength)
{
	if (*pulRemainingLength < sizeof(FILE_POSITION_INFORMATION)) {
		return STATUS_BUFFER_OVERFLOW;
	}

	// this field is filled by driver
	pPosInfo->CurrentByteOffset.QuadPart = 0;//fileObject->CurrentByteOffset;
				
	*pulRemainingLength -= sizeof(FILE_POSITION_INFORMATION);
	
	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FillFileAllInfo(PFILE_ALL_INFORMATION pAllInfo, PBY_HANDLE_FILE_INFORMATION pFileInfo,
	PULONG pulRemainingLength, PEVENT_CONTEXT pEventContext)
{
	ULONG	ulAllRemainingLength = *pulRemainingLength;

	if (*pulRemainingLength < sizeof(FILE_ALL_INFORMATION)) {
		return STATUS_BUFFER_OVERFLOW;
	}
	
	// FileBasicInformation
	FillFileBasicInfo(&pAllInfo->BasicInformation, pFileInfo, pulRemainingLength);
	
	// FileStandardInformation
	FillFileStandardInfo(&pAllInfo->StandardInformation, pFileInfo, pulRemainingLength);
	
	// FilePositionInformation
	FillFilePositionInfo(&pAllInfo->PositionInformation, pFileInfo, pulRemainingLength);

	// there is not enough space to fill FileNameInformation
	if (ulAllRemainingLength < sizeof(FILE_ALL_INFORMATION) + pEventContext->File.FileNameLength) {
		// fill out to the limit
		// FileNameInformation
		pAllInfo->NameInformation.FileNameLength = pEventContext->File.FileNameLength;
		pAllInfo->NameInformation.FileName[0] = pEventContext->File.FileName[0];
					
		ulAllRemainingLength -= sizeof(FILE_ALL_INFORMATION);
		*pulRemainingLength = ulAllRemainingLength;
		return STATUS_BUFFER_OVERFLOW;
	}

	// FileNameInformation
	pAllInfo->NameInformation.FileNameLength = pEventContext->File.FileNameLength;
	RtlCopyMemory(&(pAllInfo->NameInformation.FileName[0]),
					pEventContext->File.FileName, pEventContext->File.FileNameLength);

	// the size except of FILE_NAME_INFORMATION
	ulAllRemainingLength -= (sizeof(FILE_ALL_INFORMATION) - sizeof(FILE_NAME_INFORMATION));

	// the size of FILE_NAME_INFORMATION
	ulAllRemainingLength -= FIELD_OFFSET(FILE_NAME_INFORMATION, FileName[0]);
	ulAllRemainingLength -= pAllInfo->NameInformation.FileNameLength;
	
	*pulRemainingLength = ulAllRemainingLength;
	
	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FillFileNameInfo(PFILE_NAME_INFORMATION pNameInfo, PBY_HANDLE_FILE_INFORMATION pFileInfo,
	PULONG pulRemainingLength, PEVENT_CONTEXT pEventContext)
{
	if (*pulRemainingLength < sizeof(FILE_NAME_INFORMATION) 
		+ pEventContext->File.FileNameLength) {
		return STATUS_BUFFER_OVERFLOW;
	}

	pNameInfo->FileNameLength = pEventContext->File.FileNameLength;
	RtlCopyMemory(&(pNameInfo->FileName[0]),
			pEventContext->File.FileName, pEventContext->File.FileNameLength);

	*pulRemainingLength -= FIELD_OFFSET(FILE_NAME_INFORMATION, FileName[0]);
	*pulRemainingLength -= pNameInfo->FileNameLength;

	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FillFileAttributeTagInfo(PFILE_ATTRIBUTE_TAG_INFORMATION pAttrTagInfo, PBY_HANDLE_FILE_INFORMATION pFileInfo, PULONG pulRemainingLength)
{
	if (*pulRemainingLength < sizeof(FILE_ATTRIBUTE_TAG_INFORMATION)) {
		return STATUS_BUFFER_OVERFLOW;
	}

	pAttrTagInfo->FileAttributes = pFileInfo->dwFileAttributes;
	pAttrTagInfo->ReparseTag = 0;

	*pulRemainingLength -= sizeof(FILE_ATTRIBUTE_TAG_INFORMATION);

	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FillNetworkOpenInfo(PFILE_NETWORK_OPEN_INFORMATION pNetInfo, PBY_HANDLE_FILE_INFORMATION pFileInfo, PULONG pulRemainingLength)
{
	if (*pulRemainingLength < sizeof(FILE_NETWORK_OPEN_INFORMATION)) {
		return STATUS_BUFFER_OVERFLOW;
	}

	pNetInfo->CreationTime.LowPart	= pFileInfo->ftCreationTime.dwLowDateTime;
	pNetInfo->CreationTime.HighPart	= pFileInfo->ftCreationTime.dwHighDateTime;
	pNetInfo->LastAccessTime.LowPart	= pFileInfo->ftLastAccessTime.dwLowDateTime;
	pNetInfo->LastAccessTime.HighPart= pFileInfo->ftLastAccessTime.dwHighDateTime;
	pNetInfo->LastWriteTime.LowPart	= pFileInfo->ftLastWriteTime.dwLowDateTime;
	pNetInfo->LastWriteTime.HighPart	= pFileInfo->ftLastWriteTime.dwHighDateTime;
	pNetInfo->ChangeTime.LowPart		= pFileInfo->ftLastWriteTime.dwLowDateTime;
	pNetInfo->ChangeTime.HighPart	= pFileInfo->ftLastWriteTime.dwHighDateTime;
	pNetInfo->AllocationSize.HighPart= pFileInfo->nFileSizeHigh;
	pNetInfo->AllocationSize.LowPart	= pFileInfo->nFileSizeLow;
	pNetInfo->EndOfFile.HighPart		= pFileInfo->nFileSizeHigh;
	pNetInfo->EndOfFile.LowPart		= pFileInfo->nFileSizeLow;
	pNetInfo->FileAttributes			= pFileInfo->dwFileAttributes;

	*pulRemainingLength -= sizeof(FILE_NETWORK_OPEN_INFORMATION);

	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FillInternalInfo(PFILE_INTERNAL_INFORMATION pInternalInfo, PBY_HANDLE_FILE_INFORMATION pFileInfo, PULONG pulRemainingLength)
{
	if (*pulRemainingLength < sizeof(FILE_INTERNAL_INFORMATION)) {
		return STATUS_BUFFER_OVERFLOW;
	}

	pInternalInfo->IndexNumber.HighPart = pFileInfo->nFileIndexHigh;
	pInternalInfo->IndexNumber.LowPart = pFileInfo->nFileIndexLow;

	*pulRemainingLength -= sizeof(FILE_INTERNAL_INFORMATION);

	return STATUS_SUCCESS;
}


//
int WINAPI CPTFSUtil::GetDiskFreeSpace(PULONGLONG pFreeBytesAvailable, PULONGLONG pTotalNumberOfBytes,
	PULONGLONG pTotalNumberOfFreeBytes, PPTFS_FILE_INFO pPTFSFileInfo)
{
	*pFreeBytesAvailable = 512*1024*1024;
	*pTotalNumberOfBytes = 1024*1024*1024;
	*pTotalNumberOfFreeBytes = 512*1024*1024;	
	return 0;
}

int WINAPI CPTFSUtil::GetVolumeInformation(LPWSTR lpwszVolumeNameBuffer, DWORD dwVolumeNameSize, LPDWORD lpdwVolumeSerialNumber,
	LPDWORD lpdwMaximumComponentLength, LPDWORD lpdwFileSystemFlags, LPWSTR lpwszFileSystemNameBuffer, DWORD dwFileSystemNameSize,
	PPTFS_FILE_INFO pPTFSFileInfo)
{
	wcscpy_s(lpwszVolumeNameBuffer, dwVolumeNameSize / sizeof(WCHAR), L"PTFS");
	*lpdwVolumeSerialNumber = 0x19831116;
	*lpdwMaximumComponentLength = 256;
	*lpdwFileSystemFlags = FILE_CASE_SENSITIVE_SEARCH | 
						FILE_CASE_PRESERVED_NAMES | 
						FILE_SUPPORTS_REMOTE_STORAGE |
						FILE_UNICODE_ON_DISK;

	wcscpy_s(lpwszFileSystemNameBuffer, dwFileSystemNameSize / sizeof(WCHAR), L"PTFS");
	return 0;
}

ULONG CPTFSUtil::FsVolumeInformation(PEVENT_INFORMATION pEventInfo, PEVENT_CONTEXT pEventContext, 
	PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	WCHAR	wszVolumeName[MAX_PATH];
	DWORD	dwVolumeSerial;
	DWORD	dwMaxComLength;
	DWORD	dwFsFlags;
	WCHAR	wszFsName[MAX_PATH];
	ULONG	ulRemainingLength;
	ULONG	ulBytesToCopy;

	int		nStatus = -1;

	PFILE_FS_VOLUME_INFORMATION pVolumeInfo = (PFILE_FS_VOLUME_INFORMATION)pEventInfo->Buffer;

	if (!pPTFSOperations->GetVolumeInformation) {
		//return STATUS_NOT_IMPLEMENTED;
		pPTFSOperations->GetVolumeInformation = CPTFSUtil::GetVolumeInformation;
	}

	ulRemainingLength = pEventContext->Volume.BufferLength;

	if (ulRemainingLength < sizeof(FILE_FS_VOLUME_INFORMATION)) {
		return STATUS_BUFFER_OVERFLOW;
	}

	RtlZeroMemory(wszVolumeName, sizeof(wszVolumeName));
	RtlZeroMemory(wszFsName, sizeof(wszFsName));

	nStatus = pPTFSOperations->GetVolumeInformation(
				wszVolumeName,							// VolumeNameBuffer
				sizeof(wszVolumeName) / sizeof(WCHAR), // VolumeNameSize
				&dwVolumeSerial,						// VolumeSerialNumber
				&dwMaxComLength,						// MaximumComponentLength
				&dwFsFlags,							// FileSystemFlags
				wszFsName,								// FileSystemNameBuffer
				sizeof(wszFsName)  / sizeof(WCHAR),	// FileSystemNameSize
				pFileInfo);

	if (nStatus < 0) {
		return STATUS_INVALID_PARAMETER;
	}

	pVolumeInfo->VolumeCreationTime.QuadPart = 0;
	pVolumeInfo->VolumeSerialNumber = dwVolumeSerial;
	pVolumeInfo->SupportsObjects = FALSE;
	
	ulRemainingLength -= FIELD_OFFSET(FILE_FS_VOLUME_INFORMATION, VolumeLabel[0]);
	
	ulBytesToCopy = (ULONG)wcslen(wszVolumeName) * sizeof(WCHAR);
	if (ulRemainingLength < ulBytesToCopy) {
		ulBytesToCopy = ulRemainingLength;
	}

	pVolumeInfo->VolumeLabelLength = ulBytesToCopy;
	RtlCopyMemory(pVolumeInfo->VolumeLabel, wszVolumeName, ulBytesToCopy);
	ulRemainingLength -= ulBytesToCopy;

	pEventInfo->BufferLength = pEventContext->Volume.BufferLength - ulRemainingLength;
	
	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FsSizeInformation(PEVENT_INFORMATION pEventInfo, PEVENT_CONTEXT pEventContext,
	PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	ULONGLONG	 freeBytesAvailable = 0;
	ULONGLONG	 totalBytes = 0;
	ULONGLONG	 freeBytes = 0;
	
	int			nStatus = -1;

	PFILE_FS_SIZE_INFORMATION pSizeInfo = (PFILE_FS_SIZE_INFORMATION)pEventInfo->Buffer;

	if (!pPTFSOperations->GetDiskFreeSpace) {
		//return STATUS_NOT_IMPLEMENTED;
		pPTFSOperations->GetDiskFreeSpace = CPTFSUtil::GetDiskFreeSpace;
	}

	if (pEventContext->Volume.BufferLength < sizeof(FILE_FS_SIZE_INFORMATION) ) {
		return STATUS_BUFFER_OVERFLOW;
	}

	nStatus = pPTFSOperations->GetDiskFreeSpace(
		&freeBytesAvailable, // FreeBytesAvailable
		&totalBytes, // TotalNumberOfBytes
		&freeBytes, // TotalNumberOfFreeBytes
		pFileInfo);

	if (nStatus < 0) {
		return STATUS_INVALID_PARAMETER;
	}

	pSizeInfo->TotalAllocationUnits.QuadPart		= totalBytes / PTFS_ALLOCATION_UNIT_SIZE;
	pSizeInfo->AvailableAllocationUnits.QuadPart	= freeBytesAvailable / PTFS_ALLOCATION_UNIT_SIZE;
	pSizeInfo->SectorsPerAllocationUnit			= PTFS_ALLOCATION_UNIT_SIZE / PTFS_SECTOR_SIZE;
	pSizeInfo->BytesPerSector					= PTFS_SECTOR_SIZE;

	pEventInfo->BufferLength = sizeof(FILE_FS_SIZE_INFORMATION);

	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FsAttributeInformation(PEVENT_INFORMATION pEventInfo, PEVENT_CONTEXT pEventContext,
	PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	WCHAR	wszVolumeName[MAX_PATH];
	DWORD	dwVolumeSerial;
	DWORD	dwMaxComLength;
	DWORD	dwFsFlags;
	WCHAR	wszFsName[MAX_PATH];
	ULONG	ulRemainingLength;
	ULONG	ulBytesToCopy;

	int		nStatus = -1;

	PFILE_FS_ATTRIBUTE_INFORMATION pAttrInfo = (PFILE_FS_ATTRIBUTE_INFORMATION)pEventInfo->Buffer;
	
	if (!pPTFSOperations->GetVolumeInformation) {
		pPTFSOperations->GetVolumeInformation = CPTFSUtil::GetVolumeInformation;
		//return STATUS_NOT_IMPLEMENTED;
	}

	ulRemainingLength = pEventContext->Volume.BufferLength;

	if (ulRemainingLength < sizeof(FILE_FS_ATTRIBUTE_INFORMATION)) {
		return STATUS_BUFFER_OVERFLOW;
	}


	RtlZeroMemory(wszVolumeName, sizeof(wszVolumeName));
	RtlZeroMemory(wszFsName, sizeof(wszFsName));

	nStatus = pPTFSOperations->GetVolumeInformation(
				wszVolumeName,			// VolumeNameBuffer
				sizeof(wszVolumeName),	// VolumeNameSize
				&dwVolumeSerial,		// VolumeSerialNumber
				&dwMaxComLength,		// MaximumComponentLength
				&dwFsFlags,			// FileSystemFlags
				wszFsName,				// FileSystemNameBuffer
				sizeof(wszFsName),		// FileSystemNameSize
				pFileInfo);

	if (nStatus < 0) {
		return STATUS_INVALID_PARAMETER;
	}

	pAttrInfo->FileSystemAttributes = dwFsFlags;
	pAttrInfo->MaximumComponentNameLength = dwMaxComLength;

	ulRemainingLength -= FIELD_OFFSET(FILE_FS_ATTRIBUTE_INFORMATION, FileSystemName[0]);
	
	ulBytesToCopy = (ULONG)wcslen(wszFsName) * sizeof(WCHAR);
	if (ulRemainingLength < ulBytesToCopy) {
		ulBytesToCopy = ulRemainingLength;
	}

	pAttrInfo->FileSystemNameLength = ulBytesToCopy;
	RtlCopyMemory(pAttrInfo->FileSystemName, wszFsName, ulBytesToCopy);
	ulRemainingLength -= ulBytesToCopy;

	pEventInfo->BufferLength = pEventContext->Volume.BufferLength - ulRemainingLength;
	
	return STATUS_SUCCESS;
}

ULONG CPTFSUtil::FsFullSizeInformation(PEVENT_INFORMATION pEventInfo, PEVENT_CONTEXT pEventContext,
	PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	ULONGLONG	 freeBytesAvailable = 0;
	ULONGLONG	 totalBytes = 0;
	ULONGLONG	 freeBytes = 0;
	
	int			nStatus = -1;

	PFILE_FS_FULL_SIZE_INFORMATION pSizeInfo = (PFILE_FS_FULL_SIZE_INFORMATION)pEventInfo->Buffer;

	if (!pPTFSOperations->GetDiskFreeSpace) {
		pPTFSOperations->GetDiskFreeSpace = CPTFSUtil::GetDiskFreeSpace;
		//return STATUS_NOT_IMPLEMENTED;
	}

	if (pEventContext->Volume.BufferLength < sizeof(FILE_FS_FULL_SIZE_INFORMATION) ) {
		return STATUS_BUFFER_OVERFLOW;
	}

	nStatus = pPTFSOperations->GetDiskFreeSpace(
		&freeBytesAvailable, // FreeBytesAvailable
		&totalBytes, // TotalNumberOfBytes
		&freeBytes, // TotalNumberOfFreeBytes
		pFileInfo);

	if (nStatus < 0) {
		return STATUS_INVALID_PARAMETER;
	}

	pSizeInfo->TotalAllocationUnits.QuadPart		= totalBytes / PTFS_ALLOCATION_UNIT_SIZE;
	pSizeInfo->ActualAvailableAllocationUnits.QuadPart = freeBytes / PTFS_ALLOCATION_UNIT_SIZE;
	pSizeInfo->CallerAvailableAllocationUnits.QuadPart = freeBytesAvailable / PTFS_ALLOCATION_UNIT_SIZE;
	pSizeInfo->SectorsPerAllocationUnit			= PTFS_ALLOCATION_UNIT_SIZE / PTFS_SECTOR_SIZE;
	pSizeInfo->BytesPerSector					= PTFS_SECTOR_SIZE;

	pEventInfo->BufferLength = sizeof(FILE_FS_FULL_SIZE_INFORMATION);

	return STATUS_SUCCESS;
}
//

//
int CPTFSUtil::SetAllocationInformation(PEVENT_CONTEXT	 pEventContext, PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	PFILE_ALLOCATION_INFORMATION pAllocInfo = (PFILE_ALLOCATION_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);

	// A file's allocation size and end-of-file position are independent of each other,
	// with the following exception: The end-of-file position must always be less than
	// or equal to the allocation size. If the allocation size is set to a value that
	// is less than the end-of-file position, the end-of-file position is automatically
	// adjusted to match the allocation size.

	if (pPTFSOperations->SetAllocationSize) {
		return pPTFSOperations->SetAllocationSize(
			pEventContext->SetFile.FileName,
			pAllocInfo->AllocationSize.QuadPart,
			pFileInfo);
	}
	// How can we check the current end-of-file position?
	if (pAllocInfo->AllocationSize.QuadPart == 0) {
		return pPTFSOperations->SetEndOfFile(
			pEventContext->SetFile.FileName,
			pAllocInfo->AllocationSize.QuadPart,
			pFileInfo);
	} else {
		CLogUtil::DebugLogString(DLEVEL_ERROR, _T("SetAllocationInformation %I64d, can't handle this parameter.\n"), pAllocInfo->AllocationSize.QuadPart);
	}

	return 0;
}

int CPTFSUtil::SetBasicInformation(PEVENT_CONTEXT pEventContext, PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	FILETIME creation, lastAccess, lastWrite;
	int nStatus = -1;

	PFILE_BASIC_INFORMATION pBasicInfo = (PFILE_BASIC_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);

	if (!pPTFSOperations->SetFileAttributes)
		return -1;
	
	if (!pPTFSOperations->SetFileTime)
		return -1;

	nStatus = pPTFSOperations->SetFileAttributes(
		pEventContext->SetFile.FileName,
		pBasicInfo->FileAttributes,
		pFileInfo);

	if (nStatus < 0)
		return nStatus;

	creation.dwLowDateTime = pBasicInfo->CreationTime.LowPart;
	creation.dwHighDateTime = pBasicInfo->CreationTime.HighPart;
	lastAccess.dwLowDateTime = pBasicInfo->LastAccessTime.LowPart;
	lastAccess.dwHighDateTime = pBasicInfo->LastAccessTime.HighPart;
	lastWrite.dwLowDateTime = pBasicInfo->LastWriteTime.LowPart;
	lastWrite.dwHighDateTime = pBasicInfo->LastWriteTime.HighPart;


	return pPTFSOperations->SetFileTime( pEventContext->SetFile.FileName,
		&creation,
		&lastAccess,
		&lastWrite,
		pFileInfo);
}

int CPTFSUtil::SetDispositionInformation(PEVENT_CONTEXT pEventContext, PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	PFILE_DISPOSITION_INFORMATION pDispositionInfo = (PFILE_DISPOSITION_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);

	if (!pPTFSOperations->DeleteFile || !pPTFSOperations->DeleteDirectory)
		return -1;

	if (!pDispositionInfo->DeleteFile) {
		return 0;
	}

	if (pFileInfo->IsDirectory) {
		return pPTFSOperations->DeleteDirectory( pEventContext->SetFile.FileName, pFileInfo);
	} else {
		return pPTFSOperations->DeleteFile(
			pEventContext->SetFile.FileName,
			pFileInfo);
	}
}

int CPTFSUtil::SetEndOfFileInformation(PEVENT_CONTEXT pEventContext, PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	PFILE_END_OF_FILE_INFORMATION pEndInfo = (PFILE_END_OF_FILE_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);

	if (!pPTFSOperations->SetEndOfFile)
		return -1;

	return pPTFSOperations->SetEndOfFile(
		pEventContext->SetFile.FileName,
		pEndInfo->EndOfFile.QuadPart,
		pFileInfo);
}

int CPTFSUtil::SetLinkInformation(PEVENT_CONTEXT pEventContext, PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	PPTFS_LINK_INFORMATION pLinkInfo = (PPTFS_LINK_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);
	return -1;
}

int CPTFSUtil::SetRenameInformation(PEVENT_CONTEXT pEventContext, PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	PPTFS_RENAME_INFORMATION pRenameInfo = (PPTFS_RENAME_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);

	WCHAR wszNewName[MAX_PATH];
	ZeroMemory(wszNewName, sizeof(wszNewName));

	if (pRenameInfo->FileName[0] != L'\\') {
		ULONG pos;
		for (pos = pEventContext->SetFile.FileNameLength/sizeof(WCHAR);
			pos != 0; --pos) {
			if (pEventContext->SetFile.FileName[pos] == '\\')
				break;
		}
		RtlCopyMemory(wszNewName, pEventContext->SetFile.FileName, (pos+1)*sizeof(WCHAR));
		RtlCopyMemory((PCHAR)wszNewName + (pos+1)*sizeof(WCHAR), pRenameInfo->FileName, pRenameInfo->FileNameLength);
	} else {
		RtlCopyMemory(wszNewName, pRenameInfo->FileName, pRenameInfo->FileNameLength);
	}

	if (!pPTFSOperations->MoveFile)
		return -1;

	return pPTFSOperations->MoveFile(
		pEventContext->SetFile.FileName,
		wszNewName,
		pRenameInfo->ReplaceIfExists,
		pFileInfo);
}

int CPTFSUtil::SetValidDataLengthInformation(PEVENT_CONTEXT pEventContext, PPTFS_FILE_INFO pFileInfo, PPTFS_OPERATIONS pPTFSOperations)
{
	PFILE_VALID_DATA_LENGTH_INFORMATION pValidInfo = (PFILE_VALID_DATA_LENGTH_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);

	if (!pPTFSOperations->SetEndOfFile)
		return -1;

	return pPTFSOperations->SetEndOfFile(
		pEventContext->SetFile.FileName,
		pValidInfo->ValidDataLength.QuadPart,
		pFileInfo);
}
//