#include "stdafx.h"
#include"PTFSMgr.h"
#include "List/List.h"
#include "create.h"
#include "cleanup.h"
#include "close.h"
#include "directory.h"
#include "fileinfo.h"
#include "flush.h"
#include "lock.h"
#include "read.h"
#include "security.h"
#include "setfile.h"
#include "volume.h"
#include "write.h"
#include <conio.h>
#include <process.h>
#include <stdlib.h>
#include <tchar.h>
#include <strsafe.h>
#include <Dbt.h>


#define PTFSMapKernelBit(dest, src, userBit, kernelBit)                       \
  if (((src) & (kernelBit)) == (kernelBit))                                    \
  (dest) |= (userBit)

CPTFSMgr::CPTFSMgr(void)
{
    m_bDebugMode = FALSE;
    m_bUseStdErr = FALSE;
    m_pPTFSInstance = NULL;

    m_hKeepAlive = INVALID_HANDLE_VALUE;
    m_hNotify = INVALID_HANDLE_VALUE;
    m_hBroadCastThread = NULL;
    m_bCompleteMount = FALSE;
    m_bCompleteWaitMount = FALSE;
    InitializeCriticalSection(&m_InstanceCriticalSection);
    InitializeListHead(&m_InstanceList);
}

CPTFSMgr::~CPTFSMgr(void)
{
}

void CPTFSMgr::SetUseStdErr(BOOL bStatus)
{
    m_bUseStdErr = bStatus;
}

void CPTFSMgr::SetDebugMode(BOOL bStatus)
{
    m_bDebugMode = bStatus;
}

BOOL CPTFSMgr::GetUseStdErr()
{
    return m_bUseStdErr;
}

BOOL CPTFSMgr::GetDebugMode()
{
    return m_bDebugMode;
}

BOOL CPTFSMgr::NewPTFSInstance()
{
    if (NULL != m_pPTFSInstance)
    {
        ZeroMemory(m_pPTFSInstance, sizeof(PTFS_INSTANCE));
        return TRUE;
    }

    m_pPTFSInstance = (PPTFS_INSTANCE)malloc(sizeof(PTFS_INSTANCE));
    if (m_pPTFSInstance == NULL)
        return FALSE;

    ZeroMemory(m_pPTFSInstance, sizeof(PTFS_INSTANCE));

#if _MSC_VER < 1300
    InitializeCriticalSection(&m_pPTFSInstance->CriticalSection);
#else
    (void)InitializeCriticalSectionAndSpinCount(&m_pPTFSInstance->CriticalSection, 0x80000400);
#endif

    InitializeListHead(&m_pPTFSInstance->ListEntry);

    EnterCriticalSection(&m_InstanceCriticalSection);
    InsertTailList(&m_InstanceList, &m_pPTFSInstance->ListEntry);
    LeaveCriticalSection(&m_InstanceCriticalSection);

    return TRUE;
}

void CPTFSMgr::DeletePTFSInstance()
{
    if (NULL != m_pPTFSInstance)
    {
        DeleteCriticalSection(&m_pPTFSInstance->CriticalSection);

        EnterCriticalSection(&m_InstanceCriticalSection);
        RemoveEntryList(&m_pPTFSInstance->ListEntry);
        LeaveCriticalSection(&m_InstanceCriticalSection);

        free(m_pPTFSInstance);
        m_pPTFSInstance = NULL;
    }
}

BOOL CPTFSMgr::IsMountPointDriveLetter(LPCWSTR mountPoint)
{
    size_t mountPointLength;

    if (!mountPoint || *mountPoint == 0)
        return FALSE;

    mountPointLength = wcslen(mountPoint);

    if (mountPointLength == 1 ||
        (mountPointLength == 2 && mountPoint[1] == L':') ||
        (mountPointLength == 3 && mountPoint[1] == L':' && mountPoint[2] == L'\\'))
        return TRUE;

    return FALSE;
}

BOOL CPTFSMgr::IsValidDriveLetter(WCHAR DriveLetter) 
{
    return (L'a' <= DriveLetter && DriveLetter <= L'z') ||
        (L'A' <= DriveLetter && DriveLetter <= L'Z');
}

VOID CPTFSMgr::GetRawDeviceName(LPCWSTR DeviceName, LPWSTR DestinationBuffer, rsize_t DestinationBufferSizeInElements) 
{
    if (DeviceName && DestinationBuffer && DestinationBufferSizeInElements > 0) {
        swprintf_s(DestinationBuffer, DestinationBufferSizeInElements, L"\\\\.%s", DeviceName);
//        wcscpy_s(DestinationBuffer, DestinationBufferSizeInElements, L"\\\\.");
//        wcscat_s(DestinationBuffer, DestinationBufferSizeInElements, DeviceName);
    }
}

BOOL CPTFSMgr::CheckDriveLetterAvailability(WCHAR DriveLetter) 
{
    DWORD result = 0;
    WCHAR buffer[MAX_PATH];
    WCHAR dosDevice[] = L"\\\\.\\C:";
    WCHAR driveName[] = L"C:";
    WCHAR driveLetter = towupper(DriveLetter);
    HANDLE hDevice = NULL;
    dosDevice[4] = driveLetter;
    driveName[0] = driveLetter;

    CleanUpMountPoints();

    if (!IsValidDriveLetter(driveLetter))
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::CheckDriveLetterAvailability Invalid Drive Letter[%c]\n", DriveLetter);
        return FALSE;
    }

    hDevice = CreateFile(dosDevice, GENERIC_READ | GENERIC_WRITE,
        FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING,
        FILE_FLAG_NO_BUFFERING, NULL);

    if (hDevice != INVALID_HANDLE_VALUE) 
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::CheckDriveLetterAvailability is already used[%c]\n", DriveLetter);
        CloseHandle(hDevice);
        return FALSE;
    }

    ZeroMemory(buffer, MAX_PATH * sizeof(WCHAR));
    result = QueryDosDevice(driveName, buffer, MAX_PATH);
    if (result > 0) 
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::CheckDriveLetterAvailability is already used(QueryDosDevice)[%c]\n", DriveLetter);
        return FALSE;
    }

    DWORD drives = GetLogicalDrives();
    result = (drives >> (driveLetter - L'A') & 0x00000001);
    if (result > 0) 
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::CheckDriveLetterAvailability is already used(GetLogicalDrives)[%c]\n", DriveLetter);
        return FALSE;
    }

    return TRUE;
}

void CPTFSMgr::CheckAllocationUnitSectorSize(PPTFS_OPTIONS pPTFSOptions) 
{
    ULONG allocationUnitSize = pPTFSOptions->AllocationUnitSize;
    ULONG sectorSize = pPTFSOptions->SectorSize;

    if ((allocationUnitSize < 512 || allocationUnitSize > 65536 ||
        (allocationUnitSize & (allocationUnitSize - 1)) != 0) 
        || (sectorSize < 512 || sectorSize > 65536 ||
            (sectorSize & (sectorSize - 1)))) 
    {
        pPTFSOptions->SectorSize = PTFS_DEFAULT_SECTOR_SIZE;
        pPTFSOptions->AllocationUnitSize = PTFS_DEFAULT_ALLOCATION_UNIT_SIZE;
    }

    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::CheckAllocationUnitSectorSize AllocationUnitSize: %d SectorSize : %d\n", 
        pPTFSOptions->AllocationUnitSize, pPTFSOptions->SectorSize);
}

PPTFS_INSTANCE CPTFSMgr::GetPTFSInstance()
{
    return m_pPTFSInstance;
}

CPTFSDriver* CPTFSMgr::GetPTFSDriver()
{
    return &m_PTFSDriver;
}

BOOL CPTFSMgr::CleanUpMountPoints()
{
    ULONG returnedLength;
    return m_PTFSDriver.SendToGlobalDevice( IOCTL_MOUNTPOINT_CLEANUP, NULL,
        0, NULL, 0, &returnedLength);
}

BOOL CPTFSMgr::DoStart() 
{
    EVENT_START eventStart;
    EVENT_DRIVER_INFO driverInfo;
    ULONG returnedLength = 0;

    ZeroMemory(&eventStart, sizeof(EVENT_START));
    ZeroMemory(&driverInfo, sizeof(EVENT_DRIVER_INFO));

    eventStart.UserVersion = PTFS_DRIVER_VERSION;
    if (m_pPTFSInstance->pPTFSOptions->Options & PTFS_OPTION_ALT_STREAM) 
        eventStart.Flags |= PTFS_EVENT_ALTERNATIVE_STREAM_ON;

    if (m_pPTFSInstance->pPTFSOptions->Options & PTFS_OPTION_NETWORK)
        eventStart.DeviceType = PTFS_NETWORK_FILE_SYSTEM;

    if (m_pPTFSInstance->pPTFSOptions->Options & PTFS_OPTION_REMOVABLE)
        eventStart.Flags |= PTFS_EVENT_REMOVABLE;
 
    if (m_pPTFSInstance->pPTFSOptions->Options & PTFS_OPTION_WRITE_PROTECT)
        eventStart.Flags |= PTFS_EVENT_WRITE_PROTECT;
    
    if (m_pPTFSInstance->pPTFSOptions->Options & PTFS_OPTION_MOUNT_MANAGER)
        eventStart.Flags |= PTFS_EVENT_MOUNT_MANAGER;
    
    if (m_pPTFSInstance->pPTFSOptions->Options & PTFS_OPTION_CURRENT_SESSION)
        eventStart.Flags |= PTFS_EVENT_CURRENT_SESSION;
    
    if (m_pPTFSInstance->pPTFSOptions->Options & PTFS_OPTION_FILELOCK_USER_MODE)
        eventStart.Flags |= PTFS_EVENT_FILELOCK_USER_MODE;
    
    if (m_pPTFSInstance->pPTFSOptions->Options & PTFS_OPTION_DISABLE_OPLOCKS)
        eventStart.Flags |= PTFS_EVENT_DISABLE_OPLOCKS;
    
    if (m_pPTFSInstance->pPTFSOptions->Options & PTFS_OPTION_ENABLE_FCB_GARBAGE_COLLECTION)
        eventStart.Flags |= PTFS_EVENT_ENABLE_FCB_GC;
 
    memcpy_s(eventStart.MountPoint, sizeof(eventStart.MountPoint),m_pPTFSInstance->MountPoint, sizeof(m_pPTFSInstance->MountPoint));
    memcpy_s(eventStart.UNCName, sizeof(eventStart.UNCName), m_pPTFSInstance->UNCName, sizeof(m_pPTFSInstance->UNCName));
    eventStart.IrpTimeout = m_pPTFSInstance->pPTFSOptions->Timeout;

    m_PTFSDriver.SendToGlobalDevice( IOCTL_EVENT_START, &eventStart,
        sizeof(EVENT_START), &driverInfo, sizeof(EVENT_DRIVER_INFO),
        &returnedLength);

    if (driverInfo.Status == PTFS_START_FAILED)
    {
        if (driverInfo.DriverVersion != eventStart.UserVersion)
            CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoStart driver version mismatch, driver %X, dll %X\n", driverInfo.DriverVersion, eventStart.UserVersion);
        else
            CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoStart driver start error\n");
        return FALSE;
    }
    else if (driverInfo.Status == PTFS_MOUNTED) 
    {
        m_pPTFSInstance->MountId = driverInfo.MountId;
        m_pPTFSInstance->DeviceNumber = driverInfo.DeviceNumber;
        wcscpy_s(m_pPTFSInstance->DeviceName, sizeof(m_pPTFSInstance->DeviceName) / sizeof(WCHAR),driverInfo.DeviceName);
        CLogUtil::DebugLogString(DLEVEL_WARN, L"PTFSMgr::DoStart  Success DeviceName[%s]\n", m_pPTFSInstance->DeviceName);
        return TRUE;
    }

    return FALSE;
}

UINT WINAPI CPTFSMgr::PTFSLoopThreadProc(PVOID pThreadParam)
{
    CPTFSMgr* pPTFSMgr = (CPTFSMgr*)pThreadParam;
    PPTFS_INSTANCE pPTFSInstance = pPTFSMgr->GetPTFSInstance();
    CPTFSDriver* pPTFSDriver = pPTFSMgr->GetPTFSDriver();
    HANDLE hRawDevice = INVALID_HANDLE_VALUE;
    char* buffer = NULL;
    BOOL status;
    ULONG returnedLength;
    DWORD result = 0;
    DWORD lastError = 0;
    WCHAR rawDeviceName[MAX_PATH];

    buffer = (char*)malloc(sizeof(char) * EVENT_CONTEXT_MAX_SIZE);
    
    if (buffer == NULL) 
    {
        result = (DWORD)-1;
        _endthreadex(result);
        return result;
    }
 
    ZeroMemory(rawDeviceName, sizeof(rawDeviceName));
    pPTFSMgr->GetRawDeviceName(pPTFSInstance->DeviceName, rawDeviceName, MAX_PATH);

    CLogUtil::DebugLogString(DLEVEL_WARN, L"PTFSMgr::PTFSLoopThreadProc Start DeviceName[%s], RawDeviceName[%s]\n", pPTFSInstance->DeviceName, rawDeviceName);

    status = TRUE;

    while (status) 
    {
        RtlZeroMemory(buffer, sizeof(char) * EVENT_CONTEXT_MAX_SIZE);

        if (FALSE == pPTFSMgr->m_bCompleteMount)
        {
            status = TRUE;
            Sleep(200);
            continue;
        }

        status = pPTFSDriver->SendToDevice(rawDeviceName, IOCTL_EVENT_WAIT, NULL, 0, buffer, sizeof(char) * EVENT_CONTEXT_MAX_SIZE, &returnedLength, &hRawDevice);

        if (!status) 
        {
            lastError = GetLastError();
            CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::PTFSLoopThreadProc Failed to SendToDevice DeviceName[%s], RawDeviceName[%s], ErrorCode[%d]\n", pPTFSInstance->DeviceName, rawDeviceName, lastError);
            
            if (lastError == ERROR_NO_SYSTEM_RESOURCES ||  FALSE == pPTFSMgr->m_bCompleteWaitMount)
            {
                CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::PTFSLoopThreadProc ERROR_NO_SYSTEM_RESOURCES DeviceName[%s], RawDeviceName[%s], ErroCode[%d], IsCompleteWaitMount[%d]\n", pPTFSInstance->DeviceName, rawDeviceName, lastError, pPTFSMgr->m_bCompleteWaitMount);
                status = TRUE;
                Sleep(200);
                continue;
            }
            break;
        }

        if (returnedLength > 0) 
        {
            PEVENT_CONTEXT context = (PEVENT_CONTEXT)buffer;
            if (context->MountId != pPTFSInstance->MountId) 
            {
                CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::PTFSLoopThreadProc Invalid MountId(expected:%d, acctual:%d)\n", pPTFSInstance->MountId, context->MountId);
                continue;
            }

            switch (context->MajorFunction) 
            {
            case IRP_MJ_CREATE:
                FSCreate::DispatchCreate(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_CLEANUP:
                FSCleanup::DispatchCleanup(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_CLOSE:
                FSClose::DispatchClose(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_DIRECTORY_CONTROL:
                FSDirectory::DispatchDirectoryInformation(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_READ:
                FSRead::DispatchRead(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_WRITE:
                FSWrite::DispatchWrite(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_QUERY_INFORMATION:
                FSFileInfo::DispatchQueryInformation(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_QUERY_VOLUME_INFORMATION:
                FSVolume::DispatchQueryVolumeInformation(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_LOCK_CONTROL:
                FSLock::DispatchLock(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_SET_INFORMATION:
                FSSetFile::DispatchSetInformation(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_FLUSH_BUFFERS:
                FSFlush::DispatchFlush(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_QUERY_SECURITY:
                FSSecurity::DispatchQuerySecurity(hRawDevice, context, pPTFSInstance);
                break;
            case IRP_MJ_SET_SECURITY:
                FSSecurity::DispatchSetSecurity(hRawDevice, context, pPTFSInstance);
                break;
            default:
                break;
            }
        }
        else
            CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::PTFSLoopThreadProc ReturnedLength< 0 [%d]\n", returnedLength);
    }

    if(hRawDevice != INVALID_HANDLE_VALUE)
        CloseHandle(hRawDevice);
    
    free(buffer);
    _endthreadex(result);
    return result;
}

BOOL CPTFSMgr::EnableTokenPrivilege(LPCTSTR lpszSystemName, BOOL bEnable)
{
    HANDLE hToken = NULL;
    if (OpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, &hToken) )
    {
        TOKEN_PRIVILEGES tp = { 0 };
        if (LookupPrivilegeValue(NULL, lpszSystemName, &tp.Privileges[0].Luid)) 
        {
            tp.PrivilegeCount = 1;
            tp.Privileges[0].Attributes = (bEnable ? SE_PRIVILEGE_ENABLED : 0);

            if (AdjustTokenPrivileges(hToken, FALSE, &tp, sizeof(TOKEN_PRIVILEGES),(PTOKEN_PRIVILEGES)NULL, NULL) )
            {
                CloseHandle(hToken);
                return GetLastError() == ERROR_SUCCESS;
            }
        }
    }

    if (hToken) 
        CloseHandle(hToken);

    return FALSE;
}

UINT WINAPI CPTFSMgr::BroadcastLinkThreadProc(PVOID pThreadParam)
{
    PBROADCAST_PARAM pBroadCastParam = (PBROADCAST_PARAM)pThreadParam;

    if (NULL == pBroadCastParam) return 0;

    DWORD receipients;
    DWORD device_event;
    DEV_BROADCAST_VOLUME params;
    WCHAR drive[4] = L"C:\\";
    LONG wEventId;

    if (!isalpha(pBroadCastParam->cLetter))
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::BroadcastLinkThreadProc invalid parameter\n");
        return 0 ;
    }

    receipients = BSM_APPLICATIONS;
    if (pBroadCastParam->bSafe && pBroadCastParam->pPTFSMgr->EnableTokenPrivilege(SE_TCB_NAME, TRUE))
        receipients |= BSM_ALLDESKTOPS;

    device_event = pBroadCastParam->bRemoved ? DBT_DEVICEREMOVECOMPLETE : DBT_DEVICEARRIVAL;

    ZeroMemory(&params, sizeof(params));
    params.dbcv_size = sizeof(params);
    params.dbcv_devicetype = DBT_DEVTYP_VOLUME;
    params.dbcv_reserved = 0;
    params.dbcv_unitmask = (1 << (toupper(pBroadCastParam->cLetter) - 'A'));
    params.dbcv_flags = 0;

    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::BroadcastLinkThreadProc BroadcastSystemMessage Begin\n");

    if (pBroadCastParam->hStartEvent) SetEvent(pBroadCastParam->hStartEvent);

    if (BroadcastSystemMessage(BSF_NOHANG | BSF_FORCEIFHUNG | BSF_NOTIMEOUTIFNOTHUNG, &receipients,
        WM_DEVICECHANGE, device_event, (LPARAM)&params) <= 0)
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::BroadcastLinkThreadProc failed to BroadcastSystemMessage Error[%d]\n", GetLastError());
    }

    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::BroadcastLinkThreadProc BroadcastSystemMessage End\n");

    if (pBroadCastParam->bSafe)
    {
        drive[0] = towupper(pBroadCastParam->cLetter);
        wEventId = pBroadCastParam->bRemoved ? SHCNE_DRIVEREMOVED : SHCNE_DRIVEADD;
        SHChangeNotify(wEventId, SHCNF_PATH, drive, NULL);
    }

    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::BroadcastLinkThreadProc SHChangeNotify End\n");

    CloseHandle(pBroadCastParam->pPTFSMgr->m_hBroadCastThread);
    pBroadCastParam->pPTFSMgr->m_hBroadCastThread = NULL;
    free(pBroadCastParam);
    return 0;
}

void CPTFSMgr::WaitBroadCastThread()
{
	DWORD dwExit = STILL_ACTIVE;
    int nCount = 0;
	CLogUtil::DebugLogString(DLEVEL_TRACE, _T("[PTFSMgr]::WaitBroadCastThread\n"));

	while (dwExit == STILL_ACTIVE)
	{
		if (!m_hBroadCastThread)
			break;

		if (!GetExitCodeThread(m_hBroadCastThread, &dwExit))
			break;

		if (dwExit == STILL_ACTIVE)
			Sleep(500);

        if (nCount >= 360)
        {
            CLogUtil::DebugLogString(DLEVEL_TRACE, _T("[PTFSMgr]::WaitBroadCastThread TerminateThread\n"));
            if (TerminateThread(m_hBroadCastThread, dwExit))
            {
                CloseHandle(m_hBroadCastThread);
                m_hBroadCastThread = NULL;
            }
            break;
        }

	}

	CLogUtil::DebugLogString(DLEVEL_TRACE, _T("[PTFSMgr]::WaitBroadCastThread End\n"));
}

void CPTFSMgr::	WaitMount(WCHAR cLetter, BOOL bRemovedMount)
{
    int nWaitCount = 0;
    CString strDosDeviceMountPoint = L"";

    strDosDeviceMountPoint.Format(L"\\DosDevices\\%c:", cLetter);

    CLogUtil::DebugLogString(DLEVEL_TRACE, _T("[PTFSMgr]::WaitMount Begin Volume[%c], bRemove[%d]\n"), cLetter, bRemovedMount);

    if (bRemovedMount)
        Sleep(500);   

    while (1)
    {
        ULONG ulRead = 0;
        PPTFS_CONTROL pResultsControl = NULL;

        pResultsControl = GetMountPointList(FALSE, &ulRead);

        if (pResultsControl)
        {
            BOOL bFindMount = FALSE;
            for (ULONG i = 0; i < ulRead; ++i)
            {
                PPTFS_CONTROL pPTFSCtrl = &pResultsControl[i];
                if (!_wcsicmp(pPTFSCtrl->MountPoint, strDosDeviceMountPoint))
                {
                    if (bRemovedMount ||  pPTFSCtrl->VolumeDeviceObject)
                    {
                        bFindMount = TRUE;
                        break;
                    }
                }
            }

            free(pResultsControl);

            if (FALSE == bRemovedMount && bFindMount)
            {
                CLogUtil::DebugLogString(DLEVEL_WARN, _T("[PTFSMgr]::WaitMount break Find MountPoint\n"));
                m_bCompleteWaitMount = TRUE;
                break;
            }
            else if (bRemovedMount && FALSE == bFindMount)
                break;
        }
        else
        {
            if (bRemovedMount) break;
        }

        Sleep(500);

        if (nWaitCount >= 20)
        {
            CLogUtil::DebugLogString(DLEVEL_WARN, _T("[PTFSMgr]::WaitMount Wait Mount break IsRemoveMount[%d]\n"), bRemovedMount);
            
            if (FALSE == bRemovedMount)
            {
                m_bCompleteWaitMount = TRUE;
            }

            break;
        }
        nWaitCount++;
    }

    CLogUtil::DebugLogString(DLEVEL_TRACE, _T("[PTFSMgr]::WaitMount End\n"));
}

void CPTFSMgr::BroadcastLink(WCHAR cLetter, BOOL bRemoved, BOOL safe)
{
    DWORD dwThreadId = 0;
    PBROADCAST_PARAM pBroadCastParam = NULL;

    if (m_hBroadCastThread)
        WaitBroadCastThread();

    pBroadCastParam = (PBROADCAST_PARAM)malloc(sizeof(BROADCAST_PARAM));

    if (NULL == pBroadCastParam)
    {
        CLogUtil::DebugLogString(DLEVEL_WARN, _T("[PTFSMgr]::BroadcastLink Failed to malloc ErrorCode[%d]\n"), GetLastError());
        return;
    }

    HANDLE hStartEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
    pBroadCastParam->cLetter = cLetter;
    pBroadCastParam->bRemoved = bRemoved;
    pBroadCastParam->bSafe = safe;
    pBroadCastParam->hStartEvent = hStartEvent;
    pBroadCastParam->pPTFSMgr = this;

    m_hBroadCastThread = ::CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)BroadcastLinkThreadProc, LPVOID(pBroadCastParam), CREATE_SUSPENDED, &dwThreadId);

    if (m_hBroadCastThread == NULL) 
    {
        free(pBroadCastParam);
        CloseHandle(hStartEvent);
        CLogUtil::DebugLogString(DLEVEL_WARN, _T("[PTFSMgr]::BroadcastLink Failed to CreateThread ErrorCode[%d]\n"), GetLastError());
        return;
    }

    ::SetThreadPriority(m_hBroadCastThread, THREAD_PRIORITY_NORMAL);
    ::ResumeThread(m_hBroadCastThread);

    if (hStartEvent)
    {
        WaitForSingleObject(hStartEvent, INFINITE);
        CloseHandle(hStartEvent);
    }

    Sleep(500);

    WaitMount(cLetter, bRemoved);

    CLogUtil::DebugLogString(DLEVEL_TRACE, _T("[PTFSMgr]::BroadcastLink end Volume[%c], bRemove[%d]\n"), cLetter, bRemoved);
}

BOOL CPTFSMgr::CreateMountPoint(LPCWSTR MountPoint, LPCWSTR DeviceName)
{
    HANDLE handle;
    PREPARSE_DATA_BUFFER reparseData;
    USHORT bufferLength;
    USHORT targetLength;
    BOOL result;
    ULONG resultLength;
    WCHAR targetDeviceName[MAX_PATH];

    ZeroMemory(targetDeviceName, sizeof(targetDeviceName));
    wcscat_s(targetDeviceName, MAX_PATH, L"\\??");
    wcscat_s(targetDeviceName, MAX_PATH, DeviceName);
    wcscat_s(targetDeviceName, MAX_PATH, L"\\");

    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::CreateMountPoint MountPoint[%s], DeviceName[%s]\n", MountPoint, DeviceName);

    handle = CreateFile(MountPoint, GENERIC_WRITE, 0, NULL, OPEN_EXISTING,
        FILE_FLAG_OPEN_REPARSE_POINT | FILE_FLAG_BACKUP_SEMANTICS,
        NULL);

    if (handle == INVALID_HANDLE_VALUE) 
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::CreateMountPoint failed to CreateFile MountPoint[%s], Error[%d]\n", MountPoint, GetLastError());
        return FALSE;
    }

    targetLength = (USHORT)wcslen(targetDeviceName) * sizeof(WCHAR);
    bufferLength = FIELD_OFFSET(REPARSE_DATA_BUFFER, MountPointReparseBuffer.PathBuffer) + targetLength + sizeof(WCHAR) + sizeof(WCHAR);

    reparseData = (PREPARSE_DATA_BUFFER)malloc(bufferLength);
    if (reparseData == NULL)
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::CreateMountPoint failed to malloc Error[%d]\n", GetLastError());
        CloseHandle(handle);
        return FALSE;
    }

    ZeroMemory(reparseData, bufferLength);

    reparseData->ReparseTag = IO_REPARSE_TAG_MOUNT_POINT;
    reparseData->ReparseDataLength = bufferLength - REPARSE_DATA_BUFFER_HEADER_SIZE;

    reparseData->MountPointReparseBuffer.SubstituteNameOffset = 0;
    reparseData->MountPointReparseBuffer.SubstituteNameLength = targetLength;
    reparseData->MountPointReparseBuffer.PrintNameOffset = targetLength + sizeof(WCHAR);
    reparseData->MountPointReparseBuffer.PrintNameLength = 0;

    RtlCopyMemory(reparseData->MountPointReparseBuffer.PathBuffer, targetDeviceName, targetLength);

    result = DeviceIoControl(handle, FSCTL_SET_REPARSE_POINT, reparseData,
        bufferLength, NULL, 0, &resultLength, NULL);

    CloseHandle(handle);
    free(reparseData);

    if (result) 
    {
        CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::CreateMountPoint successed to [%s -> %s]\n", MountPoint, targetDeviceName);
    }
    else 
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::CreateMountPoint failed to [%s -> %s], error[%d]\n", MountPoint, targetDeviceName, GetLastError());
    }
    return result;
}

BOOL CPTFSMgr::IsReadyToMount(WCHAR cLetter)
{
    int nWaitCount = 0;
    CString strDosDeviceMountPoint = L"";
    BOOL bFindReadyToMount = FALSE;

    strDosDeviceMountPoint.Format(L"\\DosDevices\\%c:", cLetter);

    CLogUtil::DebugLogString(DLEVEL_WARN, _T("[PTFSMgr]::IsReadyToMount Begin Volume[%c]\n"), cLetter);

    while (1)
    {
        ULONG ulRead = 0;
        PPTFS_CONTROL pResultsControl = NULL;

        pResultsControl = GetMountPointList(FALSE, &ulRead);

        if (pResultsControl)
        {
            for (ULONG i = 0; i < ulRead; ++i)
            {
                PPTFS_CONTROL pPTFSCtrl = &pResultsControl[i];

				if (!_wcsicmp(pPTFSCtrl->MountPoint, strDosDeviceMountPoint))
				{
					bFindReadyToMount = TRUE;
					break;
				}
			}

            free(pResultsControl);
        }
 
        if (bFindReadyToMount) break;

        Sleep(500);
        if (nWaitCount >= 10)
        {
            CLogUtil::DebugLogString(DLEVEL_ERROR, _T("[PTFSMgr]::IsReadyToMount Wait break\n"));
            break;
        }

        nWaitCount++;
    }

    if (bFindReadyToMount) m_bCompleteMount = TRUE;

    CLogUtil::DebugLogString(DLEVEL_WARN, _T("[PTFSMgr]::IsReadyToMount End Volume[%c], IsReadyToMount[%d]\n"), cLetter, bFindReadyToMount);
    return bFindReadyToMount;
}


UINT WINAPI CPTFSMgr::PTFSKeepAlive(PVOID pThreadParam) 
{
    CPTFSMgr* pPTFSMgr = (CPTFSMgr*)pThreadParam;
    PPTFS_INSTANCE pPTFSInstance = pPTFSMgr->GetPTFSInstance();
    CPTFSDriver* pPTFSDriver = pPTFSMgr->GetPTFSDriver();

    HANDLE hDevice = INVALID_HANDLE_VALUE;
    ULONG ReturnedLength;
    WCHAR rawDeviceName[MAX_PATH];

    ZeroMemory(rawDeviceName, sizeof(rawDeviceName));
    pPTFSMgr->GetRawDeviceName(pPTFSInstance->DeviceName, rawDeviceName, MAX_PATH);

    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::PTFSKeepAlive Start DeviceName[%s], RawDeviceName[%s]\n", pPTFSInstance->DeviceName, rawDeviceName);

    while (TRUE) 
    {
        hDevice = CreateFile(rawDeviceName,
            GENERIC_READ | GENERIC_WRITE,
            FILE_SHARE_READ | FILE_SHARE_WRITE,
            NULL,
            OPEN_EXISTING,
            0,
            NULL 
        );

        if (hDevice == INVALID_HANDLE_VALUE)
        {
            CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::PTFSKeepAlive Failed to CreateFile Path[%s], Error[%d]\n", rawDeviceName, GetLastError());
            break;
        }

        BOOL status = DeviceIoControl(hDevice, IOCTL_KEEPALIVE,
            NULL, 0, NULL, 0,
            &ReturnedLength,
            NULL
        );

        CloseHandle(hDevice);

        if (!status)
            break;

        Sleep(PTFS_KEEPALIVE_TIME);
    }

    _endthreadex(0);
    return STATUS_SUCCESS;
}


BOOL CPTFSMgr::DeleteMountPoint(LPCWSTR MountPoint)
{
    HANDLE handle;
    BOOL result;
    ULONG resultLength;
    REPARSE_GUID_DATA_BUFFER reparseData = { 0 };

    handle = CreateFile(MountPoint, GENERIC_WRITE, 0, NULL, OPEN_EXISTING,
        FILE_FLAG_OPEN_REPARSE_POINT | FILE_FLAG_BACKUP_SEMANTICS,
        NULL);

    if (handle == INVALID_HANDLE_VALUE)
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DeleteMountPoint Failed to CreateFile Path[%s], Error[%d]\n", MountPoint, GetLastError());
        return FALSE;
    }

    reparseData.ReparseTag = IO_REPARSE_TAG_MOUNT_POINT;

    result = DeviceIoControl(handle, FSCTL_DELETE_REPARSE_POINT, &reparseData,
        REPARSE_GUID_DATA_BUFFER_HEADER_SIZE, NULL, 0,
        &resultLength, NULL);

    CloseHandle(handle);

    if (result) 
    {
        CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::DeleteMountPoint Success MountPoint[%s]\n", MountPoint);
    }
    else 
    {
        if (GetLastError() == ERROR_NOT_A_REPARSE_POINT) 
            return TRUE;

        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DeleteMountPoint failed to MountPoint[%s], error[%d]\n", MountPoint, GetLastError());
    }

    return result;
}

BOOL CPTFSMgr::RemoveMountPointEx(LPCWSTR MountPoint, BOOL Safe)
{
    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::RemoveMountPointEx MountPoint[%s], IsSafe[%d]\n", MountPoint, Safe);

    if (MountPoint != NULL) 
    {
        size_t length = wcslen(MountPoint);
        if (length > 0) 
        {
            WCHAR mountPoint[MAX_PATH];

            if (IsMountPointDriveLetter(MountPoint))
            {
                wcscpy_s(mountPoint, sizeof(mountPoint) / sizeof(WCHAR), L"C:");
                mountPoint[0] = MountPoint[0];
            }
            else 
            {
                wcscpy_s(mountPoint, sizeof(mountPoint) / sizeof(WCHAR), MountPoint);
                if (mountPoint[length - 1] == L'\\') 
                    mountPoint[length - 1] = L'\0';
            }

            if (SendGlobalReleaseIRP(mountPoint)) 
            {
                if (!IsMountPointDriveLetter(MountPoint)) 
                {
                    wcscat_s(mountPoint, sizeof(mountPoint) / sizeof(WCHAR), L"\\");
                    return DeleteMountPoint(mountPoint);
                }
                else 
                {
//                    BroadcastLink(MountPoint[0], TRUE, Safe);
                    return TRUE;
                }
            }
            else
            {
                WaitMount(MountPoint[0], TRUE);
                if (FALSE == Safe)
                {
                    SendReleaseIRP(m_pPTFSInstance->DeviceName);
                    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::RemoveMountPointEx Force Remove, DeviceName[%s]\n", m_pPTFSInstance->DeviceName);
                }
            }
        }
    }
    return FALSE;
}

BOOL CPTFSMgr::PTFSRemoveMountPoint(LPCWSTR MountPoint, BOOL bForce)
{
    return RemoveMountPointEx(MountPoint, !bForce);
}

BOOL CPTFSMgr::PTFSMount(LPCWSTR MountPoint, LPCWSTR DeviceName, PPTFS_OPTIONS pPTFSOptions)
{
    if (MountPoint != NULL) 
    {
        if (!IsMountPointDriveLetter(MountPoint)) 
        {
            if (pPTFSOptions->Options & PTFS_OPTION_MOUNT_MANAGER) 
                return TRUE;

            return CreateMountPoint(MountPoint, DeviceName);
        }
 //       else 
 //           BroadcastLink(MountPoint[0], FALSE, TRUE);
    }
    return TRUE;
}

BOOL CPTFSMgr::PTFSUnmount(WCHAR DriveLetter, BOOL bForce)
{
    WCHAR mountPoint[MAX_PATH] = L"M:";
    mountPoint[0] = DriveLetter;

    if (m_hKeepAlive != INVALID_HANDLE_VALUE)
    {
        CloseHandle(m_hKeepAlive);
        m_hKeepAlive = INVALID_HANDLE_VALUE;
    }

    return PTFSRemoveMountPoint(mountPoint, bForce);
}

int  CPTFSMgr::DoInintialize(PPTFS_OPTIONS pPTFSOptions, PPTFS_OPERATIONS pPTFSOperations, HANDLE hMountWaitEvent)
{
	HANDLE threadIds[PTFS_MAX_THREAD];
	HANDLE legacyKeepAliveThreadIds = NULL;
	BOOL keepalive_active = FALSE;

	if (pPTFSOptions->Options & PTFS_OPTION_DEBUG)
		SetDebugMode(TRUE);

	if (pPTFSOptions->Options & PTFS_OPTION_STDERR)
	{
		SetUseStdErr(TRUE);
		SetDebugMode(TRUE);
	}

	if (pPTFSOptions->Options & PTFS_OPTION_NETWORK && !IsMountPointDriveLetter(pPTFSOptions->MountPoint))
	{
		pPTFSOptions->Options &= ~PTFS_OPTION_NETWORK;
		CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::DoInintialize Mount point folder is specified with network Device Disable network hDevice\n");
	}

	CheckAllocationUnitSectorSize(pPTFSOptions);

	if (pPTFSOptions->ThreadCount == 0)
		pPTFSOptions->ThreadCount = 5;
	else if (PTFS_MAX_THREAD < pPTFSOptions->ThreadCount)
		pPTFSOptions->ThreadCount = PTFS_MAX_THREAD;

	if (FALSE == m_PTFSDriver.OpenGlobalDevice())
	{
		CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoInintialize Failed to Install PTFileSystem driver\n");
        if (hMountWaitEvent) SetEvent(hMountWaitEvent);
		return PTFS_DRIVER_INSTALL_ERROR;
	}

	if (FALSE == NewPTFSInstance())
	{
		CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoInintialize Failed to New m_pPTFSInstance\n");
        if (hMountWaitEvent) SetEvent(hMountWaitEvent);
        return PTFS_DRIVER_INSTALL_ERROR;
	}

	m_pPTFSInstance->pPTFSOptions = pPTFSOptions;
	m_pPTFSInstance->pPTFSOperations = pPTFSOperations;

	if (pPTFSOptions->MountPoint != NULL)
	{
		wcscpy_s(m_pPTFSInstance->MountPoint, sizeof(m_pPTFSInstance->MountPoint) / sizeof(WCHAR), pPTFSOptions->MountPoint);
		if (IsMountPointDriveLetter(m_pPTFSInstance->MountPoint) && !CheckDriveLetterAvailability(m_pPTFSInstance->MountPoint[0]))
		{
			CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoInintialize Failed to CheckDriveLetterAvailability\n");
			DeletePTFSInstance();
            if (hMountWaitEvent) SetEvent(hMountWaitEvent);
            return PTFS_MOUNT_ERROR;
		}
	}

	if (pPTFSOptions->UNCName != NULL)
		wcscpy_s(m_pPTFSInstance->UNCName, sizeof(m_pPTFSInstance->UNCName) / sizeof(WCHAR), pPTFSOptions->UNCName);

	if (!DoStart())
	{
		CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoInintialize Failed to DoStart MountPoint[%s]\n", m_pPTFSInstance->MountPoint);
		DeletePTFSInstance();
        if (hMountWaitEvent) SetEvent(hMountWaitEvent);
        return PTFS_START_ERROR;
	}

    m_bCompleteMount = m_bCompleteWaitMount = FALSE;

	for (ULONG i = 0; i < pPTFSOptions->ThreadCount; ++i)
	{
		threadIds[i] = (HANDLE)_beginthreadex(NULL,
			0,
			PTFSLoopThreadProc,
			this, // param
			0, // create flag
			NULL);
	}

	if (!PTFSMount(m_pPTFSInstance->MountPoint, m_pPTFSInstance->DeviceName, pPTFSOptions))
	{
		SendReleaseIRP(m_pPTFSInstance->DeviceName);
		CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoInintialize Failed to PTFSMount\n");
        m_bCompleteMount = TRUE;
        if (hMountWaitEvent) SetEvent(hMountWaitEvent);
        return PTFS_MOUNT_ERROR;
	}

    if (hMountWaitEvent) SetEvent(hMountWaitEvent);

	wchar_t keepalive_path[128];
	StringCbPrintfW(keepalive_path, sizeof(keepalive_path), L"\\\\.%s%s", m_pPTFSInstance->DeviceName, KEEPALIVE_FILE_NAME);
	m_hKeepAlive = CreateFile(keepalive_path, GENERIC_READ, 0, NULL, OPEN_EXISTING, 0, NULL);
	if (m_hKeepAlive == INVALID_HANDLE_VALUE)
	{
		CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoInintialize Failed to open keepalive file: Path[%s], ErrorCode[%d]\n", keepalive_path, GetLastError());
	}
	else
	{
		DWORD keepalive_bytes_returned = 0;
		keepalive_active = DeviceIoControl(m_hKeepAlive, FSCTL_ACTIVATE_KEEPALIVE, NULL, 0,
			NULL, 0, &keepalive_bytes_returned, NULL);

		if (!keepalive_active)
			CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoInintialize Failed to activate keepalive handle.\n");
	}

	if (!keepalive_active)
	{
		CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoInintialize Enable legacy keepalive.\n");

		legacyKeepAliveThreadIds = (HANDLE)_beginthreadex(NULL,
			0,
			PTFSKeepAlive,
			this,
			0,
			NULL);
	}

	if (pPTFSOptions->Options & PTFS_OPTION_ENABLE_NOTIFICATION_API)
	{
		wchar_t notify_path[128];
		StringCbPrintfW(notify_path, sizeof(notify_path), L"\\\\.%s%s", m_pPTFSInstance->DeviceName, NOTIFICATION_FILE_NAME);
		m_hNotify = CreateFile(notify_path, GENERIC_READ,
			FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE, NULL,
			OPEN_EXISTING, FILE_FLAG_BACKUP_SEMANTICS, NULL);
		if (m_hNotify == INVALID_HANDLE_VALUE)
		{
			CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::DoInintialize Failed to open notify file: %s\n", notify_path);
		}
	}

	CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::DoInintialize mounted[%s -> %s]\n", m_pPTFSInstance->MountPoint, m_pPTFSInstance->DeviceName);

	if (pPTFSOperations->Mounted)
	{
		PTFS_FILE_INFO fileInfo;
		RtlZeroMemory(&fileInfo, sizeof(PTFS_FILE_INFO));
		fileInfo.pPTFSOptions = pPTFSOptions;
		pPTFSOperations->Mounted(&fileInfo);
	}

	WaitForMultipleObjects(pPTFSOptions->ThreadCount, threadIds, TRUE, INFINITE);
	for (ULONG i = 0; i < pPTFSOptions->ThreadCount; ++i) 
    {
		CloseHandle(threadIds[i]);
	}

	if (legacyKeepAliveThreadIds) 
    {
		WaitForSingleObject(legacyKeepAliveThreadIds, INFINITE);
		CloseHandle(legacyKeepAliveThreadIds);
	}

    if (m_hNotify != INVALID_HANDLE_VALUE)
    {
        CloseHandle(m_hNotify);
        m_hNotify = INVALID_HANDLE_VALUE;
    }

    if (m_hKeepAlive != INVALID_HANDLE_VALUE)
    {
        CloseHandle(m_hKeepAlive);
        m_hKeepAlive = INVALID_HANDLE_VALUE;
    }


	if (pPTFSOperations->Unmounted) 
    {
		PTFS_FILE_INFO fileInfo;
		RtlZeroMemory(&fileInfo, sizeof(PTFS_FILE_INFO));
		fileInfo.pPTFSOptions = pPTFSOptions;
		pPTFSOperations->Unmounted(&fileInfo);
	}

	Sleep(1000);

    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::DoInintialize unload\n");
	DeletePTFSInstance();
	return PTFS_SUCCESS;
}

BOOL CPTFSMgr::SendReleaseIRP(LPCWSTR DeviceName)
{
    ULONG returnedLength;
    WCHAR rawDeviceName[MAX_PATH];

    ZeroMemory(rawDeviceName, sizeof(rawDeviceName));
    GetRawDeviceName(DeviceName, rawDeviceName, MAX_PATH);

    CLogUtil::DebugLogString(DLEVEL_TRACE, L"PTFSMgr::SendReleaseIRP send release to DeviceName[%s], RawDeviceName[%s]\n", DeviceName, rawDeviceName);

    if (!m_PTFSDriver.SendToDevice(rawDeviceName,IOCTL_EVENT_RELEASE, NULL, 0, NULL, 0, &returnedLength))
    {
        CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::SendReleaseIRP failed to SendToDevice [%s]\n", DeviceName);
        return FALSE;
    }

    return TRUE;
}

BOOL CPTFSMgr::SendGlobalReleaseIRP(LPCWSTR MountPoint) 
{
    if (MountPoint != NULL) 
    {
        size_t length = wcslen(MountPoint);
        if (length > 0) 
        {
            ULONG returnedLength;
            ULONG inputLength = sizeof(PUNICODE_STRING_INTERMEDIATE) + (MAX_PATH * sizeof(WCHAR));
            PUNICODE_STRING_INTERMEDIATE szMountPoint = (PUNICODE_STRING_INTERMEDIATE)malloc(inputLength);

            if (szMountPoint != NULL) 
            {
                ZeroMemory(szMountPoint, inputLength);
                szMountPoint->MaximumLength = MAX_PATH * sizeof(WCHAR);
                szMountPoint->Length = (USHORT)(length * sizeof(WCHAR));
                CopyMemory(szMountPoint->Buffer, MountPoint, szMountPoint->Length);

                CLogUtil::DebugLogString(DLEVEL_WARN, L"PTFSMgr::SendGlobalReleaseIRP send global release [%s]\n", MountPoint);

                if (!m_PTFSDriver.SendToGlobalDevice( IOCTL_EVENT_RELEASE,
                    szMountPoint, inputLength, NULL, 0, &returnedLength)) 
                {
                    CLogUtil::DebugLogString(DLEVEL_ERROR, L"PTFSMgr::SendGlobalReleaseIRP failed to SendToDevice [%s]\n", MountPoint);
                    free(szMountPoint);
                    return FALSE;
                }

                free(szMountPoint);
                return TRUE;
            }
        }
    }

    return FALSE;
}

BOOL CPTFSMgr::SetDebugModeInDevice(ULONG Mode) 
{
    ULONG returnedLength;
    return m_PTFSDriver.SendToGlobalDevice(IOCTL_SET_DEBUG_MODE, &Mode,
        sizeof(ULONG), NULL, 0, &returnedLength);
}

PPTFS_CONTROL CPTFSMgr::GetMountPointList(BOOL uncOnly, PULONG nbRead)
{
    ULONG returnedLength = 0;
    PPTFS_CONTROL pPTFSControl = NULL;
    PPTFS_CONTROL pResultsControl = NULL;
    ULONG bufferLength = 26 * sizeof(*pPTFSControl);
    BOOL success;

    *nbRead = 0;

    do 
    {
        if (pPTFSControl != NULL)
        {
            free(pPTFSControl);
            pPTFSControl = NULL;
        }

        pPTFSControl = (PPTFS_CONTROL)malloc(bufferLength);
        if (pPTFSControl == NULL)
            return NULL;

        ZeroMemory(pPTFSControl, bufferLength);

        success = m_PTFSDriver.SendToGlobalDevice( IOCTL_EVENT_MOUNTPOINT_LIST,
                NULL, 0, pPTFSControl, bufferLength, &returnedLength);

        if (!success && GetLastError() != ERROR_MORE_DATA) 
        {
            free(pPTFSControl);
            return NULL;
        }

        bufferLength *= 2;
    } while (!success);

    if (returnedLength == 0)
    {
        free(pPTFSControl);
        return NULL;
    }

    *nbRead = returnedLength / sizeof(PTFS_CONTROL);
    pResultsControl = (PPTFS_CONTROL)malloc(returnedLength);
    if (pResultsControl != NULL) 
    {
        ZeroMemory(pResultsControl, returnedLength);
        for (ULONG i = 0; i < *nbRead; ++i)
        {
            if (!uncOnly || wcscmp(pPTFSControl[i].UNCName, L"") != 0)
                CopyMemory(&pResultsControl[i], &pPTFSControl[i], sizeof(PTFS_CONTROL));
        }
    }
    free(pPTFSControl);
    return pResultsControl;
}

VOID CPTFSMgr::ReleaseMountPointList(PPTFS_CONTROL list) 
{ 
    free(list); 
}

void  CPTFSMgr::DoFinalize()
{
    EnterCriticalSection(&m_InstanceCriticalSection);

    while (!IsListEmpty(&m_InstanceList)) 
    {
        PLIST_ENTRY entry = RemoveHeadList(&m_InstanceList);
        PPTFS_INSTANCE m_pPTFSInstance = CONTAINING_RECORD(entry, PTFS_INSTANCE, ListEntry);
        RemoveMountPointEx(m_pPTFSInstance->MountPoint, FALSE);
        free(m_pPTFSInstance);
    }

    LeaveCriticalSection(&m_InstanceCriticalSection);
    DeleteCriticalSection(&m_InstanceCriticalSection);
}

void CPTFSMgr::MapKernelToUserCreateFileFlags(ACCESS_MASK DesiredAccess, ULONG FileAttributes, ULONG CreateOptions, ULONG CreateDisposition,
	ACCESS_MASK* outDesiredAccess, DWORD* outFileAttributesAndFlags, DWORD* outCreationDisposition)
{
	BOOL genericRead = FALSE, genericWrite = FALSE, genericExecute = FALSE, genericAll = FALSE;

	if (outFileAttributesAndFlags)
	{
		*outFileAttributesAndFlags = FileAttributes;
		PTFSMapKernelBit(*outFileAttributesAndFlags, CreateOptions, FILE_FLAG_WRITE_THROUGH, FILE_WRITE_THROUGH);
		PTFSMapKernelBit(*outFileAttributesAndFlags, CreateOptions, FILE_FLAG_SEQUENTIAL_SCAN, FILE_SEQUENTIAL_ONLY);
		PTFSMapKernelBit(*outFileAttributesAndFlags, CreateOptions, FILE_FLAG_RANDOM_ACCESS, FILE_RANDOM_ACCESS);
		PTFSMapKernelBit(*outFileAttributesAndFlags, CreateOptions, FILE_FLAG_NO_BUFFERING, FILE_NO_INTERMEDIATE_BUFFERING);
		PTFSMapKernelBit(*outFileAttributesAndFlags, CreateOptions, FILE_FLAG_OPEN_REPARSE_POINT, FILE_OPEN_REPARSE_POINT);
		PTFSMapKernelBit(*outFileAttributesAndFlags, CreateOptions, FILE_FLAG_DELETE_ON_CLOSE, FILE_DELETE_ON_CLOSE);
		PTFSMapKernelBit(*outFileAttributesAndFlags, CreateOptions, FILE_FLAG_BACKUP_SEMANTICS, FILE_OPEN_FOR_BACKUP_INTENT);

#if (_WIN32_WINNT >= _WIN32_WINNT_WIN8)
		PTFSMapKernelBit(*outFileAttributesAndFlags, CreateOptions,FILE_FLAG_SESSION_AWARE, FILE_SESSION_AWARE);
#endif
	}

	if (outCreationDisposition) 
    {
		switch (CreateDisposition) {
		case FILE_CREATE:
			*outCreationDisposition = CREATE_NEW;
			break;
		case FILE_OPEN:
			*outCreationDisposition = OPEN_EXISTING;
			break;
		case FILE_OPEN_IF:
			*outCreationDisposition = OPEN_ALWAYS;
			break;
		case FILE_OVERWRITE:
			*outCreationDisposition = TRUNCATE_EXISTING;
			break;
		case FILE_SUPERSEDE:
		case FILE_OVERWRITE_IF:
			*outCreationDisposition = CREATE_ALWAYS;
			break;
		default:
			*outCreationDisposition = 0;
			break;
		}
	}

	if (outDesiredAccess) 
    {
		*outDesiredAccess = DesiredAccess;

		if ((*outDesiredAccess & FILE_GENERIC_READ) == FILE_GENERIC_READ) 
        {
			*outDesiredAccess |= GENERIC_READ;
			genericRead = TRUE;
		}

		if ((*outDesiredAccess & FILE_GENERIC_WRITE) == FILE_GENERIC_WRITE)
        {
			*outDesiredAccess |= GENERIC_WRITE;
			genericWrite = TRUE;
		}

		if ((*outDesiredAccess & FILE_GENERIC_EXECUTE) == FILE_GENERIC_EXECUTE) 
        {
			*outDesiredAccess |= GENERIC_EXECUTE;
			genericExecute = TRUE;
		}

		if ((*outDesiredAccess & FILE_ALL_ACCESS) == FILE_ALL_ACCESS) 
        {
			*outDesiredAccess |= GENERIC_ALL;
			genericAll = TRUE;
		}

		if (genericRead)
			*outDesiredAccess &= ~FILE_GENERIC_READ;
		if (genericWrite)
			*outDesiredAccess &= ~FILE_GENERIC_WRITE;
		if (genericExecute)
			*outDesiredAccess &= ~FILE_GENERIC_EXECUTE;
		if (genericAll)
			*outDesiredAccess &= ~FILE_ALL_ACCESS;
	}
}

