#include "StdAfx.h"
#include "PTFSDispatch.h"
#include "PTFSLib.h"
#include "PTFSUtil.h"
#include "util/util.h"
#include "PTFSFileinfo.h"
#include "List/List.h"
#include <winioctl.h>

CPTFSDispatch::CPTFSDispatch(void)
{
}

CPTFSDispatch::~CPTFSDispatch(void)
{
}
///////////////////////////////////////////////////////////////

PEVENT_INFORMATION CPTFSDispatch::DispatchCommon(PEVENT_CONTEXT pEventContext, ULONG ulSizeOfEventInfo, PPTFS_INSTANCE pPTFSInstance,
	PPTFS_FILE_INFO pPTFSFileInfo, PPTFS_OPEN_INFO* ppPTFSOpenInfo)
{
	PEVENT_INFORMATION pEventInfo = (PEVENT_INFORMATION)malloc(ulSizeOfEventInfo);

	if(NULL == pEventInfo)
		return pEventInfo;

	RtlZeroMemory(pEventInfo, ulSizeOfEventInfo);
	RtlZeroMemory(pPTFSFileInfo, sizeof(PTFS_FILE_INFO));

	pEventInfo->BufferLength = 0;
	pEventInfo->SerialNumber = pEventContext->SerialNumber;

	pPTFSFileInfo->ProcessId	= pEventContext->ProcessId;
	pPTFSFileInfo->PTFSOptions = pPTFSInstance->PTFSOptions;
	if (pEventContext->FileFlags & PTFS_DELETE_ON_CLOSE) {
		pPTFSFileInfo->DeleteOnClose = 1;
	}
	if (pEventContext->FileFlags & PTFS_PAGING_IO) {
		pPTFSFileInfo->PagingIo = 1;
	}
	if (pEventContext->FileFlags & PTFS_WRITE_TO_END_OF_FILE) {
		pPTFSFileInfo->WriteToEndOfFile = 1;
	}
	if (pEventContext->FileFlags & PTFS_SYNCHRONOUS_IO) {
		pPTFSFileInfo->SynchronousIo = 1;
	}
	if (pEventContext->FileFlags & PTFS_NOCACHE) {
		pPTFSFileInfo->Nocache = 1;
	}

	*ppPTFSOpenInfo = CPTFSUtil::GetPTFSOpenInfo(pEventContext, pPTFSInstance);
	if (*ppPTFSOpenInfo == NULL) {
		CLogUtil::DebugLogString(DLEVEL_LOUD, _T("error pOpenInfo is NULL\n"));
		return pEventInfo;
	}

	pPTFSFileInfo->Context		= (ULONG64)(*ppPTFSOpenInfo)->UserContext;
	pPTFSFileInfo->IsDirectory	= (UCHAR)(*ppPTFSOpenInfo)->IsDirectory;
	pPTFSFileInfo->PTFSContext = (ULONG64)(*ppPTFSOpenInfo);

	pEventInfo->Context = (ULONG64)(*ppPTFSOpenInfo);

	return pEventInfo;
}

int WINAPI CPTFSDispatch::FillFileData(PWIN32_FIND_DATAW	pFindData, PPTFS_FILE_INFO	FileInfo)
{
	PLIST_ENTRY pListHead = ((PPTFS_OPEN_INFO)FileInfo->PTFSContext)->DirListHead;
	PPTFS_FIND_DATA	 pPTFSFindData;
	
	pPTFSFindData = (PPTFS_FIND_DATA)malloc(sizeof(PTFS_FIND_DATA));

	if(NULL == pPTFSFindData)
		return 0;

	ZeroMemory(pPTFSFindData, sizeof(PTFS_FIND_DATA));
	InitializeListHead(&pPTFSFindData->ListEntry);

	pPTFSFindData->FindData = *pFindData;

	InsertTailList(pListHead, &pPTFSFindData->ListEntry);
	return 0;
}

VOID CPTFSDispatch::SendWriteRequest(HANDLE hDevice, PEVENT_INFORMATION pEventInfo, ULONG ulEventLength,
	PVOID	 pBuffer, ULONG ulBufferLength)
{
	BOOL		bStatus;
	ULONG	ulReturnedLength;

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("SendWriteRequest\n"));

	bStatus = DeviceIoControl(
					hDevice,		            
					IOCTL_EVENT_WRITE,
					pEventInfo,	
					ulEventLength,
					pBuffer,	
					ulBufferLength,
					&ulReturnedLength,
					NULL);

	if ( !bStatus ) {
		CLogUtil::DebugLogString(DLEVEL_WARN, _T("Ioctl failed with code %d\n"), GetLastError());
	}

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("SendWriteRequest got %d bytes\n"), ulReturnedLength);
}

//////////////////////////////////////////////////////////////

//IRP_MJ_CREATE
VOID CPTFSDispatch::DispatchCreate(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	static ULONG					ulEventId = 0;
	ULONG						ulLength	  = sizeof(EVENT_INFORMATION);
	PEVENT_INFORMATION	pEventInfo = (PEVENT_INFORMATION)malloc(ulLength);
	int								nStatus;
	PTFS_FILE_INFO		fileInfo;
	DWORD						dwDisposition;
	PPTFS_OPEN_INFO		pOpenInfo;
	BOOL							bDirectoryRequested = FALSE;
	DWORD						dwOptions;

	CPTFSUtil::CheckFileName(pEventContext->Create.FileName);

	RtlZeroMemory(pEventInfo, ulLength);
	RtlZeroMemory(&fileInfo, sizeof(PTFS_FILE_INFO));

	pEventInfo->BufferLength = 0;
	pEventInfo->SerialNumber = pEventContext->SerialNumber;

	fileInfo.ProcessId = pEventContext->ProcessId;
	fileInfo.PTFSOptions = pPTFSInstance->PTFSOptions;

	// PTFS_OPEN_INFO is structure for a opened file
	// this will be freed by Close
	pOpenInfo = (PPTFS_OPEN_INFO)malloc(sizeof(PTFS_OPEN_INFO));

	if(NULL == pOpenInfo)
		return;

	ZeroMemory(pOpenInfo, sizeof(PTFS_OPEN_INFO));
	pOpenInfo->OpenCount = 2;
	pOpenInfo->EventContext = pEventContext;
	pOpenInfo->PTFSInstance = pPTFSInstance;
	fileInfo.PTFSContext = (ULONG64)pOpenInfo;

	// pass it to driver and when the same handle is used get it back
	pEventInfo->Context = (ULONG64)pOpenInfo;

	// The high 8 bits of this parameter correspond to the Disposition parameter
	dwDisposition = (pEventContext->Create.CreateOptions >> 24) & 0x000000ff;

	nStatus = -1; // in case being not dispatched
	
	// The low 24 bits of this member correspond to the CreateOptions parameter
	dwOptions = pEventContext->Create.CreateOptions & FILE_VALID_OPTION_FLAGS;

	// to open directory
	// even if this flag is not specifed, 
	// there is a case to open a directory
	if (dwOptions & FILE_DIRECTORY_FILE) {
		bDirectoryRequested = TRUE;
	}

	// to open no directory file
	// event if this flag is not specified,
	// there is a case to open non directory file
	if (dwOptions & FILE_NON_DIRECTORY_FILE) {
	}

	if (dwOptions & FILE_DELETE_ON_CLOSE) {
		pEventContext->Create.FileAttributes |= FILE_FLAG_DELETE_ON_CLOSE;
	}

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("###Create %04d\n"), ulEventId);
	pOpenInfo->EventId = ulEventId++;

	// make a directory or open
	if (bDirectoryRequested) {
		fileInfo.IsDirectory = TRUE;

		if (dwDisposition == FILE_CREATE || dwDisposition == FILE_OPEN_IF) {
			if (pPTFSInstance->PTFSOperations->CreateDirectory) {
				nStatus = pPTFSInstance->PTFSOperations->CreateDirectory(
							pEventContext->Create.FileName, &fileInfo);
			}
		} else if(dwDisposition == FILE_OPEN) {
			if (pPTFSInstance->PTFSOperations->OpenDirectory) {
				nStatus = pPTFSInstance->PTFSOperations->OpenDirectory(
							pEventContext->Create.FileName, &fileInfo);
			}
		} else {
			CLogUtil::DebugLogString(DLEVEL_LOUD, _T("### Create other disposition : %d\n"), dwDisposition);
		}
	
	// open a file
	} else {
		DWORD creationDisposition = OPEN_EXISTING;
		fileInfo.IsDirectory = FALSE;
		CLogUtil::DebugLogString(DLEVEL_LOUD, _T("CreateDisposition %X\n"), dwDisposition);
		switch(dwDisposition) {
			case FILE_CREATE:
				creationDisposition = CREATE_NEW;
				break;
			case FILE_OPEN:
				creationDisposition = OPEN_EXISTING;
				break;
			case FILE_OPEN_IF:
				creationDisposition = OPEN_ALWAYS;
				break;
			case FILE_OVERWRITE:
				creationDisposition = TRUNCATE_EXISTING;
				break;
			case FILE_OVERWRITE_IF:
				creationDisposition = CREATE_ALWAYS;
				break;
			default:
				CLogUtil::DebugLogString(DLEVEL_LOUD, _T("### Create other disposition : %d\n"), dwDisposition);
				break;
		}
		
		if(pPTFSInstance->PTFSOperations->CreateFile) {
			nStatus = pPTFSInstance->PTFSOperations->CreateFile(
									pEventContext->Create.FileName,
									pEventContext->Create.DesiredAccess,
									pEventContext->Create.ShareAccess,
									creationDisposition,
									pEventContext->Create.FileAttributes,
									&fileInfo);
		}
	}

	// save the information about this access in PTFS_OPEN_INFO
	pOpenInfo->IsDirectory = fileInfo.IsDirectory;
	pOpenInfo->UserContext = fileInfo.Context;

	// FILE_CREATED
	// FILE_DOES_NOT_EXIST
	// FILE_EXISTS
	// FILE_OPENED
	// FILE_OVERWRITTEN
	// FILE_SUPERSEDED

	if (nStatus < 0) {

		int nError = nStatus * -1;		
		CLogUtil::DebugLogString(DLEVEL_LOUD, _T("CreateFile Status = %d\n"), nStatus);

		if (pEventContext->Flags & SL_OPEN_TARGET_DIRECTORY) {
			CLogUtil::DebugLogString(DLEVEL_LOUD, _T("SL_OPEN_TARGET_DIRECTORY spcefied\n"));
		}
		pEventInfo->Create.Information = FILE_DOES_NOT_EXIST;

		switch(nError) {
		case ERROR_FILE_NOT_FOUND:
			if (pEventContext->Flags & SL_OPEN_TARGET_DIRECTORY)
				pEventInfo->Status = STATUS_SUCCESS;
			else
				pEventInfo->Status = STATUS_OBJECT_NAME_NOT_FOUND;
			break;
		case ERROR_PATH_NOT_FOUND:
			pEventInfo->Status = STATUS_OBJECT_PATH_NOT_FOUND;
			break;
		case ERROR_ACCESS_DENIED:
			pEventInfo->Status = STATUS_ACCESS_DENIED;
			break;
		case ERROR_SHARING_VIOLATION:
			pEventInfo->Status = STATUS_SHARING_VIOLATION;
			break;
		case ERROR_INVALID_NAME:
			pEventInfo->Status = STATUS_OBJECT_NAME_NOT_FOUND;
			break;
		case ERROR_FILE_EXISTS:
		case ERROR_ALREADY_EXISTS:		
			pEventInfo->Status = STATUS_OBJECT_NAME_COLLISION;
			pEventInfo->Create.Information = FILE_EXISTS;
			break;
		case ERROR_PRIVILEGE_NOT_HELD:
			pEventInfo->Status = STATUS_PRIVILEGE_NOT_HELD;
			break;
		case ERROR_NOT_READY:
			pEventInfo->Status = STATUS_DEVICE_NOT_READY;
			break;
		default:
			pEventInfo->Status = STATUS_INVALID_PARAMETER;
			CLogUtil::DebugLogString(DLEVEL_ERROR, _T("Create got unknown error code %d\n"), nError);
		}


		if (pEventInfo->Status != STATUS_SUCCESS) {
			// Needs to free pOpenInfo because Close is never called.
			free(pOpenInfo);
			pEventInfo->Context = 0;
		}

	} else {
		pEventInfo->Status = STATUS_SUCCESS;
		pEventInfo->Create.Information = FILE_OPENED;

		if (dwDisposition == FILE_CREATE ||
			dwDisposition == FILE_OPEN_IF ||
			dwDisposition == FILE_OVERWRITE_IF) {

			if (nStatus != ERROR_ALREADY_EXISTS) {
				pEventInfo->Create.Information = FILE_CREATED;
			}
		}

		if ((dwDisposition == FILE_OVERWRITE_IF || dwDisposition == FILE_OVERWRITE) &&
			pEventInfo->Create.Information != FILE_CREATED) {
			
			pEventInfo->Create.Information = FILE_OVERWRITTEN;
		}

		if (fileInfo.IsDirectory)
			pEventInfo->Create.Flags |= PTFS_FILE_DIRECTORY;
	}
	
	theApp.SendEventInformation(hDevice, pEventInfo, ulLength, pPTFSInstance);
	free(pEventInfo);
	return;
}

VOID CPTFSDispatch::DispatchCleanup(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION	pEventInfo;
	PTFS_FILE_INFO		fileInfo;	
	PPTFS_OPEN_INFO		pOpenInfo = NULL;
	ULONG						ulSizeOfEventInfo = sizeof(EVENT_INFORMATION);

	CPTFSUtil::CheckFileName(pEventContext->Cleanup.FileName);

	pEventInfo = DispatchCommon(pEventContext, ulSizeOfEventInfo, pPTFSInstance, &fileInfo, &pOpenInfo);
	
	pEventInfo->Status = STATUS_SUCCESS; // return success at any case

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("###Cleanup %04d\n"), pOpenInfo != NULL ? pOpenInfo->EventId : -1);

	if (pPTFSInstance->PTFSOperations->Cleanup) {
		// ignore return value
		pPTFSInstance->PTFSOperations->Cleanup(pEventContext->Cleanup.FileName,&fileInfo);
	}

	if(pOpenInfo)
		pOpenInfo->UserContext = fileInfo.Context;

	theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, pPTFSInstance);
	free(pEventInfo);
	return;
}

VOID CPTFSDispatch::DispatchClose(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION	pEventInfo;
	PTFS_FILE_INFO		fileInfo;	
	PPTFS_OPEN_INFO		pOpenInfo = NULL;
	ULONG						ulSizeOfEventInfo = sizeof(EVENT_INFORMATION);

	CPTFSUtil::CheckFileName(pEventContext->Close.FileName);

	pEventInfo = DispatchCommon(pEventContext, ulSizeOfEventInfo, pPTFSInstance, &fileInfo, &pOpenInfo);

	pEventInfo->Status = STATUS_SUCCESS; // return success at any case

	CLogUtil::DebugLogString(DLEVEL_LOUD,_T("###Close %04d\n"), pOpenInfo != NULL ? pOpenInfo->EventId : -1);

	if (pPTFSInstance->PTFSOperations->CloseFile) {
		// ignore return value
		pPTFSInstance->PTFSOperations->CloseFile(pEventContext->Close.FileName, &fileInfo);
	}

	// do not send it to the driver
	//SendEventInformation(hDevice, pEventInfo, length);

	if (pOpenInfo != NULL) {
		EnterCriticalSection(&pPTFSInstance->CriticalSection);
		pOpenInfo->OpenCount--;
		LeaveCriticalSection(&pPTFSInstance->CriticalSection);
	}
	CPTFSUtil::ReleasePTFSOpenInfo(pEventInfo, pPTFSInstance);
	free(pEventInfo);

	return;
}

VOID CPTFSDispatch::DispatchDirectoryInformation(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION	pEventInfo;
	PTFS_FILE_INFO	fileInfo;
	PPTFS_OPEN_INFO	pOpenInfo = NULL;
	int					nStatus = 0;
	ULONG				ulFileInfoClass = pEventContext->Directory.FileInformationClass;
	ULONG				ulSizeOfEventInfo = sizeof(EVENT_INFORMATION) - 8 + pEventContext->Directory.BufferLength;
	BOOLEAN				bPatternCheck = TRUE;

	CPTFSUtil::CheckFileName(pEventContext->Directory.DirectoryName);

	pEventInfo = DispatchCommon(pEventContext, ulSizeOfEventInfo, pPTFSInstance, &fileInfo, &pOpenInfo);

	// check whether this is handled FileInfoClass
	if (ulFileInfoClass != FileDirectoryInformation &&
		ulFileInfoClass != FileFullDirectoryInformation &&
		ulFileInfoClass != FileNamesInformation &&
		ulFileInfoClass != FileIdBothDirectoryInformation &&
		ulFileInfoClass != FileBothDirectoryInformation) {
		
		CLogUtil::DebugLogString(DLEVEL_INFO,_T("not suported type %d\n"), ulFileInfoClass);

		// send directory info to driver
		pEventInfo->BufferLength = 0;
		pEventInfo->Status = STATUS_NOT_IMPLEMENTED;
		theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, pPTFSInstance);
		free(pEventInfo);
		return;
	}


	// IMPORTANT!!
	// this buffer length is fixed in MatchFiles funciton
	pEventInfo->BufferLength		= pEventContext->Directory.BufferLength; 

	if (pOpenInfo->DirListHead == NULL) {
		pOpenInfo->DirListHead = (PLIST_ENTRY)malloc(sizeof(LIST_ENTRY));
		InitializeListHead(pOpenInfo->DirListHead);
	}

	if (pEventContext->Directory.FileIndex == 0) {
		CPTFSUtil::ClearFindData(pOpenInfo->DirListHead);
	}

	if (IsListEmpty(pOpenInfo->DirListHead)) {

		CLogUtil::DebugLogString(DLEVEL_LOUD,_T("###FindFiles %04d\n"), pOpenInfo->EventId);

		// if user defined FindFilesWithPattern
		if (pPTFSInstance->PTFSOperations->FindFilesWithPattern) {
			LPCWSTR	pattern = L"*";
		
			// if search pattern is specified
			if (pEventContext->Directory.SearchPatternLength != 0) {
				pattern = (PWCHAR)((SIZE_T)&pEventContext->Directory.SearchPatternBase[0]
						+ (SIZE_T)pEventContext->Directory.SearchPatternOffset);
			}

			bPatternCheck = FALSE; // do not recheck pattern later in MatchFiles

			nStatus = pPTFSInstance->PTFSOperations->FindFilesWithPattern(
						pEventContext->Directory.DirectoryName,
						pattern,
						CPTFSDispatch::FillFileData,
						&fileInfo);
	
		} else if (pPTFSInstance->PTFSOperations->FindFiles) {

			bPatternCheck = TRUE; // do pattern check later in MachFiles

			// call FileSystem specifeid callback routine
			nStatus = pPTFSInstance->PTFSOperations->FindFiles(
						pEventContext->Directory.DirectoryName,
						CPTFSDispatch::FillFileData,
						CPTFSUtil::ClearFindData2,
						&fileInfo);
		} else {
			nStatus = -1;
		}
	}



	if (nStatus < 0) {

		if (pEventContext->Directory.FileIndex == 0) {
			CLogUtil::DebugLogString(DLEVEL_LOUD,_T("  STATUS_NO_SUCH_FILE\n"));
			pEventInfo->Status = STATUS_NO_SUCH_FILE;
		} else {
			CLogUtil::DebugLogString(DLEVEL_LOUD,_T("  STATUS_NO_MORE_FILES\n"));
			pEventInfo->Status = STATUS_NO_MORE_FILES;
		}

		pEventInfo->BufferLength = 0;
		pEventInfo->Directory.Index = pEventContext->Directory.FileIndex;
		// free all of list entries
		CPTFSUtil::ClearFindData(pOpenInfo->DirListHead);
	} else {
		LONG	lIndex;
		pEventInfo->Status = STATUS_SUCCESS;

		CLogUtil::DebugLogString(DLEVEL_LOUD,_T("index from %d\n"), pEventContext->Directory.FileIndex);
		// extract entries that match search pattern from FindFiles result
		lIndex = CPTFSUtil::MatchFiles(pEventContext, pEventInfo, pOpenInfo->DirListHead, bPatternCheck);

		// there is no matched file
		if (lIndex <0) {
			if (pEventContext->Directory.FileIndex == 0) {
				CLogUtil::DebugLogString(DLEVEL_LOUD,_T("STATUS_NO_SUCH_FILE\n"));
				pEventInfo->Status = STATUS_NO_SUCH_FILE;
			} else {
				CLogUtil::DebugLogString(DLEVEL_LOUD,_T("STATUS_NO_MORE_FILES\n"));
				pEventInfo->Status = STATUS_NO_MORE_FILES;
			}
			pEventInfo->BufferLength = 0;
			pEventInfo->Directory.Index = pEventContext->Directory.FileIndex;

			CPTFSUtil::ClearFindData(pOpenInfo->DirListHead);

		} else {
			CLogUtil::DebugLogString(DLEVEL_LOUD,_T("index to %d\n"), lIndex);
			pEventInfo->Directory.Index	= lIndex;
		}
		
	}

	// information for FileSystem
	pOpenInfo->UserContext = fileInfo.Context;

	// send directory information to driver
	theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, pPTFSInstance);
	free(pEventInfo);
	return;
}

VOID CPTFSDispatch::DispatchRead(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION	pEventInfo;
	PPTFS_OPEN_INFO		pOpenInfo;
	ULONG						ulReadLength = 0;
	int								nStatus = 0;
	PTFS_FILE_INFO		fileInfo;
	ULONG						ulSizeOfEventInfo;
	
	ulSizeOfEventInfo = sizeof(EVENT_INFORMATION) - 8 + pEventContext->Read.BufferLength;

	CPTFSUtil::CheckFileName(pEventContext->Read.FileName);

	pEventInfo = DispatchCommon(pEventContext, ulSizeOfEventInfo, pPTFSInstance, &fileInfo, &pOpenInfo);

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("###Read %04d\n"), pOpenInfo != NULL ? pOpenInfo->EventId : -1);

	if (pPTFSInstance->PTFSOperations->ReadFile) {
		nStatus = pPTFSInstance->PTFSOperations->ReadFile(
						pEventContext->Read.FileName,
						pEventInfo->Buffer,
						pEventContext->Read.BufferLength,
						&ulReadLength,
						pEventContext->Read.ByteOffset.QuadPart,
						&fileInfo);
	} else {
		nStatus = -1;
	}

	pOpenInfo->UserContext = fileInfo.Context;
	pEventInfo->BufferLength = 0;

	if (nStatus < 0) {
		pEventInfo->Status = STATUS_INVALID_PARAMETER;
	} else if(ulReadLength == 0) {
		pEventInfo->Status = STATUS_END_OF_FILE;
	} else {
		pEventInfo->Status = STATUS_SUCCESS;
		pEventInfo->BufferLength = ulReadLength;
		pEventInfo->Read.CurrentByteOffset.QuadPart = pEventContext->Read.ByteOffset.QuadPart + ulReadLength;
	}

	theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, pPTFSInstance);
	free(pEventInfo);
	return;
}

VOID CPTFSDispatch::DispatchWrite(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION	pEventInfo;
	PPTFS_OPEN_INFO		pOpenInfo;
	ULONG						ulWrittenLength = 0;
	int								nStatus = 0;
	PTFS_FILE_INFO		fileInfo;
	BOOL							bBufferAllocated = FALSE;
	ULONG						ulSizeOfEventInfo = sizeof(EVENT_INFORMATION);

	pEventInfo = DispatchCommon(pEventContext, ulSizeOfEventInfo, pPTFSInstance, &fileInfo, &pOpenInfo);

	// Since driver requested bigger memory,
	// allocate enough memory and send it to driver
	if (pEventContext->Write.RequestLength > 0) {
		ULONG ulContextLength = pEventContext->Write.RequestLength;
		PEVENT_CONTEXT	pContextBuf = (PEVENT_CONTEXT)malloc(ulContextLength);
		SendWriteRequest(hDevice, pEventInfo, ulSizeOfEventInfo, pContextBuf, ulContextLength);
		pEventContext = pContextBuf;
		bBufferAllocated = TRUE;
	}

	CPTFSUtil::CheckFileName(pEventContext->Write.FileName);

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("###WriteFile %04d\n"), pOpenInfo != NULL ? pOpenInfo->EventId : -1);

	if (pPTFSInstance->PTFSOperations->WriteFile) {
		nStatus = pPTFSInstance->PTFSOperations->WriteFile(
						pEventContext->Write.FileName,
						(PCHAR)pEventContext + pEventContext->Write.BufferOffset,
						pEventContext->Write.BufferLength,
						&ulWrittenLength,
						pEventContext->Write.ByteOffset.QuadPart,
						&fileInfo);
	} else {
		nStatus = -1;
	}

	pOpenInfo->UserContext = fileInfo.Context;
	pEventInfo->BufferLength = 0;

	if (nStatus < 0) {
		pEventInfo->Status = STATUS_INVALID_PARAMETER;
	
	} else {
		pEventInfo->Status = STATUS_SUCCESS;
		pEventInfo->BufferLength = ulWrittenLength;
		pEventInfo->Write.CurrentByteOffset.QuadPart = pEventContext->Write.ByteOffset.QuadPart + ulWrittenLength;
	}

	theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, pPTFSInstance);
	free(pEventInfo);

	if (bBufferAllocated)
		free(pEventContext);

	return;
}

VOID CPTFSDispatch::DispatchQueryInformation(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION		pEventInfo;
	PTFS_FILE_INFO			fileInfo;
	BY_HANDLE_FILE_INFORMATION	byHandleFileInfo;
	ULONG				ulRemainingLength;
	ULONG				ulStatus = STATUS_NOT_IMPLEMENTED;
	int						nResult;
	PPTFS_OPEN_INFO	pOpenInfo = NULL;
	ULONG					ulSizeOfEventInfo;

	ulSizeOfEventInfo = sizeof(EVENT_INFORMATION) - 8 + pEventContext->File.BufferLength;

	CPTFSUtil::CheckFileName(pEventContext->File.FileName);

	ZeroMemory(&byHandleFileInfo, sizeof(BY_HANDLE_FILE_INFORMATION));

	pEventInfo = DispatchCommon(pEventContext, ulSizeOfEventInfo, pPTFSInstance, &fileInfo, &pOpenInfo);
	
	pEventInfo->BufferLength = pEventContext->File.BufferLength;

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("###GetFileInfo %04d\n"), pOpenInfo != NULL ? pOpenInfo->EventId : -1);

	if (pPTFSInstance->PTFSOperations->GetFileInformation) {
		nResult = pPTFSInstance->PTFSOperations->GetFileInformation(
										pEventContext->File.FileName,
										&byHandleFileInfo,
										&fileInfo);
	} else {
		nResult = -1;
	}

	ulRemainingLength = pEventInfo->BufferLength;

	if (nResult < 0) {
		pEventInfo->Status = STATUS_INVALID_PARAMETER;
		pEventInfo->BufferLength = 0;
	
	} else {

		switch(pEventContext->File.FileInformationClass) {
		case FileBasicInformation:
			ulStatus = CPTFSUtil::FillFileBasicInfo((PFILE_BASIC_INFORMATION)pEventInfo->Buffer, &byHandleFileInfo, &ulRemainingLength);
			break;

		case FileInternalInformation:
			ulStatus = CPTFSUtil::FillInternalInfo((PFILE_INTERNAL_INFORMATION)pEventInfo->Buffer, &byHandleFileInfo, &ulRemainingLength);
			break;

		case FileEaInformation:
			//ulStatus = STATUS_NOT_IMPLEMENTED;
			ulStatus = STATUS_SUCCESS;
			ulRemainingLength -= sizeof(FILE_EA_INFORMATION);
			break;

		case FileStandardInformation:
			ulStatus = CPTFSUtil::FillFileStandardInfo((PFILE_STANDARD_INFORMATION)pEventInfo->Buffer, &byHandleFileInfo, &ulRemainingLength);
			break;

		case FileAllInformation:
			ulStatus = CPTFSUtil::FillFileAllInfo((PFILE_ALL_INFORMATION)pEventInfo->Buffer, &byHandleFileInfo, &ulRemainingLength, pEventContext);
			break;

		case FileAlternateNameInformation:
			ulStatus = STATUS_NOT_IMPLEMENTED;
			break;

		case FileAttributeTagInformation:
			ulStatus = CPTFSUtil::FillFileAttributeTagInfo((PFILE_ATTRIBUTE_TAG_INFORMATION)pEventInfo->Buffer,&byHandleFileInfo, &ulRemainingLength);
			break;

		case FileCompressionInformation:
			ulStatus = STATUS_NOT_IMPLEMENTED;
			break;

		case FileNameInformation:
			// this case is not used because driver deal with
			ulStatus = CPTFSUtil::FillFileNameInfo((PFILE_NAME_INFORMATION)pEventInfo->Buffer, &byHandleFileInfo, &ulRemainingLength, pEventContext);
			break;

		case FileNetworkOpenInformation:
			ulStatus = CPTFSUtil::FillNetworkOpenInfo((PFILE_NETWORK_OPEN_INFORMATION)pEventInfo->Buffer, &byHandleFileInfo, &ulRemainingLength);
			break;

		case FilePositionInformation:
			// this case is not used because driver deal with
			ulStatus = CPTFSUtil::FillFilePositionInfo((PFILE_POSITION_INFORMATION)pEventInfo->Buffer, &byHandleFileInfo, &ulRemainingLength);
			break;
		case FileStreamInformation:
			ulStatus = STATUS_NOT_IMPLEMENTED;
			break;
        default:
			{
				ulStatus = STATUS_NOT_IMPLEMENTED;
				CLogUtil::DebugLogString(DLEVEL_ERROR, _T("unknown type:%d\n"), pEventContext->File.FileInformationClass);
			}
            break;
		}
	
		pEventInfo->Status = ulStatus;
		pEventInfo->BufferLength = pEventContext->File.BufferLength - ulRemainingLength;
	}

	// information for FileSystem
	if(pOpenInfo)
		pOpenInfo->UserContext = fileInfo.Context;

	theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, pPTFSInstance);
	free(pEventInfo);
	return;
}

VOID CPTFSDispatch::DispatchQueryVolumeInformation(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION	pEventInfo;
	PTFS_FILE_INFO		fileInfo;
	PPTFS_OPEN_INFO		pOpenInfo;
	ULONG						ulSizeOfEventInfo = sizeof(EVENT_INFORMATION) - 8 + pEventContext->Volume.BufferLength;

	pEventInfo = (PEVENT_INFORMATION)malloc(ulSizeOfEventInfo);

	RtlZeroMemory(pEventInfo, ulSizeOfEventInfo);
	RtlZeroMemory(&fileInfo, sizeof(PTFS_FILE_INFO));

	// There is no Context because file is not opened
	// so DispatchCommon is not used here
	pOpenInfo = (PPTFS_OPEN_INFO)pEventContext->Context;
	
	pEventInfo->BufferLength = 0;
	pEventInfo->SerialNumber = pEventContext->SerialNumber;

	fileInfo.ProcessId = pEventContext->ProcessId;
	fileInfo.PTFSOptions = pPTFSInstance->PTFSOptions;

	pEventInfo->Status = STATUS_NOT_IMPLEMENTED;
	pEventInfo->BufferLength = 0;

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("###QueryVolumeInfo %04d\n"), pOpenInfo ? pOpenInfo->EventId : -1);

	switch (pEventContext->Volume.FsInformationClass) {
	case FileFsVolumeInformation:
		pEventInfo->Status = CPTFSUtil::FsVolumeInformation(pEventInfo, pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
	case FileFsSizeInformation:
		pEventInfo->Status = CPTFSUtil::FsSizeInformation(pEventInfo, pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
	case FileFsAttributeInformation:
		pEventInfo->Status = CPTFSUtil::FsAttributeInformation(pEventInfo, pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
	case FileFsFullSizeInformation:
		pEventInfo->Status = CPTFSUtil::FsFullSizeInformation(pEventInfo, pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
	default:
		CLogUtil::DebugLogString(DLEVEL_ERROR, _T("error unknown volume info %d\n"), pEventContext->Volume.FsInformationClass);
	}

	theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, NULL);
	free(pEventInfo);
	return;
}

VOID CPTFSDispatch::DispatchLock(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PTFS_FILE_INFO		fileInfo;
	PEVENT_INFORMATION	pEventInfo;
	ULONG						ulSizeOfEventInfo = sizeof(EVENT_INFORMATION);
	PPTFS_OPEN_INFO		pOpenInfo;
	int nStatus = 0;

	CPTFSUtil::CheckFileName(pEventContext->Lock.FileName);

	pEventInfo = DispatchCommon(pEventContext, ulSizeOfEventInfo, pPTFSInstance, &fileInfo, &pOpenInfo);

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("###Lock %04d\n"), pOpenInfo != NULL ? pOpenInfo->EventId : -1);

	pEventInfo->Status = STATUS_NOT_IMPLEMENTED;

	switch (pEventContext->MinorFunction) {
	case IRP_MN_LOCK:
		if (pPTFSInstance->PTFSOperations->LockFile) {

			nStatus = pPTFSInstance->PTFSOperations->LockFile(
						pEventContext->Lock.FileName,
						pEventContext->Lock.ByteOffset.QuadPart,
						pEventContext->Lock.Length.QuadPart,
						//pEventContext->Lock.Key,
						&fileInfo);

			pEventInfo->Status = nStatus < 0 ? STATUS_LOCK_NOT_GRANTED : STATUS_SUCCESS;
		}
		break;
	case IRP_MN_UNLOCK_ALL:
		break;
	case IRP_MN_UNLOCK_ALL_BY_KEY:
		break;
	case IRP_MN_UNLOCK_SINGLE:
		if (pPTFSInstance->PTFSOperations->UnlockFile) {
		
			nStatus = pPTFSInstance->PTFSOperations->UnlockFile(
						pEventContext->Lock.FileName,
						pEventContext->Lock.ByteOffset.QuadPart,
						pEventContext->Lock.Length.QuadPart,
						//pEventContext->Lock.Key,
						&fileInfo);

			pEventInfo->Status = STATUS_SUCCESS; // at any time return success ?
		}
		break;
	default:
		CLogUtil::DebugLogString(DLEVEL_ERROR, _T("unkown lock function %d\n"), pEventContext->MinorFunction);
	}

	pOpenInfo->UserContext = fileInfo.Context;

	theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, pPTFSInstance);

	free(pEventInfo);
	return;
}

VOID CPTFSDispatch::DispatchSetInformation(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION	pEventInfo;
	PPTFS_OPEN_INFO		pOpenInfo;
	PTFS_FILE_INFO		fileInfo;
	int								nStatus = 0;
	ULONG						ulSizeOfEventInfo = sizeof(EVENT_INFORMATION);

	if (pEventContext->SetFile.FileInformationClass == FileRenameInformation || pEventContext->SetFile.FileInformationClass == FileRenameInformationEx) {
		PPTFS_RENAME_INFORMATION renameInfo = (PPTFS_RENAME_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);
		ulSizeOfEventInfo += renameInfo->FileNameLength;
	}

	CPTFSUtil::CheckFileName(pEventContext->SetFile.FileName);

	pEventInfo = DispatchCommon(pEventContext, ulSizeOfEventInfo, pPTFSInstance, &fileInfo, &pOpenInfo);
	
	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("###SetFileInfo %04d\n"), pOpenInfo != NULL ? pOpenInfo->EventId : -1);

	switch (pEventContext->SetFile.FileInformationClass) {
	case FileAllocationInformation:
		nStatus = CPTFSUtil::SetAllocationInformation(pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
	
	case FileBasicInformation:
		nStatus = CPTFSUtil::SetBasicInformation(pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
		
	case FileDispositionInformation:
	case FileDispositionInformationEx:
		nStatus = CPTFSUtil::SetDispositionInformation(pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
		
	case FileEndOfFileInformation:
		nStatus = CPTFSUtil::SetEndOfFileInformation(pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
		
	case FileLinkInformation:
		nStatus = CPTFSUtil::SetLinkInformation(pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
	
	case FilePositionInformation:
		// this case is dealed with by driver
		nStatus = -1;
		break;
		
	case FileRenameInformation:
	case FileRenameInformationEx:
		nStatus = CPTFSUtil::SetRenameInformation(pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;

	case FileValidDataLengthInformation:
		nStatus = CPTFSUtil::SetValidDataLengthInformation(pEventContext, &fileInfo, pPTFSInstance->PTFSOperations);
		break;
	}

	pOpenInfo->UserContext = fileInfo.Context;

	pEventInfo->BufferLength = 0;

	if (pEventContext->SetFile.FileInformationClass == FileDispositionInformation ||
		pEventContext->SetFile.FileInformationClass == FileDispositionInformationEx) {
		if (nStatus == 0) {
			PFILE_DISPOSITION_INFORMATION dispositionInfo = (PFILE_DISPOSITION_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);
			pEventInfo->Delete.DeleteOnClose = dispositionInfo->DeleteFile ? TRUE : FALSE;
			CLogUtil::DebugLogString(DLEVEL_LOUD, _T("dispositionInfo->DeleteFile = %d\n"), dispositionInfo->DeleteFile);
			pEventInfo->Status = STATUS_SUCCESS;
		} else if (nStatus == -ERROR_DIR_NOT_EMPTY) {
			CLogUtil::DebugLogString(DLEVEL_LOUD, _T("DispositionInfo status = STATUS_DIRECTORY_NOT_EMPTY\n"));
			pEventInfo->Status = STATUS_DIRECTORY_NOT_EMPTY;
		} else if (nStatus < 0) {
			CLogUtil::DebugLogString(DLEVEL_LOUD,_T("DispositionInfo status = STATUS_CANNOT_DELETE\n"));
			pEventInfo->Status = STATUS_CANNOT_DELETE;
		}

	} else {
		if (nStatus < 0) {
			int nError = nStatus * -1;
			pEventInfo->Status = CPTFSUtil::GetNTStatus(nError);
		
		} else {
			pEventInfo->Status = STATUS_SUCCESS;
			// notice new file name to driver
			if (pEventContext->SetFile.FileInformationClass == FileRenameInformation ||
				pEventContext->SetFile.FileInformationClass == FileRenameInformationEx) {
				PPTFS_RENAME_INFORMATION renameInfo = (PPTFS_RENAME_INFORMATION)((PCHAR)pEventContext + pEventContext->SetFile.BufferOffset);
				pEventInfo->BufferLength = renameInfo->FileNameLength;
				CopyMemory(pEventInfo->Buffer, renameInfo->FileName, renameInfo->FileNameLength);
			}
		}
	}

	theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, pPTFSInstance);
	free(pEventInfo);
	return;
}

VOID CPTFSDispatch::DispatchFlush(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PTFS_FILE_INFO		fileInfo;
	PEVENT_INFORMATION	pEventInfo;
	ULONG						ulSizeOfEventInfo = sizeof(EVENT_INFORMATION);
	PPTFS_OPEN_INFO		pOpenInfo;
	int								nStatus;

	CPTFSUtil::CheckFileName(pEventContext->Flush.FileName);

	pEventInfo = DispatchCommon(pEventContext, ulSizeOfEventInfo, pPTFSInstance, &fileInfo, &pOpenInfo);

	CLogUtil::DebugLogString(DLEVEL_LOUD, _T("###Flush %04d\n"), pOpenInfo != NULL ? pOpenInfo->EventId : -1);

	pEventInfo->Status = STATUS_SUCCESS;

	if (pPTFSInstance->PTFSOperations->FlushFileBuffers) {

		nStatus = pPTFSInstance->PTFSOperations->FlushFileBuffers(pEventContext->Flush.FileName, &fileInfo);
		pEventInfo->Status = nStatus < 0 ? STATUS_NOT_SUPPORTED : STATUS_SUCCESS;
	}

	pOpenInfo->UserContext = fileInfo.Context;
	theApp.SendEventInformation(hDevice, pEventInfo, ulSizeOfEventInfo, pPTFSInstance);

	free(pEventInfo);
	return;
}

VOID CPTFSDispatch::DispatchQuerySecurity(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION	pEventInfo;
	PTFS_FILE_INFO		fileInfo;
	PPTFS_OPEN_INFO		pOpenInfo;
	ULONG						ulEventInfoLength;
	int			nStatus = -ERROR_CALL_NOT_IMPLEMENTED;
	ULONG	ulLengthNeeded = 0;

	ulEventInfoLength = sizeof(EVENT_INFORMATION) - 8 + pEventContext->Security.BufferLength;
	CPTFSUtil::CheckFileName(pEventContext->Security.FileName);

	pEventInfo = DispatchCommon(pEventContext, ulEventInfoLength, pPTFSInstance, &fileInfo, &pOpenInfo);
	
	if (PTFS_SECURITY_SUPPORTED_VERSION <= pPTFSInstance->PTFSOptions->Version &&
		pPTFSInstance->PTFSOperations->GetFileSecurity) {
		nStatus = pPTFSInstance->PTFSOperations->GetFileSecurity(
					pEventContext->Security.FileName,
					&pEventContext->Security.SecurityInformation,
					&pEventInfo->Buffer,
					pEventContext->Security.BufferLength,
					&ulLengthNeeded,
					&fileInfo);
	}

	if (nStatus < 0) {
		int error = nStatus * -1;
		if (error == ERROR_INSUFFICIENT_BUFFER && ulLengthNeeded > 0) {
			pEventInfo->Status = STATUS_BUFFER_OVERFLOW;
			pEventInfo->BufferLength = ulLengthNeeded;
		} else if (error == ERROR_ACCESS_DENIED) {
			pEventInfo->Status = STATUS_ACCESS_DENIED;
			pEventInfo->BufferLength = 0;
		} else if (error == ERROR_CALL_NOT_IMPLEMENTED) {
			pEventInfo->Status = STATUS_NOT_IMPLEMENTED;
			pEventInfo->BufferLength = 0;
		} else {
			pEventInfo->Status = STATUS_INVALID_PARAMETER;
			pEventInfo->BufferLength = 0;
		}
	} else {
		if (pEventContext->Security.BufferLength < ulLengthNeeded) {
			// Filesystem Application should return -ERROR_BUFFER_OVERFLOW in this case.
			pEventInfo->BufferLength = ulLengthNeeded;
			pEventInfo->Status = STATUS_BUFFER_OVERFLOW;
		} else {
			pEventInfo->BufferLength = ulLengthNeeded;
			pEventInfo->Status = STATUS_SUCCESS;
		}
	}

	theApp.SendEventInformation(hDevice, pEventInfo, ulEventInfoLength, pPTFSInstance);
	free(pEventInfo);
}

VOID CPTFSDispatch::DispatchSetSecurity(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PEVENT_INFORMATION	pEventInfo;
	PTFS_FILE_INFO		fileInfo;
	PPTFS_OPEN_INFO		pOpenInfo;
	ULONG	ulEventInfoLength;
	int			nStatus = -1;
	PSECURITY_DESCRIPTOR	pSecurityDescriptor;
	
	ulEventInfoLength = sizeof(EVENT_INFORMATION);
	CPTFSUtil::CheckFileName(pEventContext->SetSecurity.FileName);

	pEventInfo = DispatchCommon(pEventContext, ulEventInfoLength, pPTFSInstance, &fileInfo, &pOpenInfo);
	
	pSecurityDescriptor = (PCHAR)pEventContext + pEventContext->SetSecurity.BufferOffset;

	if (PTFS_SECURITY_SUPPORTED_VERSION <= pPTFSInstance->PTFSOptions->Version &&
		pPTFSInstance->PTFSOperations->SetFileSecurity) {
		nStatus = pPTFSInstance->PTFSOperations->SetFileSecurity(
					pEventContext->SetSecurity.FileName,
					&pEventContext->SetSecurity.SecurityInformation,
					pSecurityDescriptor,
					pEventContext->SetSecurity.BufferLength,
					&fileInfo);
	}

	if (nStatus < 0) {
		pEventInfo->Status = STATUS_INVALID_PARAMETER;
		pEventInfo->BufferLength = 0;
	} else {
		pEventInfo->Status = STATUS_SUCCESS;
		pEventInfo->BufferLength = 0;
	}

	theApp.SendEventInformation(hDevice, pEventInfo, ulEventInfoLength, pPTFSInstance);
	free(pEventInfo);
}

VOID CPTFSDispatch::DispatchUnmount(HANDLE hDevice, PEVENT_CONTEXT pEventContext, PPTFS_INSTANCE pPTFSInstance)
{
	PTFS_FILE_INFO fileInfo;
	static int	nCount = 0;

	// Unmount is called only once
	EnterCriticalSection(&pPTFSInstance->CriticalSection); 
	
	if (nCount > 0) {
		LeaveCriticalSection(&pPTFSInstance->CriticalSection);
		return;
	}
	nCount++;

	RtlZeroMemory(&fileInfo, sizeof(PTFS_FILE_INFO));

	fileInfo.ProcessId = pEventContext->ProcessId;

	if (pPTFSInstance->PTFSOperations->Unmount) {
		// ignore return value
		pPTFSInstance->PTFSOperations->Unmount(&fileInfo);
	}

	LeaveCriticalSection(&pPTFSInstance->CriticalSection);

	// do not notice enything to the driver
	return;
}