#include "StdAfx.h"
#include "FSManager.h"

#include <sddl.h>
#include <winioctl.h>
#include <Dbt.h>
#include "PTFSLib.h"
#include "List/list.h"
#include "util/util.h"

CFSManager::CFSManager(void)
{
	m_hDevice = INVALID_HANDLE_VALUE;

	InitializeCriticalSectionAndSpinCount(&m_CriticalSection, 0x80000400);
			
	InitializeListHead(&m_MountList);
}

CFSManager::~CFSManager(void)
{
	DeleteCriticalSection(&m_CriticalSection);
}

BOOL CFSManager::CreateMountPoint(LPCWSTR lpcwszMountPoint, LPCWSTR lpcwszDeviceName)
{
	HANDLE	hMountPoint;
	PREPARSE_DATA_BUFFER pReparseData;
	USHORT	bufferLength;
	USHORT	targetLength;
	BOOL		bResult;
	ULONG	ulResultLength;
	WCHAR	wszTargetDeviceName[MAX_PATH] =  L"\\??";

	wcscat_s(wszTargetDeviceName, MAX_PATH, lpcwszDeviceName);
	wcscat_s(wszTargetDeviceName, MAX_PATH, L"\\");

	hMountPoint = CreateFile(
		lpcwszMountPoint, GENERIC_WRITE, 0, NULL, OPEN_EXISTING,
		FILE_FLAG_OPEN_REPARSE_POINT | FILE_FLAG_BACKUP_SEMANTICS, NULL);

	if (hMountPoint == INVALID_HANDLE_VALUE) {
		return FALSE;
	}

	targetLength = (USHORT)wcslen(wszTargetDeviceName) * sizeof(WCHAR);
	bufferLength = FIELD_OFFSET(REPARSE_DATA_BUFFER, MountPointReparseBuffer.PathBuffer) + targetLength + sizeof(WCHAR) + sizeof(WCHAR);

	pReparseData = (PREPARSE_DATA_BUFFER)malloc(bufferLength);

	ZeroMemory(pReparseData, bufferLength);

	pReparseData->ReparseTag = IO_REPARSE_TAG_MOUNT_POINT;
	pReparseData->ReparseDataLength = bufferLength - REPARSE_DATA_BUFFER_HEADER_SIZE;

	pReparseData->MountPointReparseBuffer.SubstituteNameOffset = 0;
	pReparseData->MountPointReparseBuffer.SubstituteNameLength = targetLength;
	pReparseData->MountPointReparseBuffer.PrintNameOffset = targetLength + sizeof(WCHAR);
	pReparseData->MountPointReparseBuffer.PrintNameLength = 0;

	RtlCopyMemory(pReparseData->MountPointReparseBuffer.PathBuffer, wszTargetDeviceName, targetLength);

	bResult = DeviceIoControl(
				hMountPoint,
				FSCTL_SET_REPARSE_POINT,
				pReparseData,
				bufferLength,
				NULL,
				0,
				&ulResultLength,
				NULL);
	
	CloseHandle(hMountPoint);
	free(pReparseData);

	return bResult;
}

BOOL CFSManager::DeleteMountPoint(LPCWSTR lpcwszMountPoint)
{
	HANDLE	hMountPoint;
	BOOL		bResult;
	ULONG	ulResultLength;
	REPARSE_GUID_DATA_BUFFER	reparseData = { 0 };

	hMountPoint = CreateFile(lpcwszMountPoint, GENERIC_WRITE, 0, NULL, OPEN_EXISTING,
		FILE_FLAG_OPEN_REPARSE_POINT | FILE_FLAG_BACKUP_SEMANTICS, NULL);

	if (hMountPoint == INVALID_HANDLE_VALUE) {
		return FALSE;
	}

	reparseData.ReparseTag = IO_REPARSE_TAG_MOUNT_POINT;

	bResult = DeviceIoControl(
				hMountPoint,
				FSCTL_DELETE_REPARSE_POINT,
				&reparseData,
				REPARSE_GUID_DATA_BUFFER_HEADER_SIZE,
				NULL,
				0,
				&ulResultLength,
				NULL);
	
	CloseHandle(hMountPoint);

	return bResult;
}

BOOL CFSManager::CreateDriveLetter(WCHAR wDriveLetter, LPCWSTR lpwszDeviceName)
{
	WCHAR   wszDosDevice[] = L"\\\\.\\C:";
	WCHAR   wszDriveName[] = L"C:";
	WCHAR	wszRawDeviceName[MAX_PATH] = L"\\Device";

	wszDosDevice[4] = wDriveLetter;
	wszDriveName[0] = wDriveLetter;
	wcscat_s(wszRawDeviceName, MAX_PATH, lpwszDeviceName);

    if (!DefineDosDevice(DDD_RAW_TARGET_PATH, wszDriveName, wszRawDeviceName)) 
	{
        return FALSE;
    }
	return TRUE;
}

BOOL CFSManager::PantaFSControlMount(LPCWSTR lpcwszMountPoint, LPCWSTR lpcwszDeviceName)
{
	ULONG ulLength = (ULONG)wcslen(lpcwszMountPoint);

	if (ulLength == 1 ||
		(ulLength == 2 && lpcwszMountPoint[1] == L':') ||
		(ulLength == 3 && lpcwszMountPoint[1] == L':' && lpcwszMountPoint[2] == L'\\'))
	{
		return CreateDriveLetter(lpcwszMountPoint[0], lpcwszDeviceName);
	} 
	else if (ulLength > 3) 
	{
		return CreateMountPoint(lpcwszMountPoint, lpcwszDeviceName);
	}
	return FALSE; 
}

BOOL CFSManager::PantaFSControlUnmount(LPCWSTR lpcwszMountPoint)
{
	ULONG ulLength = (ULONG)wcslen(lpcwszMountPoint);

	if (ulLength == 1 ||
		(ulLength == 2 && lpcwszMountPoint[1] == L':') ||
		(ulLength == 3 && lpcwszMountPoint[1] == L':' && lpcwszMountPoint[2] == L'\\'))
	{
		WCHAR   wszDrive[] = L"C:";	
	    wszDrive[0] = lpcwszMountPoint[0];

		if (!DefineDosDevice(DDD_REMOVE_DEFINITION, wszDrive, NULL))
		{
			return FALSE;
		}
		else 
		{
			return TRUE;
		}

	} else if (ulLength > 3 ) {
		return DeleteMountPoint(lpcwszMountPoint);
	}

	return FALSE;
}

PMOUNT_ENTRY CFSManager::InsertMountEntry(PPTFS_PIPE_CONTROL pPantaFSControl)
{
	PMOUNT_ENTRY	pMountEntry;
	pMountEntry = (PMOUNT_ENTRY)malloc(sizeof(MOUNT_ENTRY));
	if (pMountEntry == NULL)
	{
		return NULL;
	}
	ZeroMemory(pMountEntry, sizeof(MOUNT_ENTRY));
	CopyMemory(&pMountEntry->MountControl, pPantaFSControl, sizeof(PTFS_PIPE_CONTROL));
	InitializeListHead(&pMountEntry->ListEntry);

	EnterCriticalSection(&m_CriticalSection);
	InsertTailList(&m_MountList, &pMountEntry->ListEntry);
	LeaveCriticalSection(&m_CriticalSection);

	return pMountEntry;
}

VOID CFSManager::RemoveMountEntry(PMOUNT_ENTRY pMountEntry)
{
	EnterCriticalSection(&m_CriticalSection);
	RemoveEntryList(&pMountEntry->ListEntry);
	LeaveCriticalSection(&m_CriticalSection);

	free(pMountEntry);
}

PMOUNT_ENTRY CFSManager::FindMountEntry(PPTFS_PIPE_CONTROL pPantaFSControl)
{
	PLIST_ENTRY		pListEntry;
	PMOUNT_ENTRY	pMountEntry;
	BOOL			bUseMountPoint = wcslen(pPantaFSControl->MountPoint) > 0;
	BOOL			bFound = FALSE;

	if (!bUseMountPoint && wcslen(pPantaFSControl->DeviceName) == 0) 
	{
		return NULL;
	}

	EnterCriticalSection(&m_CriticalSection);

    for (pListEntry = m_MountList.Flink; pListEntry != &m_MountList; pListEntry = pListEntry->Flink) {
		pMountEntry = CONTAINING_RECORD(pListEntry, MOUNT_ENTRY, ListEntry);
		if (bUseMountPoint) {
			if (wcscmp(pPantaFSControl->MountPoint, pMountEntry->MountControl.MountPoint) == 0) {
				bFound = TRUE;
				break;
			}
		} else {
			if (wcscmp(pPantaFSControl->DeviceName, pMountEntry->MountControl.DeviceName) == 0) {
				bFound = TRUE;
				break;
			}
		}
	}

	LeaveCriticalSection(&m_CriticalSection);

	if (bFound) {
		return pMountEntry;
	} else {
		return NULL;
	}
}

BOOL CFSManager::PantaFSControlFind(PPTFS_PIPE_CONTROL pControl)
{
	PMOUNT_ENTRY	pMountEntry;

	pMountEntry = FindMountEntry(pControl);

	if (pMountEntry == NULL) 
		return FALSE;

	wcscpy_s(pControl->DeviceName, sizeof(pControl->DeviceName) / sizeof(WCHAR), pMountEntry->MountControl.DeviceName);
	wcscpy_s(pControl->MountPoint, sizeof(pControl->MountPoint) / sizeof(WCHAR), pMountEntry->MountControl.MountPoint);
	return TRUE;
}

VOID CFSManager::PantaFSControlList(PPTFS_PIPE_CONTROL pControl)
{
	PLIST_ENTRY			pListEntry;
	PMOUNT_ENTRY	pMountEntry;
	ULONG				ulIndex = 0;

	EnterCriticalSection(&m_CriticalSection);
	pControl->Status = PTFS_CONTROL_FAIL;

	for (pListEntry = m_MountList.Flink; pListEntry != &m_MountList; pListEntry = pListEntry->Flink) 
	{
		pMountEntry = CONTAINING_RECORD(pListEntry, MOUNT_ENTRY, ListEntry);
		if (pControl->Option == ulIndex++) {
			wcscpy_s(pControl->DeviceName, sizeof(pControl->DeviceName) / sizeof(WCHAR),
					pMountEntry->MountControl.DeviceName);
			wcscpy_s(pControl->MountPoint, sizeof(pControl->MountPoint) / sizeof(WCHAR),
					pMountEntry->MountControl.MountPoint);
			pControl->Status = PTFS_CONTROL_SUCCESS;
			break;
		}
	}
	LeaveCriticalSection(&m_CriticalSection);
}

void CFSManager::PantaFSControl(PPTFS_PIPE_CONTROL pControl)
{
	PMOUNT_ENTRY	pMountEntry;

	pControl->Status = PTFS_CONTROL_FAIL;

	switch (pControl->Type)
	{
	case PTFS_PIPE_CONTROL_MOUNT:
		if (PantaFSControlMount(pControl->MountPoint, pControl->DeviceName)) {
			pControl->Status = PTFS_CONTROL_SUCCESS;
			InsertMountEntry(pControl);
		} else {
			pControl->Status = PTFS_CONTROL_FAIL;
		}

		break;

	case PTFS_PIPE_CONTROL_UNMOUNT:

		pMountEntry = FindMountEntry(pControl);
		if (pMountEntry == NULL) {
			if (pControl->Option == PTFS_CONTROL_OPTION_FORCE_UNMOUNT &&
				PantaFSControlUnmount(pControl->MountPoint)){
				pControl->Status = PTFS_CONTROL_SUCCESS;
				break;
			}
			pControl->Status = PTFS_CONTROL_FAIL;
			break;	
		}

		if (PantaFSControlUnmount(pMountEntry->MountControl.MountPoint)) {
			pControl->Status = PTFS_CONTROL_SUCCESS;
			if (wcslen(pControl->DeviceName) == 0) {
				wcscpy_s(pControl->DeviceName, sizeof(pControl->DeviceName) / sizeof(WCHAR), pMountEntry->MountControl.DeviceName);
			}
			RemoveMountEntry(pMountEntry);
		} else {
			pMountEntry->MountControl.Status = PTFS_CONTROL_FAIL;
			pControl->Status = PTFS_CONTROL_FAIL;
		}

		break;

	case PTFS_PIPE_CONTROL_CHECK:
		{
			pControl->Status = 0;
		}
		break;

	case PTFS_PIPE_CONTROL_FIND:
		{
			PantaFSControlFind(pControl);
		}
		break;

	case PTFS_PIPE_CONTROL_LIST:
		{
			PantaFSControlList(pControl);
		}
		break;

	case PTFS_PIPE_CONTROL_RELOGIN:
		{
			pControl->Status = PTFS_CONTROL_SUCCESS;
		}
		break;

	default:
		break;
	}

	return;
}

BOOL CFSManager::Initialize()
{
	return TRUE;
}

void CFSManager::RemoveAllMountList()
{
	PMOUNT_ENTRY	pMountEntry = NULL;

	EnterCriticalSection(&m_CriticalSection);

	while(!IsListEmpty(&m_MountList))
	{
		PLIST_ENTRY pListEntry = RemoveHeadList(&m_MountList);
		pMountEntry = CONTAINING_RECORD(pListEntry, MOUNT_ENTRY, ListEntry);
		
		if(pMountEntry)
		{
			PantaFSControlUnmount(pMountEntry->MountControl.MountPoint);
			free(pMountEntry);
			pMountEntry = NULL;
		}
	}
	
	LeaveCriticalSection(&m_CriticalSection);
}

BOOL CFSManager::IsMountList()
{
	BOOL bMountList = FALSE;

	EnterCriticalSection(&m_CriticalSection);
	bMountList = IsListEmpty(&m_MountList);
	LeaveCriticalSection(&m_CriticalSection);

	return !bMountList;
}

BOOL CFSManager::IsPantaFSMountDrive(LPCWSTR lpcwszMountDrive)
{	
	PTFS_PIPE_CONTROL Control;

	ZeroMemory(&Control, sizeof(PTFS_PIPE_CONTROL));

	wcscpy_s(Control.MountPoint, lpcwszMountDrive);

	PantaFSControlFind(&Control);

	if(PTFS_CONTROL_SUCCESS == Control.Status)
		return TRUE;

	return FALSE;
}