#include "write.h"
#include "init.h"
#include "util.h"
#include "notification.h"
#include "event.h"


#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSDispatchWrite)
#endif


NTSTATUS
PTFSDispatchWrite(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
    NTSTATUS status = STATUS_INVALID_PARAMETER;
    PIO_STACK_LOCATION pIrpSp = NULL;
    PFILE_OBJECT pFileObject = NULL;
    PEVENT_CONTEXT pEventContext = NULL;
    ULONG eventLength = 0;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;
    PPTFSVCB pVcb = NULL;
    PVOID pBuffer = NULL;
    BOOLEAN writeToEoF = FALSE;
    BOOLEAN isPagingIo = FALSE;
    BOOLEAN isNonCached = FALSE;
    BOOLEAN isSynchronousIo = FALSE;
    BOOLEAN fcbLocked = FALSE;
    LARGE_INTEGER safeEventLength;

    __try 
    {

        KdPrint(("[PTFS]::PTFSDispatchWrite Start PID[%lu]\n", IoGetRequestorProcessId(pIrp)));

        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
        pFileObject = pIrpSp->FileObject;

        if (pIrpSp->Parameters.Write.Length == 0) 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite Parameters.Write.Length == 0\n"));
            pIrp->IoStatus.Information = 0;
            status = STATUS_SUCCESS;
            __leave;
        }

        if (pFileObject == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite FileObjet is null\n"));
            status = STATUS_INVALID_DEVICE_REQUEST;
            __leave;
        }

        pVcb = pDeviceObject->DeviceExtension;
        if (GETIDENTIFIERTYPE(pVcb) != VCB || !PTFSCheckCCB(pVcb->pDcb, pFileObject->FsContext2))
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite invalid identifier type\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        pCcb = pFileObject->FsContext2;
        ASSERT(pCcb != NULL);

        pFcb = pCcb->pFcb;
        ASSERT(pFcb != NULL);

        if (PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY)) 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite directory\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        if (pIrp->MdlAddress)
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite use MdlAddress\n"));
            pBuffer = MmGetSystemAddressForMdlNormalSafe(pIrp->MdlAddress);
        }
        else 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite use UserBuffer\n"));
            pBuffer = pIrp->UserBuffer;
        }

        if (pBuffer == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite buffer is null\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        if (pIrpSp->Parameters.Write.ByteOffset.LowPart == FILE_WRITE_TO_END_OF_FILE &&
            pIrpSp->Parameters.Write.ByteOffset.HighPart == -1) 
        {
            writeToEoF = TRUE;
        }

        isPagingIo = (pIrp->Flags & IRP_PAGING_IO);

        if (pIrp->Flags & IRP_NOCACHE)
            isNonCached = TRUE;

        if (pFileObject->Flags & FO_SYNCHRONOUS_IO)
            isSynchronousIo = TRUE;

        if (!isPagingIo && (pFileObject->SectionObjectPointer != NULL) &&
            (pFileObject->SectionObjectPointer->DataSectionObject != NULL)) 
        {
            PTFSPagingIoLock(pFcb, FALSE);
            CcFlushCache(&pFcb->SectionObjectPointers, writeToEoF ? NULL : &pIrpSp->Parameters.Write.ByteOffset, pIrpSp->Parameters.Write.Length, NULL);
            CcPurgeCacheSection(&pFcb->SectionObjectPointers, writeToEoF ? NULL : &pIrpSp->Parameters.Write.ByteOffset, pIrpSp->Parameters.Write.Length, FALSE);
            PTFSPagingIoUnlock(pFcb);
        }

        if (writeToEoF && isPagingIo) 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite writeToEoF & isPagingIo\n"));
            pIrp->IoStatus.Information = 0;
            status = STATUS_SUCCESS;
            __leave;
        }

        PTFSFcbLock(pFcb, TRUE);
        fcbLocked = TRUE;

        safeEventLength.QuadPart = (LONGLONG)(sizeof(EVENT_CONTEXT) + pIrpSp->Parameters.Write.Length + pFcb->unstrFileName.Length);
        if (safeEventLength.HighPart != 0 || safeEventLength.QuadPart < (LONGLONG)(sizeof(EVENT_CONTEXT) + pFcb->unstrFileName.Length)) 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite Write with unsupported total size[%I64u]\n", safeEventLength.QuadPart));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }
        eventLength = safeEventLength.LowPart;
        pEventContext = AllocateEventContext(pVcb->pDcb, pIrp, eventLength, pCcb);
        if (pEventContext == NULL)
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite failed to AllocateEventContext\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        pEventContext->Context = pCcb->UserContext;
        pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_EVENT] = pEventContext;

        if (isPagingIo) 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite Paging IO\n"));
            pEventContext->FileFlags |= PTFS_PAGING_IO;
        }

        if (isSynchronousIo)
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite Synchronous IO\n"));
            pEventContext->FileFlags |= PTFS_SYNCHRONOUS_IO;
        }

        if (isNonCached) 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite Nocache\n"));
            pEventContext->FileFlags |= PTFS_NOCACHE;
        }

        pEventContext->Operation.Write.ByteOffset = pIrpSp->Parameters.Write.ByteOffset;

        if (writeToEoF) 
        {
            pEventContext->FileFlags |= PTFS_WRITE_TO_END_OF_FILE;
            KdPrint(("[PTFS]::PTFSDispatchWrite WriteOffset = end of file\n"));
        }

        if ( isSynchronousIo && ((pIrpSp->Parameters.Write.ByteOffset.LowPart == FILE_USE_FILE_POINTER_POSITION) &&
                (pIrpSp->Parameters.Write.ByteOffset.HighPart == -1))) 
        {
            pEventContext->Operation.Write.ByteOffset.QuadPart = pFileObject->CurrentByteOffset.QuadPart;
        }

        pEventContext->Operation.Write.BufferLength = pIrpSp->Parameters.Write.Length;
        pEventContext->Operation.Write.BufferOffset = FIELD_OFFSET(EVENT_CONTEXT, Operation.Write.FileName[0]) + pFcb->unstrFileName.Length + sizeof(WCHAR);
        RtlCopyMemory((PCHAR)pEventContext + pEventContext->Operation.Write.BufferOffset, pBuffer, pIrpSp->Parameters.Write.Length);
        pEventContext->Operation.Write.FileNameLength = pFcb->unstrFileName.Length;
        RtlCopyMemory(pEventContext->Operation.Write.FileName, pFcb->unstrFileName.Buffer, pFcb->unstrFileName.Length);

        if (eventLength <= EVENT_CONTEXT_MAX_SIZE) 
        {
            KdPrint(("[PTFS]::PTFSDispatchWrite Offset[%d:%d], Length[%d]\n", 
                pIrpSp->Parameters.Write.ByteOffset.HighPart, pIrpSp->Parameters.Write.ByteOffset.LowPart, pIrpSp->Parameters.Write.Length));

            pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_EVENT] = 0;

            if (!FlagOn(pIrp->Flags, IRP_PAGING_IO)) 
            {
                status = PTFSCheckOplock(pFcb, pIrp, pEventContext, PTFSOplockComplete, PTFSPrePostIrp);

                if (status != STATUS_SUCCESS) 
                {
                    if (status == STATUS_PENDING) 
                        KdPrint(("[PTFS]::PTFSDispatchWrite FsRtlCheckOplock returned STATUS_PENDING\n"));
                    else 
                        FreeEventContext(pEventContext);

                    __leave;
                }
            }

            status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, 0);
        }
        else 
        {
            ULONG requestContextLength = max(sizeof(EVENT_CONTEXT), pEventContext->Operation.Write.BufferOffset);
            PEVENT_CONTEXT pRequestContext = AllocateEventContext(pVcb->pDcb, pIrp, requestContextLength, pCcb);

            if (pRequestContext == NULL) 
            {
                KdPrint(("[PTFS]::PTFSDispatchWrite failed to AllocateEventContext(request)\n"));
                status = STATUS_INSUFFICIENT_RESOURCES;
                pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_EVENT] = 0;
                FreeEventContext(pEventContext);
                __leave;
            }

            KdPrint(("[PTFS]::PTFSDispatchWrite Offset[%d:%d], Length[%d](request)\n",
                pIrpSp->Parameters.Write.ByteOffset.HighPart, pIrpSp->Parameters.Write.ByteOffset.LowPart, pIrpSp->Parameters.Write.Length));

            RtlCopyMemory(pRequestContext, pEventContext, pEventContext->Operation.Write.BufferOffset);
            pRequestContext->Length = requestContextLength;
            pRequestContext->Operation.Write.RequestLength = eventLength;

            if (!FlagOn(pIrp->Flags, IRP_PAGING_IO)) 
            {
                status = FsRtlCheckOplock(PTFSGetFcbOplock(pFcb), pIrp, pRequestContext, PTFSOplockComplete, PTFSPrePostIrp);
                if (status != STATUS_SUCCESS) 
                {
                    if (status == STATUS_PENDING) 
                        KdPrint(("[PTFS]::PTFSDispatchWrite FsRtlCheckOplock returned STATUS_PENDING(request)\n"));
                    else
                    {
                        FreeEventContext(pRequestContext);
                        pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_EVENT] = 0;
                        FreeEventContext(pEventContext);
                    }
                    __leave;
                }
            }

            status = RegisterPendingIrp(pDeviceObject, pIrp, pRequestContext, 0);
        }
    }
    __finally 
    {
        if (fcbLocked)
            PTFSFcbUnlock(pFcb);

        PTFSCompleteIrpRequest(pIrp, status, 0);

        KdPrint(("[PTFS]::PTFSDispatchWrite End\n"));
    }

    return status;
}

VOID
PTFSCompleteWrite(
	IN PIRP_ENTRY pIrpEntry,
	IN PEVENT_INFORMATION pEventInfo
)
{
    PIRP pIrp = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status = STATUS_SUCCESS;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;
    PFILE_OBJECT pFileObject = NULL;
    BOOLEAN isPagingIo = FALSE;

    pFileObject = pIrpEntry->pFileObject;
    ASSERT(pFileObject != NULL);

    KdPrint(("[PTFS]::PTFSCompleteWrite start [%wZ]\n", &pFileObject->FileName));

    pIrp = pIrpEntry->pIrp;
    pIrpSp = pIrpEntry->pIrpSp;

    isPagingIo = (pIrp->Flags & IRP_PAGING_IO);

    pCcb = pFileObject->FsContext2;
    ASSERT(pCcb != NULL);

    pFcb = pCcb->pFcb;
    ASSERT(pFcb != NULL);

    pCcb->UserContext = pEventInfo->Context;
    status = pEventInfo->Status;
    pIrp->IoStatus.Status = status;
    pIrp->IoStatus.Information = pEventInfo->BufferLength;

    if (NT_SUCCESS(status)) 
    {
        if (pFcb->AdvancedFCBHeader.FileSize.QuadPart < pEventInfo->Operation.Write.CurrentByteOffset.QuadPart) 
        {
            if (!isPagingIo)
                PTFSFcbLock(pFcb, TRUE);
       
            NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_SIZE, FILE_ACTION_MODIFIED);

            if (!isPagingIo)
                PTFSFcbUnlock(pFcb);

            InterlockedExchange64( &pFcb->AdvancedFCBHeader.FileSize.QuadPart, pEventInfo->Operation.Write.CurrentByteOffset.QuadPart);
        }

        PTFSFCBFlagsSetBit(pFcb, PTFS_FILE_CHANGE_LAST_WRITE);

        if (!isPagingIo)
            SetFlag(pFileObject->Flags, FO_FILE_MODIFIED);

        if (pEventInfo->BufferLength != 0 && pFileObject->Flags & FO_SYNCHRONOUS_IO && !isPagingIo) 
        {
            pFileObject->CurrentByteOffset.QuadPart = pEventInfo->Operation.Write.CurrentByteOffset.QuadPart;
            KdPrint(("[PTFS]::PTFSCompleteWriteUpdated CurrentByteOffset[%I64d]\n", pFileObject->CurrentByteOffset.QuadPart));
        }
    }

    PTFSCompleteIrpRequest(pIrp, pIrp->IoStatus.Status, pIrp->IoStatus.Information);
    KdPrint(("[PTFS]::PTFSCompleteWrite end\n"));
}
