#include "volume.h"
#include "init.h"
#include "util.h"
#include "notification.h"
#include "event.h"

#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSDispatchQueryVolumeInformation)
#pragma alloc_text (PAGE, PTFSDispatchSetVolumeInformation)
#endif


NTSTATUS
PTFSDispatchQueryVolumeInformation(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
	NTSTATUS status = STATUS_INVALID_PARAMETER;
	PIO_STACK_LOCATION pIrpSp = NULL;
	PVOID pBuffer = NULL;
	PFILE_OBJECT pFileObject = NULL;
	PPTFSVCB pVcb = NULL;
	PPTFSDCB pDcb = NULL;
	PPTFSCCB pCcb = NULL;
	ULONG info = 0;
	ULONG RequiredLength = 0;

	__try
	{

		KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation Start PID[%lu]\n", IoGetRequestorProcessId(pIrp)));

		pVcb = pDeviceObject->DeviceExtension;
		if (GETIDENTIFIERTYPE(pVcb) != VCB)
		{
			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation Invalid identifier type\n"));
			status = STATUS_INVALID_PARAMETER;
			__leave;
		}

		pDcb = pVcb->pDcb;

		pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
		pBuffer = pIrp->AssociatedIrp.SystemBuffer;
		pFileObject = pIrpSp->FileObject;

		if (pFileObject == NULL)
		{
			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileObject is null\n"));
			status = STATUS_INVALID_PARAMETER;
			__leave;
		}

		KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileName[%wZ]\n", pFileObject->FileName));
		switch (pIrpSp->Parameters.QueryVolume.FsInformationClass)
		{
		case FileFsVolumeInformation:
		{
			PFILE_FS_VOLUME_INFORMATION pFsVolInfo = NULL;

			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsVolumeInformation\n"));
			if (pVcb->bHasEventWait)
				break;

			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation Still no threads for processing available FileFsVolumeInformation\n"));
			if (pIrpSp->Parameters.QueryVolume.Length < sizeof(FILE_FS_VOLUME_INFORMATION))
			{
				KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsVolumeInformation Invalid Length\n"));
				status = STATUS_BUFFER_OVERFLOW;
				__leave;
			}

			pFsVolInfo = (PFILE_FS_VOLUME_INFORMATION)pBuffer;
			pFsVolInfo->VolumeCreationTime.QuadPart = 0;
			pFsVolInfo->VolumeSerialNumber = PTFS_VOLUME_SERIAL_NUMBER;
			pFsVolInfo->VolumeLabelLength = (USHORT)wcslen(VOLUME_LABEL) * sizeof(WCHAR);
			pFsVolInfo->SupportsObjects = FALSE;
			RequiredLength = sizeof(FILE_FS_VOLUME_INFORMATION) + pFsVolInfo->VolumeLabelLength - sizeof(WCHAR);
			if (pIrpSp->Parameters.QueryVolume.Length < RequiredLength)
			{
				KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsVolumeInformation Invalid Length2\n"));
				pIrp->IoStatus.Information = sizeof(FILE_FS_VOLUME_INFORMATION);
				status = STATUS_BUFFER_OVERFLOW;
				__leave;
			}

			RtlCopyMemory(pFsVolInfo->VolumeLabel, VOLUME_LABEL, pFsVolInfo->VolumeLabelLength);
			pIrp->IoStatus.Information = RequiredLength;
			status = STATUS_SUCCESS;
			__leave;
		}
		break;

		case FileFsLabelInformation:
			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsLabelInformation\n"));
			break;

		case FileFsSizeInformation:
		{
			PFILE_FS_SIZE_INFORMATION pSizeInfo = NULL;
			ULONGLONG freeBytesAvailable = 512 * 1024 * 1024;
			ULONGLONG totalBytes = 1024 * 1024 * 1024;

			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsSizeInformation\n"));
			if (pVcb->bHasEventWait)
				break;

			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation Still no threads for processing available FileFsSizeInformation\n"));
			if (pIrpSp->Parameters.QueryVolume.Length < sizeof(FILE_FS_SIZE_INFORMATION))
			{
				KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsSizeInformation Invalid Length\n"));
				status = STATUS_BUFFER_OVERFLOW;
				__leave;
			}

			pSizeInfo = (PFILE_FS_SIZE_INFORMATION)pBuffer;
			pSizeInfo->TotalAllocationUnits.QuadPart = totalBytes / PTFS_DEFAULT_ALLOCATION_UNIT_SIZE;
			pSizeInfo->AvailableAllocationUnits.QuadPart = freeBytesAvailable / PTFS_DEFAULT_ALLOCATION_UNIT_SIZE;
			pSizeInfo->SectorsPerAllocationUnit = PTFS_DEFAULT_ALLOCATION_UNIT_SIZE / PTFS_DEFAULT_SECTOR_SIZE;
			pSizeInfo->BytesPerSector = PTFS_DEFAULT_SECTOR_SIZE;
			pIrp->IoStatus.Information = sizeof(FILE_FS_SIZE_INFORMATION);
			status = STATUS_SUCCESS;
			__leave;
		}
		break;

		case FileFsDeviceInformation:
		{
			PFILE_FS_DEVICE_INFORMATION pDeviceInfo = NULL;

			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsDeviceInformation\n"));
			pDeviceInfo = (PFILE_FS_DEVICE_INFORMATION)pIrp->AssociatedIrp.SystemBuffer;
			if (pIrpSp->Parameters.QueryVolume.Length < sizeof(FILE_FS_DEVICE_INFORMATION))
			{
				KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsDeviceInformation Invalid Length\n"));
				status = STATUS_BUFFER_TOO_SMALL;
				info = sizeof(FILE_FS_DEVICE_INFORMATION);
				__leave;
			}
			pDeviceInfo->DeviceType = pDcb->DeviceType;
			pDeviceInfo->Characteristics = pDcb->ulDeviceCharacteristics;
			status = STATUS_SUCCESS;
			info = sizeof(FILE_FS_DEVICE_INFORMATION);
			__leave;
		}
		break;

		case FileFsAttributeInformation:
		{
			PFILE_FS_ATTRIBUTE_INFORMATION pFsAttrInfo = NULL;

			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsAttributeInformation\n"));
			if (pVcb->bHasEventWait)
				break;

			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation Still no threads for processing available FileFsAttributeInformation\n"));
			if (pIrpSp->Parameters.QueryVolume.Length < sizeof(FILE_FS_ATTRIBUTE_INFORMATION))
			{
				KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsAttributeInformation Invalid Length\n"));
				status = STATUS_BUFFER_OVERFLOW;
				__leave;
			}

			pFsAttrInfo = (PFILE_FS_ATTRIBUTE_INFORMATION)pIrp->AssociatedIrp.SystemBuffer;
			pFsAttrInfo->FileSystemAttributes = FILE_SUPPORTS_HARD_LINKS | FILE_CASE_SENSITIVE_SEARCH | FILE_CASE_PRESERVED_NAMES;
			pFsAttrInfo->MaximumComponentNameLength = 255;
			pFsAttrInfo->FileSystemNameLength = 8;
			RequiredLength = sizeof(FILE_FS_ATTRIBUTE_INFORMATION) + pFsAttrInfo->FileSystemNameLength - sizeof(WCHAR);

			if (pIrpSp->Parameters.QueryVolume.Length < RequiredLength)
			{
				KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsAttributeInformation Invalid Length2\n"));
				pIrp->IoStatus.Information = sizeof(FILE_FS_ATTRIBUTE_INFORMATION);
				status = STATUS_BUFFER_OVERFLOW;
				__leave;
			}

			RtlCopyMemory(pFsAttrInfo->FileSystemName, L"PTFS", pFsAttrInfo->FileSystemNameLength);
			pIrp->IoStatus.Information = RequiredLength;
			status = STATUS_SUCCESS;
			__leave;
		}
		break;

		case FileFsControlInformation:
			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsControlInformation\n"));
			break;

		case FileFsFullSizeInformation:
			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsFullSizeInformation\n"));
			break;
		case FileFsObjectIdInformation:
			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsObjectIdInformation\n"));
			break;

		case FileFsMaximumInformation:
			KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation FileFsMaximumInformation\n"));
			break;

		default:
			break;
		}

		if (pIrpSp->Parameters.QueryVolume.FsInformationClass == FileFsVolumeInformation ||
			pIrpSp->Parameters.QueryVolume.FsInformationClass == FileFsSizeInformation ||
			pIrpSp->Parameters.QueryVolume.FsInformationClass == FileFsAttributeInformation ||
			pIrpSp->Parameters.QueryVolume.FsInformationClass == FileFsFullSizeInformation)
		{
			ULONG eventLength = sizeof(EVENT_CONTEXT);
			PEVENT_CONTEXT pEventContext = NULL;

			pCcb = pFileObject->FsContext2;
			if (pCcb && !PTFSCheckCCB(pVcb->pDcb, pFileObject->FsContext2))
			{
				KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation invalid ccb\n"));
				status = STATUS_INVALID_PARAMETER;
				__leave;
			}

			pEventContext = AllocateEventContext(pVcb->pDcb, pIrp, eventLength, NULL);
			if (pEventContext == NULL)
			{
				KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation failed to AllocateEventContext\n"));
				status = STATUS_INSUFFICIENT_RESOURCES;
				__leave;
			}

			if (pCcb)
			{
				pEventContext->Context = pCcb->UserContext;
				pEventContext->FileFlags = PTFSCCBFlagsGet(pCcb);
			}

			pEventContext->Operation.Volume.FsInformationClass = pIrpSp->Parameters.QueryVolume.FsInformationClass;
			pEventContext->Operation.Volume.BufferLength = pIrpSp->Parameters.QueryVolume.Length;

			status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, 0);
		}
	}
	__finally
	{
		PTFSCompleteIrpRequest(pIrp, status, info);
		KdPrint(("[PTFS]::PTFSDispatchQueryVolumeInformation end\n"));
	}

	return status;
}

NTSTATUS
PTFSDispatchSetVolumeInformation(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
	NTSTATUS status = STATUS_INVALID_PARAMETER;
	PPTFSVCB pVcb = NULL;
	PPTFSDCB pDcb = NULL;
	PIO_STACK_LOCATION pIrpSp = NULL;
	PVOID pBuffer = NULL;
	FS_INFORMATION_CLASS FsInformationClass;

	__try
	{
		KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation Start PID[%lu]\n", IoGetRequestorProcessId(pIrp)));

		pVcb = pDeviceObject->DeviceExtension;
		if (GETIDENTIFIERTYPE(pVcb) != VCB)
		{
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation Invalid identifier type(VCB) \n"));
			status = STATUS_INVALID_PARAMETER;
			__leave;
		}

		pDcb = pVcb->pDcb;
		pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
		pBuffer = pIrp->AssociatedIrp.SystemBuffer;
		FsInformationClass = pIrpSp->Parameters.SetVolume.FsInformationClass;

		switch (FsInformationClass)
		{
		case FileFsLabelInformation:
		{
			PFILE_FS_LABEL_INFORMATION pLabelInfo = (PFILE_FS_LABEL_INFORMATION)pBuffer;

			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsLabelInformation\n"));

			if (sizeof(FILE_FS_LABEL_INFORMATION) > pIrpSp->Parameters.SetVolume.Length)
			{
				KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsLabelInformation Invalid Length\n"));
				status = STATUS_INVALID_PARAMETER;
				__leave;
			}

			ExAcquireResourceExclusiveLite(&pDcb->Resource, TRUE);
			if (pDcb->pwszVolumeLabel != NULL)
				PTFSFree(pDcb->pwszVolumeLabel);

			pDcb->pwszVolumeLabel = PTFSAllocate(pLabelInfo->VolumeLabelLength + sizeof(WCHAR));
			if (pDcb->pwszVolumeLabel == NULL)
			{
				KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsLabelInformation failed to allocate\n"));
				status = STATUS_INSUFFICIENT_RESOURCES;
				__leave;
			}

			RtlCopyMemory(pDcb->pwszVolumeLabel, pLabelInfo->VolumeLabel, pLabelInfo->VolumeLabelLength);
			pDcb->pwszVolumeLabel[pLabelInfo->VolumeLabelLength / sizeof(WCHAR)] = '\0';
			ExReleaseResourceLite(&pDcb->Resource);
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsLabelInformation volume label changed to [%ws]\n", pDcb->pwszVolumeLabel));
			status = STATUS_SUCCESS;
		}
		break;
		case FileFsVolumeInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsVolumeInformation\n"));
			break;
		case FileFsSizeInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsSizeInformation\n"));
			break;
		case FileFsDeviceInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsDeviceInformation\n"));
			break;
		case FileFsAttributeInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsAttributeInformation\n"));
			break;
		case FileFsControlInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsControlInformation\n"));
			break;
		case FileFsFullSizeInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsFullSizeInformation\n"));
			break;
		case FileFsObjectIdInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsObjectIdInformation\n"));
			break;
		case FileFsDriverPathInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsDriverPathInformation\n"));
			break;
		case FileFsVolumeFlagsInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsVolumeFlagsInformation\n"));
			break;
		case FileFsSectorSizeInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsSectorSizeInformation\n"));
			break;
		case FileFsDataCopyInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsDataCopyInformation\n"));
			break;
		case FileFsMetadataSizeInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsMetadataSizeInformation\n"));
			break;
/*
		case FileFsMaximumInformation:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation FileFsMaximumInformation\n"));
			break;
*/
		default:
			KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation unknown FsInformationClass[%d]\n", FsInformationClass));
			break;
		}

	}
	__finally
	{
		PTFSCompleteIrpRequest(pIrp, status, 0);
		KdPrint(("[PTFS]::PTFSDispatchSetVolumeInformation end\n"));
	}

	return status;
}

VOID
PTFSCompleteQueryVolumeInformation(
	IN PIRP_ENTRY pIrpEntry,
	IN PEVENT_INFORMATION pEventInfo,
	IN PDEVICE_OBJECT pDeviceObject
)
{
	PIRP pIrp = NULL;
	PIO_STACK_LOCATION pIrpSp = NULL;
	NTSTATUS status = STATUS_SUCCESS;
	ULONG info = 0;
	ULONG bufferLen = 0;
	PVOID pBuffer = NULL;
	PPTFSDCB pDcb = NULL;
	PPTFSVCB pVcb = NULL;

	KdPrint(("[PTFS]::PTFSCompleteQueryVolumeInformation start\n"));

	pIrp = pIrpEntry->pIrp;
	pIrpSp = pIrpEntry->pIrpSp;

	pVcb = pDeviceObject->DeviceExtension;
	pDcb = pVcb->pDcb;

	pBuffer = pIrp->AssociatedIrp.SystemBuffer;
	bufferLen = pIrpSp->Parameters.QueryVolume.Length;

	if (bufferLen == 0 || pBuffer == NULL || bufferLen < pEventInfo->BufferLength)
	{
		KdPrint(("[PTFS]::PTFSCompleteQueryVolumeInformation invalid Length\n"));
		info = 0;
		status = STATUS_INSUFFICIENT_RESOURCES;
	}
	else
	{
		if (pIrpSp->Parameters.QueryVolume.FsInformationClass == FileFsAttributeInformation &&
			IS_DEVICE_READ_ONLY(pIrpEntry->pIrpSp->DeviceObject))
		{
			PFILE_FS_ATTRIBUTE_INFORMATION pAttrInfo = (PFILE_FS_ATTRIBUTE_INFORMATION)pEventInfo->Buffer;
			pAttrInfo->FileSystemAttributes |= FILE_READ_ONLY_VOLUME;

			KdPrint(("[PTFS]::PTFSCompleteQueryVolumeInformation Adding FILE_READ_ONLY_VOLUME flag to attributes\n"));
		}

		ASSERT(pBuffer != NULL);

		if (pIrpSp->Parameters.QueryVolume.FsInformationClass == FileFsVolumeInformation &&
			pDcb->pwszVolumeLabel != NULL)
		{
			PFILE_FS_VOLUME_INFORMATION pVolumeInfo = (PFILE_FS_VOLUME_INFORMATION)pEventInfo->Buffer;
			ULONG remainingLength = bufferLen;
			ULONG bytesToCopy = 0;

			remainingLength -= FIELD_OFFSET(FILE_FS_VOLUME_INFORMATION, VolumeLabel[0]);
			bytesToCopy = (ULONG)wcslen(pDcb->pwszVolumeLabel) * sizeof(WCHAR);

			if (remainingLength < bytesToCopy)
				bytesToCopy = remainingLength;

			pVolumeInfo->VolumeLabelLength = bytesToCopy;
			RtlCopyMemory(pVolumeInfo->VolumeLabel, pDcb->pwszVolumeLabel, bytesToCopy);
			remainingLength -= bytesToCopy;
			pEventInfo->BufferLength = bufferLen - remainingLength;
		}

		RtlZeroMemory(pBuffer, bufferLen);
		RtlCopyMemory(pBuffer, pEventInfo->Buffer, pEventInfo->BufferLength);
		info = pEventInfo->BufferLength;
		status = pEventInfo->Status;
	}

	PTFSCompleteIrpRequest(pIrp, status, info);

	KdPrint(("[PTFS]::PTFSCompleteQueryVolumeInformation end\n"));
}