#include "timeout.h"
#include "init.h"
#include "util.h"
#include "notification.h"
#include <ntstrsafe.h>
#include "irp_buffer_helper.h"
#include "dispatch.h"

VOID
PTFSUnmount(
    IN PPTFSDCB pDcb
)
{
    ULONG eventLength = 0;
    PEVENT_CONTEXT pEventContext = NULL;
    PDRIVER_EVENT_CONTEXT pDriverEventContext = NULL;
    PKEVENT pCompletedEvent = NULL;
    LARGE_INTEGER timeout;
    PPTFSVCB pVcb = pDcb->pVcb;
    ULONG deviceNamePos = 0;

    KdPrint(("[PTFS]::PTFSUnmount Start\n"));

    eventLength = sizeof(EVENT_CONTEXT);
    pEventContext = AllocateEventContextRaw(eventLength);

    if (pEventContext == NULL)
    {
        KdPrint(("[PTFS]::PTFSUnmount Failed to Allocate event context\n"));
        if (pVcb) PTFSEventRelease(pVcb->pDeviceObject, NULL);
        return;
    }

    pDriverEventContext = CONTAINING_RECORD(pEventContext, DRIVER_EVENT_CONTEXT, EventContext);
    pCompletedEvent = PTFSAllocate(sizeof(KEVENT));
     if (pCompletedEvent) 
    {
        KeInitializeEvent(pCompletedEvent, NotificationEvent, FALSE);
        pDriverEventContext->pCompletedEvent = pCompletedEvent;
    }

    deviceNamePos = pDcb->punstrSymbolicLinkName->Length / sizeof(WCHAR) - 1;
    deviceNamePos = SearchWcharinUnicodeStringWithUlong(pDcb->punstrSymbolicLinkName, L'\\', deviceNamePos, 0);
    RtlStringCchCopyW(pEventContext->Operation.Unmount.DeviceName, sizeof(pEventContext->Operation.Unmount.DeviceName) / sizeof(WCHAR),
        &(pDcb->punstrSymbolicLinkName->Buffer[deviceNamePos]));

    KdPrint(("[PTFS]::PTFSUnmount Send Unmount to DeviceName[%ws]\n", pEventContext->Operation.Unmount.DeviceName));

    EventNotification(&pDcb->pPTFSGlobal->NotifyServiceIrpList, pEventContext);

    if (pCompletedEvent) 
    {
        timeout.QuadPart = -1 * 10 * 1000 * 10; // 10 sec
        KeWaitForSingleObject(pCompletedEvent, Executive, KernelMode, FALSE, &timeout);
    }

    if (pVcb)
        PTFSEventRelease(pVcb->pDeviceObject, NULL);
 
    if (pCompletedEvent)
        PTFSFree(pCompletedEvent);

    KdPrint(("[PTFS]::PTFSUnmount End\n"));
}

VOID
CheckKeepAlive(
    IN PPTFSDCB pDcb
)
{
    LARGE_INTEGER tickCount;
    PPTFSVCB pVcb = NULL;

    KeEnterCriticalRegion();
    KeQueryTickCount(&tickCount);
    ExAcquireResourceSharedLite(&pDcb->Resource, TRUE);

    if (pDcb->liTickCount.QuadPart < tickCount.QuadPart) 
    {
        pVcb = pDcb->pVcb;
        ExReleaseResourceLite(&pDcb->Resource);

        KdPrint(("[PTFS]::CheckKeepAlive Timeout reached so perform an umount\n"));

        if (IsUnmountPendingVcb(pVcb)) 
        {
            KdPrint(("[PTFS]::CheckKeepAlive  Volume is not mounted\n"));
            KeLeaveCriticalRegion();
            return;
        }

   //     PTFSUnmount(pDcb);
    }
    else
        ExReleaseResourceLite(&pDcb->Resource);

    KeLeaveCriticalRegion();
}

NTSTATUS
ReleaseTimeoutPendingIrp(
    IN PPTFSDCB pDcb
)
{
    KIRQL oldIrql;
    PLIST_ENTRY thisEntry, nextEntry, listHead;
    PIRP_ENTRY pIrpEntry = NULL;
    LARGE_INTEGER tickCount;
    LIST_ENTRY completeList;
    PIRP pIrp = NULL;
    BOOLEAN shouldUnmount = FALSE;
    PPTFSVCB pVcb = pDcb->pVcb;

    KdPrint(("[PTFS]::ReleaseTimeoutPendingIrp start\n"));

    InitializeListHead(&completeList);
    ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);
    KeAcquireSpinLock(&pDcb->PendingIrpList.ListSpinLock, &oldIrql);

    if (IsListEmpty(&pDcb->PendingIrpList.ListHead)) 
    {
        KeReleaseSpinLock(&pDcb->PendingIrpList.ListSpinLock, oldIrql);
        KdPrint(("[PTFS]::ReleaseTimeoutPendingIrp IrpQueue is Empty\n"));
        return STATUS_SUCCESS;
    }

    KeQueryTickCount(&tickCount);

    listHead = &pDcb->PendingIrpList.ListHead;

    for (thisEntry = listHead->Flink; thisEntry != listHead; thisEntry = nextEntry) 
    {
        nextEntry = thisEntry->Flink;
        pIrpEntry = CONTAINING_RECORD(thisEntry, IRP_ENTRY, ListEntry);

        if (pIrpEntry->AsyncStatus == STATUS_SUCCESS && tickCount.QuadPart < pIrpEntry->liTickCount.QuadPart)
            continue;

  //      if (pIrpEntry->pIrpSp->MajorFunction == IRP_MJ_DIRECTORY_CONTROL) #226164  ̽  غ   ....
 //           continue;

        RemoveEntryList(thisEntry);

        KdPrint(("[PTFS]::ReleaseTimeoutPendingIrp timeout Irp[%d]\n", pIrpEntry->ulSerialNumber));
        pIrp = pIrpEntry->pIrp;

        if (pIrpEntry->pIrpSp->MajorFunction != IRP_MJ_CREATE)
        {
            if (pIrp == NULL) 
            {
                ASSERT(pIrpEntry->bCancelRoutineFreeMemory == FALSE);
                FreeIrpEntry(pIrpEntry);
                continue;
            }

            if (IoSetCancelRoutine(pIrp, NULL) == NULL) 
            {
                InitializeListHead(&pIrpEntry->ListEntry);
                pIrpEntry->bCancelRoutineFreeMemory = TRUE;
                continue;
            }
        }

        pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_IRP_ENTRY] = NULL;
        InsertTailList(&completeList, &pIrpEntry->ListEntry);
    }

    if (IsListEmpty(&pDcb->PendingIrpList.ListHead)) 
        KeClearEvent(&pDcb->PendingIrpList.NotEmptyEvent);

    KeReleaseSpinLock(&pDcb->PendingIrpList.ListSpinLock, oldIrql);
    shouldUnmount = !pVcb->bIsKeepaliveActive && !IsListEmpty(&completeList);

    while (!IsListEmpty(&completeList)) 
    {
        PIO_STACK_LOCATION pIrpSp = NULL;
        listHead = RemoveHeadList(&completeList);
        pIrpEntry = CONTAINING_RECORD(listHead, IRP_ENTRY, ListEntry);
        pIrp = pIrpEntry->pIrp;
        pIrpSp = pIrpEntry->pIrpSp;

        if (pIrpSp->MajorFunction == IRP_MJ_CREATE) 
        {
            BOOLEAN canceled = (pIrpEntry->liTickCount.QuadPart == 0);
            PFILE_OBJECT pFileObject = pIrpEntry->pFileObject;

            if (pFileObject != NULL) 
            {
                PPTFSCCB pCcb = pFileObject->FsContext2;
                if (pCcb != NULL)
                {
#ifdef _DBG
                    PPTFSFCB pFcb = pCcb->pFcb;
                    KdPrint(("[PTFS]::ReleaseTimeoutPendingIrp not null ccb Fcb[0x%x]\n", pFcb));
#endif
                }
            }

            PTFSCancelCreateIrp(pDcb->pDeviceObject, pIrpEntry, canceled ? STATUS_CANCELLED : STATUS_INSUFFICIENT_RESOURCES);
        }
        else 
            PTFSCompleteIrpRequest(pIrp, STATUS_INSUFFICIENT_RESOURCES, 0);

        FreeIrpEntry(pIrpEntry);
    }

    KdPrint(("[PTFS]::ReleaseTimeoutPendingIrp timeout End\n"));

    if (shouldUnmount)
    {
        KdPrint(("[PTFS]::ReleaseTimeoutPendingIrp shouldUnmount\n"));
  //      PTFSUnmount(pDcb);
    }

    return STATUS_SUCCESS;
}

NTSTATUS
ResetPendingIrpTimeout(
    IN PDEVICE_OBJECT pDeviceObject,
    IN OUT PIRP pIrp
)
{
    KIRQL oldIrql;
    PLIST_ENTRY thisEntry, nextEntry, listHead;
    PIRP_ENTRY pIrpEntry = NULL;
    PPTFSVCB pVcb = NULL;
    PEVENT_INFORMATION pEventInfo = NULL;
    ULONG timeout = 0;

    KdPrint(("[PTFS]::ResetPendingIrpTimeout Start\\n"));

    GET_IRP_BUFFER_OR_RETURN(pIrp, pEventInfo)

    timeout = pEventInfo->Operation.ResetTimeout.Timeout;

    if (PTFS_IRP_PENDING_TIMEOUT_RESET_MAX < timeout)
        timeout = PTFS_IRP_PENDING_TIMEOUT_RESET_MAX;

    pVcb = pDeviceObject->DeviceExtension;
    if (GETIDENTIFIERTYPE(pVcb) != VCB)
    {
        KdPrint(("[PTFS]::ResetPendingIrpTimeout not matched identitype\\n"));
        return STATUS_INVALID_PARAMETER;
    }

    ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);
    KeAcquireSpinLock(&pVcb->pDcb->PendingIrpList.ListSpinLock, &oldIrql);

    listHead = &pVcb->pDcb->PendingIrpList.ListHead;

    for (thisEntry = listHead->Flink; thisEntry != listHead; thisEntry = nextEntry) 
    {
        nextEntry = thisEntry->Flink;
        pIrpEntry = CONTAINING_RECORD(thisEntry, IRP_ENTRY, ListEntry);

        if (pIrpEntry->ulSerialNumber != pEventInfo->SerialNumber)
            continue;

        UpdateTimeout(&pIrpEntry->liTickCount, timeout);
        break;
    }

    KeReleaseSpinLock(&pVcb->pDcb->PendingIrpList.ListSpinLock, oldIrql);
    KdPrint(("[PTFS]::ResetPendingIrpTimeout End\\n"));
    return STATUS_SUCCESS;
}

KSTART_ROUTINE PTFSTimeoutThread;
VOID
PTFSTimeoutThread(
    IN PVOID pDcbParam
)
{
    NTSTATUS status;
    KTIMER timer;
    PVOID pPollevents[3];
    LARGE_INTEGER timeout = { 0 };
    BOOLEAN waitObj = TRUE;
    LARGE_INTEGER LastTime = { 0 };
    LARGE_INTEGER CurrentTime = { 0 };
    PPTFSVCB pVcb = NULL;
    PPTFSDCB pDcb = pDcbParam;

    KdPrint(("[PTFS]::PTFSTimeoutThread Start\\n"));

    KeInitializeTimerEx(&timer, SynchronizationTimer);

    pPollevents[0] = (PVOID)&pDcb->KillEvent;
    pPollevents[1] = (PVOID)&pDcb->ForceTimeoutEvent;
    pPollevents[2] = (PVOID)&timer;

    pVcb = pDcb->pVcb;

    KeSetTimerEx(&timer, timeout, PTFS_CHECK_INTERVAL, NULL);

    KeQuerySystemTime(&LastTime);

    while (waitObj)
    {
        status = KeWaitForMultipleObjects(3, pPollevents, WaitAny, Executive, KernelMode, FALSE, NULL, NULL);

        if (!NT_SUCCESS(status) || status == STATUS_WAIT_0) 
        {
            KdPrint(("[PTFS]::PTFSTimeoutThread catched KillEvent\\n"));
            waitObj = FALSE;
        }
        else 
        {
            KeClearEvent(&pDcb->ForceTimeoutEvent);
            KeQuerySystemTime(&CurrentTime);

            if ((CurrentTime.QuadPart - LastTime.QuadPart) > ((PTFS_CHECK_INTERVAL + 2000) * 10000)) 
            {
                KdPrint(("[PTFS]::PTFSTimeoutThread Wake from sleep detected\\n"));
            }
            else 
            {
                ReleaseTimeoutPendingIrp(pDcb);
                if (!pVcb->bIsKeepaliveActive)
                    CheckKeepAlive(pDcb);
            }
            KeQuerySystemTime(&LastTime);
        }
    }

    KeCancelTimer(&timer);
    KdPrint(("[PTFS]::PTFSTimeoutThread End\\n"));
    PsTerminateSystemThread(STATUS_SUCCESS);
}

NTSTATUS
StartCheckThread(
    IN PPTFSDCB pDcb
)
{
    NTSTATUS status;
    HANDLE thread;

    KdPrint(("[PTFS]::StartCheckThread Start\n"));

    status = PsCreateSystemThread(&thread, THREAD_ALL_ACCESS, NULL, NULL, NULL, (PKSTART_ROUTINE)PTFSTimeoutThread, pDcb);

    if (!NT_SUCCESS(status))
    {
        KdPrint(("[PTFS]::StartCheckThread Failed to PsCreateSystemThread status[0x%x]\n", status));
        return status;
    }

    ObReferenceObjectByHandle(thread, THREAD_ALL_ACCESS, NULL, KernelMode,
        (PVOID*)&pDcb->TimeoutThread, NULL);

    ZwClose(thread);

    KdPrint(("[PTFS]::StartCheckThread End\n"));
    return STATUS_SUCCESS;
}

VOID StopCheckThread(
    IN PPTFSDCB pDcb
)
{
    KdPrint(("[PTFS]::StopCheckThread Start\n"));

    if (KeSetEvent(&pDcb->KillEvent, 0, FALSE) > 0 && pDcb->TimeoutThread) 
    {
        KdPrint(("[PTFS]::StopCheckThread Waiting for Timeout thread to terminate.\n"));
        ASSERT(KeGetCurrentIrql() <= APC_LEVEL);
        KeWaitForSingleObject(pDcb->TimeoutThread, Executive, KernelMode, FALSE, NULL);
        KdPrint(("[PTFS]::StopCheckThread Timeout thread successfully terminated.\n"));
        ObDereferenceObject(pDcb->TimeoutThread);
        pDcb->TimeoutThread = NULL;
    }

    KdPrint(("[PTFS]::StopCheckThread End\n"));
}

VOID
UpdateTimeout(
    OUT PLARGE_INTEGER TickCount,
    IN ULONG Timeout
)
{
    KeQueryTickCount(TickCount);
    TickCount->QuadPart += Timeout * 1000 * 10 / KeQueryTimeIncrement();
}


