#include "Security.h"
#include "init.h"
#include "util.h"
#include "notification.h"
#include "event.h"


#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSDispatchQuerySecurity)
#pragma alloc_text (PAGE, PTFSDispatchSetSecurity)
#endif


NTSTATUS
PTFSDispatchQuerySecurity(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status = STATUS_NOT_IMPLEMENTED;
    PFILE_OBJECT pFileObject = NULL;
    ULONG info = 0;
    ULONG bufferLength = 0;
    PSECURITY_INFORMATION pSecurityInfo = NULL;
    PPTFSFCB pFcb = NULL;
    PPTFSDCB pDcb = NULL;
    PPTFSVCB pVcb = NULL;
    PPTFSCCB pCcb = NULL;
    ULONG eventLength = 0;
    PEVENT_CONTEXT pEventContext = NULL;
    ULONG flags = 0;

    __try 
    {
        KdPrint(("[PTFS]::PTFSDispatchQuerySecurity start\n"));

        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
        pFileObject = pIrpSp->FileObject;

        if (pFileObject == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity FileObjet is null\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        pVcb = pDeviceObject->DeviceExtension;
        if (GETIDENTIFIERTYPE(pVcb) != VCB) 
        {
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity Invalid identifier type\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        pDcb = pVcb->pDcb;

        KdPrint(("[PTFS]::PTFSDispatchQuerySecurity PID[%lu]\n", IoGetRequestorProcessId(pIrp)));

        pCcb = pFileObject->FsContext2;
        if (pCcb == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity ccb is null\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }
        pFcb = pCcb->pFcb;
        if (pFcb == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity fcb is null\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        bufferLength = pIrpSp->Parameters.QuerySecurity.Length;
        pSecurityInfo = &pIrpSp->Parameters.QuerySecurity.SecurityInformation;

        if (*pSecurityInfo & OWNER_SECURITY_INFORMATION) 
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity OWNER_SECURITY_INFORMATION\n"));

        if (*pSecurityInfo & GROUP_SECURITY_INFORMATION)
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity GROUP_SECURITY_INFORMATION\n"));

        if (*pSecurityInfo & DACL_SECURITY_INFORMATION)
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity DACL_SECURITY_INFORMATION\n"));

        if (*pSecurityInfo & SACL_SECURITY_INFORMATION)
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity SACL_SECURITY_INFORMATION\n"));

        if (*pSecurityInfo & LABEL_SECURITY_INFORMATION)
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity LABEL_SECURITY_INFORMATION\n"));

        PTFSFcbLock(pFcb, TRUE);

        eventLength = sizeof(EVENT_CONTEXT) + pFcb->unstrFileName.Length;
        pEventContext = AllocateEventContext(pDcb, pIrp, eventLength, pCcb);
        if (pEventContext == NULL)
        {
            KdPrint(("[PTFS]::PTFSDispatchQuerySecurity failed to AllocateEventContext\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        if (pIrp->UserBuffer != NULL && bufferLength > 0) 
        {
            if (pIrp->MdlAddress == NULL) 
            {
                status = AllocateMdl(pIrp, bufferLength);
                if (!NT_SUCCESS(status)) 
                {
                    KdPrint(("[PTFS]::PTFSDispatchQuerySecurity failed to AllocateMdl status[0x%x]\n", status));
                    FreeEventContext(pEventContext);
                    __leave;
                }
                flags = PTFS_MDL_ALLOCATED;
            }
        }

        pEventContext->Context = pCcb->UserContext;
        pEventContext->Operation.Security.SecurityInformation = *pSecurityInfo;
        pEventContext->Operation.Security.BufferLength = bufferLength;
        pEventContext->Operation.Security.FileNameLength = pFcb->unstrFileName.Length;
        RtlCopyMemory(pEventContext->Operation.Security.FileName, pFcb->unstrFileName.Buffer, pFcb->unstrFileName.Length);

        status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, flags);
    }
    __finally 
    {
        if (pFcb)
            PTFSFcbUnlock(pFcb);

        PTFSCompleteIrpRequest(pIrp, status, info);

        KdPrint(("[PTFS]::PTFSDispatchQuerySecurity end\n"));
    }

    return status;
}

NTSTATUS
PTFSDispatchSetSecurity(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
	PIO_STACK_LOCATION pIrpSp = NULL;
	PFILE_OBJECT pFileObject = NULL;
	PPTFSVCB pVcb = NULL;
	PPTFSDCB pDcb = NULL;
	PPTFSCCB pCcb = NULL;
	PPTFSFCB pFcb = NULL;
	NTSTATUS status = STATUS_NOT_IMPLEMENTED;
	ULONG info = 0;
	PSECURITY_INFORMATION pSecurityInfo = NULL;
	PSECURITY_DESCRIPTOR pSecurityDescriptor = NULL;
	ULONG securityDescLength = 0;
	ULONG eventLength = 0;
	PEVENT_CONTEXT pEventContext = NULL;

	__try
	{
		KdPrint(("[PTFS]::PTFSDispatchQuerySecurity start\n"));

		pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
		pFileObject = pIrpSp->FileObject;

		if (pFileObject == NULL)
		{
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity FileObject is null\n"));
			status = STATUS_INVALID_PARAMETER;
			__leave;
		}

		pVcb = pDeviceObject->DeviceExtension;
		if (GETIDENTIFIERTYPE(pVcb) != VCB)
		{
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity invalid identifier Type\n"));
			status = STATUS_INVALID_PARAMETER;
			__leave;
		}

		pDcb = pVcb->pDcb;

		KdPrint(("[PTFS]::PTFSDispatchQuerySecurity ProcessId[%lu]\n", IoGetRequestorProcessId(pIrp)));

		pCcb = pFileObject->FsContext2;
		if (pCcb == NULL)
		{
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity ccb is null\n"));
			status = STATUS_INVALID_PARAMETER;
			__leave;
		}

		pFcb = pCcb->pFcb;
		if (pFcb == NULL)
		{
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity fcb is null\n"));
			status = STATUS_INSUFFICIENT_RESOURCES;
			__leave;
		}

		PTFSFcbLock(pFcb, TRUE);

		pSecurityInfo = &pIrpSp->Parameters.SetSecurity.SecurityInformation;

		if (*pSecurityInfo & OWNER_SECURITY_INFORMATION)
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity OWNER_SECURITY_INFORMATION\n"));

		if (*pSecurityInfo & GROUP_SECURITY_INFORMATION)
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity GROUP_SECURITY_INFORMATION\n"));

		if (*pSecurityInfo & DACL_SECURITY_INFORMATION)
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity DACL_SECURITY_INFORMATION\n"));

		if (*pSecurityInfo & SACL_SECURITY_INFORMATION)
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity SACL_SECURITY_INFORMATION\n"));

		if (*pSecurityInfo & LABEL_SECURITY_INFORMATION)
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity LABEL_SECURITY_INFORMATION\n"));

		pSecurityDescriptor = pIrpSp->Parameters.SetSecurity.SecurityDescriptor;
		securityDescLength = RtlLengthSecurityDescriptor(pSecurityDescriptor);
		eventLength = sizeof(EVENT_CONTEXT) + securityDescLength + pFcb->unstrFileName.Length + 3;

		if (EVENT_CONTEXT_MAX_SIZE < eventLength)
		{
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity SecurityDescriptor is too big size[%d], (limit size[%d])\n", eventLength, EVENT_CONTEXT_MAX_SIZE));
			status = STATUS_INSUFFICIENT_RESOURCES;
			__leave;
		}

		pEventContext = AllocateEventContext(pVcb->pDcb, pIrp, eventLength, pCcb);
		if (pEventContext == NULL)
		{
			KdPrint(("[PTFS]::PTFSDispatchQuerySecurity Failed to AllocateEventContext\n"));
			status = STATUS_INSUFFICIENT_RESOURCES;
			__leave;
		}

		pEventContext->Context = pCcb->UserContext;
		pEventContext->Operation.SetSecurity.SecurityInformation = *pSecurityInfo;
		pEventContext->Operation.SetSecurity.BufferLength = securityDescLength;
		pEventContext->Operation.SetSecurity.BufferOffset = (FIELD_OFFSET(EVENT_CONTEXT, Operation.SetSecurity.FileName[0]) + pFcb->unstrFileName.Length + sizeof(WCHAR) + 3) & ~0x03;

		RtlCopyMemory((PCHAR)pEventContext + pEventContext->Operation.SetSecurity.BufferOffset, pSecurityDescriptor, securityDescLength);
		pEventContext->Operation.SetSecurity.FileNameLength = pFcb->unstrFileName.Length;
		RtlCopyMemory(pEventContext->Operation.SetSecurity.FileName, pFcb->unstrFileName.Buffer, pFcb->unstrFileName.Length);

		status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, 0);
	}
	__finally
	{
		if (pFcb)
			PTFSFcbUnlock(pFcb);

		PTFSCompleteIrpRequest(pIrp, status, info);

		KdPrint(("[PTFS]::PTFSDispatchQuerySecurity end\n"));
	}

	return status;
}

VOID
PTFSCompleteQuerySecurity(
    IN PIRP_ENTRY pIrpEntry,
    IN PEVENT_INFORMATION pEventInfo
)
{
    PIRP pIrp = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status;
    PVOID pBuffer = NULL;
    ULONG bufferLength = 0;
    ULONG info = 0;
    PFILE_OBJECT pFileObject = NULL;
    PPTFSCCB pCcb = NULL;

    KdPrint(("[PTFS]::PTFSCompleteQuerySecurity start\n"));

    pIrp = pIrpEntry->pIrp;
    pIrpSp = pIrpEntry->pIrpSp;

    if (pIrp->MdlAddress)
        pBuffer = MmGetSystemAddressForMdlNormalSafe(pIrp->MdlAddress);

    bufferLength = pIrpSp->Parameters.QuerySecurity.Length;

    if (pEventInfo->Status == STATUS_SUCCESS && pEventInfo->BufferLength <= bufferLength && pBuffer != NULL)
    {
        if (!RtlValidRelativeSecurityDescriptor(pEventInfo->Buffer, pEventInfo->BufferLength, pIrpSp->Parameters.QuerySecurity.SecurityInformation))
        {
            KdPrint(("[PTFS]::PTFSCompleteQuerySecurity Security Descriptor is not valid\n"));
            info = 0;
            status = STATUS_INVALID_PARAMETER;
        }
        else
        {
            RtlCopyMemory(pBuffer, pEventInfo->Buffer, pEventInfo->BufferLength);
            info = pEventInfo->BufferLength;
            status = STATUS_SUCCESS;
        }
    }
    else if (pEventInfo->Status == STATUS_BUFFER_OVERFLOW || (pEventInfo->Status == STATUS_SUCCESS && bufferLength < pEventInfo->BufferLength))
    {
        info = pEventInfo->BufferLength;
        status = STATUS_BUFFER_OVERFLOW;
    }
    else
    {
        info = 0;
        status = pEventInfo->Status;
    }

    if (pIrpEntry->ulFlags & PTFS_MDL_ALLOCATED)
    {
        FreeMdl(pIrp);
        pIrpEntry->ulFlags &= ~PTFS_MDL_ALLOCATED;
    }

    pFileObject = pIrpEntry->pFileObject;
    ASSERT(pFileObject != NULL);

    pCcb = pFileObject->FsContext2;

    if (pCcb != NULL)
        pCcb->UserContext = pEventInfo->Context;
    else
        KdPrint(("[PTFS]::PTFSCompleteQuerySecurity ccb is null\n"));

    PTFSCompleteIrpRequest(pIrp, status, info);
    KdPrint(("[PTFS]::PTFSCompleteQuerySecurity end\n"));
}

VOID
PTFSCompleteSetSecurity(
	IN PIRP_ENTRY pIrpEntry,
	IN PEVENT_INFORMATION pEventInfo
)
{
    PIRP pIrp = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    PFILE_OBJECT pFileObject = NULL;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;

    KdPrint(("[PTFS]::PTFSCompleteSetSecurity start\n"));

    pIrp = pIrpEntry->pIrp;
    pIrpSp = pIrpEntry->pIrpSp;

    pFileObject = pIrpEntry->pFileObject;
    ASSERT(pFileObject != NULL);

    pCcb = pFileObject->FsContext2;
    if (pCcb != NULL)
    {
        pCcb->UserContext = pEventInfo->Context;
        pFcb = pCcb->pFcb;
        ASSERT(pFcb != NULL);
    }
    else 
        KdPrint(("[PTFS]::PTFSCompleteSetSecurity ccb is null\n"));

    if (pFcb && NT_SUCCESS(pEventInfo->Status)) 
    {
        PTFSFcbLock(pFcb, TRUE);
        NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_SECURITY,  FILE_ACTION_MODIFIED);
        PTFSFcbUnlock(pFcb);
    }

    PTFSCompleteIrpRequest(pIrp, pEventInfo->Status, 0);
    KdPrint(("[PTFS]::PTFSCompleteSetSecurity end\n"));
}
