#include "Read.h"
#include "init.h"
#include "util.h"
#include "notification.h"
#include "event.h"


#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSDispatchRead)
#endif


NTSTATUS
PTFSDispatchRead(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
    PIO_STACK_LOCATION pIrpSp = NULL;
    PFILE_OBJECT pFileObject = NULL;
    ULONG bufferLength = 0;
    LARGE_INTEGER byteOffset;
    NTSTATUS status = STATUS_INVALID_PARAMETER;
    ULONG readLength = 0;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;
    PPTFSVCB pVcb = NULL;
    PVOID pCurrentAddress = NULL;
    PEVENT_CONTEXT pEventContext = NULL;
    ULONG eventLength = 0;
    BOOLEAN fcbLocked = FALSE;
    BOOLEAN isPagingIo = FALSE;
    BOOLEAN isSynchronousIo = FALSE;
    BOOLEAN noCache = FALSE;

    __try 
    {
        KdPrint(("[PTFS]::PTFSDispatchRead Start\n"));

        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
        pFileObject = pIrpSp->FileObject;

        if (pIrpSp->Parameters.Read.Length == 0) 
        {
            KdPrint(("[PTFS]::PTFSDispatchRead Parameters.Read.Length == 0\n"));
            pIrp->IoStatus.Information = 0;
            status = STATUS_SUCCESS;
            __leave;
        }

        if (pIrpSp->MinorFunction == IRP_MN_COMPLETE) 
        {
            KdPrint(("[PTFS]::PTFSDispatchRead IRP_MN_COMPLETE\n"));
            pIrp->MdlAddress = NULL;
            status = STATUS_SUCCESS;
            __leave;
        }

        if (pFileObject == NULL && pIrp->MdlAddress != NULL) 
        {
            pCurrentAddress = MmGetSystemAddressForMdlNormalSafe(pIrp->MdlAddress);
            if (pCurrentAddress == NULL) 
            {
                KdPrint(("[PTFS]::PTFSDispatchRead empty Address\n"));
                status = STATUS_INSUFFICIENT_RESOURCES;
                __leave;
            }

            KdPrint(("[PTFS]::PTFSDispatchRead FileObject is null and mdladrdress is not null\n"));
            readLength = pIrpSp->Parameters.Read.Length;
            status = STATUS_SUCCESS;
            __leave;
        }

        if (pFileObject == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchRead FileObject is null\n"));
            status = STATUS_INVALID_DEVICE_REQUEST;
            __leave;
        }

        pVcb = pDeviceObject->DeviceExtension;
        if (GETIDENTIFIERTYPE(pVcb) != VCB ||!PTFSCheckCCB(pVcb->pDcb, pFileObject->FsContext2)) 
        {
            KdPrint(("[PTFS]::PTFSDispatchRead invalid identifier type\n"));
            status = STATUS_INVALID_DEVICE_REQUEST;
            __leave;
        }

        bufferLength = pIrpSp->Parameters.Read.Length;

        if (pIrpSp->Parameters.Read.ByteOffset.LowPart == FILE_USE_FILE_POINTER_POSITION &&
            pIrpSp->Parameters.Read.ByteOffset.HighPart == -1) 
        {
            KdPrint(("[PTFS]::PTFSDispatchRead use FileObject ByteOffset\n"));
            byteOffset = pFileObject->CurrentByteOffset;
        }
        else 
        {
            byteOffset = pIrpSp->Parameters.Read.ByteOffset;
        }

        KdPrint(("[PTFS]::PTFSDispatchRead PID[%lu], ByteCount[%lu], ByteOffset[%I64d]\n", IoGetRequestorProcessId(pIrp), bufferLength, byteOffset.QuadPart));

        if (bufferLength == 0) 
        {
            status = STATUS_SUCCESS;
            readLength = 0;
            __leave;
        }

        if (pIrp->MdlAddress == NULL) 
        {
            status = AllocateMdl(pIrp, pIrpSp->Parameters.Read.Length);
            if (!NT_SUCCESS(status)) 
            {
                KdPrint(("[PTFS]::PTFSDispatchRead failed to AllocateMdl status[0x%x]\n", status));
                __leave;
            }
        }

        pCcb = pFileObject->FsContext2;
        ASSERT(pCcb != NULL);

        pFcb = pCcb->pFcb;
        ASSERT(pFcb != NULL);

        if (pFcb->bBlockUserModeDispatch)
        {
            KdPrint(("[PTFS]::PTFSDispatchRead BlockUserModeDispatch\n"));
            pIrp->IoStatus.Information = 0;
            status = STATUS_SUCCESS;
            __leave;
        }

        if (PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY)) 
        {
            KdPrint(("[PTFS]::PTFSDispatchRead PTFS_FILE_DIRECTORY[%p]\n", pFcb));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        isPagingIo = (pIrp->Flags & IRP_PAGING_IO);

        if (pFileObject->Flags & FO_SYNCHRONOUS_IO)
            isSynchronousIo = TRUE;

        if (pIrp->Flags & IRP_NOCACHE)
            noCache = TRUE;


        if (!isPagingIo && (pFileObject->SectionObjectPointer != NULL) &&
            (pFileObject->SectionObjectPointer->DataSectionObject != NULL))
        {
            PTFSPagingIoLock(pFcb, FALSE);
            CcFlushCache(&pFcb->SectionObjectPointers, &pIrpSp->Parameters.Read.ByteOffset, pIrpSp->Parameters.Read.Length, NULL);
            PTFSPagingIoUnlock(pFcb);
        }

        PTFSFcbLock(pFcb, TRUE);
        fcbLocked = TRUE;
        eventLength = sizeof(EVENT_CONTEXT) + pFcb->unstrFileName.Length;

        pEventContext = AllocateEventContext(pVcb->pDcb, pIrp, eventLength, pCcb);
        if (pEventContext == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchRead failed to AllocateEventContext\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        pEventContext->Context = pCcb->UserContext;

        if (isPagingIo) 
        {
            KdPrint(("[PTFS]::PTFSDispatchRead Paging IO\n"));
            pEventContext->FileFlags |= PTFS_PAGING_IO;
        }

        if (isSynchronousIo)
        {
            KdPrint(("[PTFS]::PTFSDispatchRead Synchronous IO\n"));
            pEventContext->FileFlags |= PTFS_SYNCHRONOUS_IO;
        }

        if (noCache) 
        {
            KdPrint(("[PTFS]::PTFSDispatchRead Nocache\n"));
            pEventContext->FileFlags |= PTFS_NOCACHE;
        }

        pEventContext->Operation.Read.ByteOffset = byteOffset;
        pEventContext->Operation.Read.BufferLength = pIrpSp->Parameters.Read.Length;
        pEventContext->Operation.Read.FileNameLength = pFcb->unstrFileName.Length;
        RtlCopyMemory(pEventContext->Operation.Read.FileName, pFcb->unstrFileName.Buffer, pFcb->unstrFileName.Length);

        if (!FlagOn(pIrp->Flags, IRP_PAGING_IO)) 
        {
            status = PTFSCheckOplock(pFcb, pIrp, pEventContext, PTFSOplockComplete, PTFSPrePostIrp);

            if (status != STATUS_SUCCESS) 
            {
                if (status == STATUS_PENDING) 
                    KdPrint(("[PTFS]::PTFSDispatchRead FsRtlCheckOplock returned STATUS_PENDING\n"));
                else 
                    FreeEventContext(pEventContext);

                __leave;
            }

            if (!FsRtlCheckLockForReadAccess(&pFcb->FileLock, pIrp)) 
            {
                KdPrint(("[PTFS]::PTFSDispatchRead failed to FsRtlCheckLockForReadAccess returned STATUS_FILE_LOCK_CONFLICT\n"));
                status = STATUS_FILE_LOCK_CONFLICT;
                __leave;
            }
        }

        status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, 0);
    }
    __finally 
    {
        if (fcbLocked)
            PTFSFcbUnlock(pFcb);

        PTFSCompleteIrpRequest(pIrp, status, readLength);

        KdPrint(("[PTFS]::PTFSDispatchRead end\n"));
    }

    return status;
}

VOID
PTFSCompleteRead(
	IN PIRP_ENTRY pIrpEntry,
	IN PEVENT_INFORMATION pEventInfo
)
{
    PIO_STACK_LOCATION pIrpSp = NULL;
    PIRP pIrp = NULL;
    NTSTATUS status = STATUS_SUCCESS;
    ULONG readLength = 0;
    ULONG bufferLen = 0;
    PVOID pBuffer = NULL;
    PPTFSCCB pCcb = NULL;
    PFILE_OBJECT pFileObject = NULL;

    pFileObject = pIrpEntry->pFileObject;
    ASSERT(pFileObject != NULL);

    KdPrint(("[PTFS]::PTFSCompleteRead start FileName[%wZ]\n", &pFileObject->FileName));

    pIrp = pIrpEntry->pIrp;
    pIrpSp = pIrpEntry->pIrpSp;

    pCcb = pFileObject->FsContext2;
    ASSERT(pCcb != NULL);

    pCcb->UserContext = pEventInfo->Context;

    if (pIrp->MdlAddress)
        pBuffer = MmGetSystemAddressForMdlNormalSafe(pIrp->MdlAddress);
    else
        pBuffer = pIrp->UserBuffer;

    bufferLen = pIrpSp->Parameters.Read.Length;

    KdPrint(("[PTFS]::PTFSCompleteRead start bufferLen[%lu], event bufferLen[%lu]\n", bufferLen, pEventInfo->BufferLength));
 
    if (bufferLen == 0 || pBuffer == NULL || bufferLen < pEventInfo->BufferLength) 
    {
        KdPrint(("[PTFS]::PTFSCompleteRead returned STATUS_INSUFFICIENT_RESOURCES\n"));
        readLength = 0;
        status = STATUS_INSUFFICIENT_RESOURCES;
    }
    else 
    {
        RtlZeroMemory(pBuffer, bufferLen);
        RtlCopyMemory(pBuffer, pEventInfo->Buffer, pEventInfo->BufferLength);

        readLength = pEventInfo->BufferLength;
        status = pEventInfo->Status;

        if ( NT_SUCCESS(status) && pEventInfo->BufferLength > 0 &&
           (pFileObject->Flags & FO_SYNCHRONOUS_IO) && !(pIrp->Flags & IRP_PAGING_IO)) 
        {
            pFileObject->CurrentByteOffset.QuadPart = pEventInfo->Operation.Read.CurrentByteOffset.QuadPart;
            KdPrint(("[PTFS]::PTFSCompleteRead Updated CurrentByteOffset[%I64d]\n", pFileObject->CurrentByteOffset.QuadPart));
        }
    }

/*
    if (pIrpEntry->ulFlags & PTFS_MDL_ALLOCATED) 
    {
        FreeMdl(pIrp);
        pIrpEntry->ulFlags &= ~PTFS_MDL_ALLOCATED;
    }
*/

    PTFSCompleteIrpRequest(pIrp, status, readLength);
    KdPrint(("[PTFS]::PTFSCompleteRead end\n"));
}
