
#include "notification.h"
#include "util.h"
#include "init.h"
#include "timeout.h"
#include "irp_buffer_helper.h"
#include <Ntstrsafe.h>

VOID
SetCommonEventContext(
	IN PPTFSDCB pDcb,
	IN PEVENT_CONTEXT pEventContext,
	IN PIRP pIrp,
	IN OPTIONAL PPTFSCCB pCcb
)
{
	PIO_STACK_LOCATION pIrpSp = IoGetCurrentIrpStackLocation(pIrp);

	pIrpSp = IoGetCurrentIrpStackLocation(pIrp);

	pEventContext->MountId = pDcb->ulMountId;
	pEventContext->MajorFunction = pIrpSp->MajorFunction;
	pEventContext->MinorFunction = pIrpSp->MinorFunction;
	pEventContext->Flags = pIrpSp->Flags;

	if (pCcb)
		pEventContext->FileFlags = PTFSCCBFlagsGet(pCcb);

	pEventContext->ProcessId = IoGetRequestorProcessId(pIrp);
}

PEVENT_CONTEXT
AllocateEventContextRaw(
	IN ULONG EventContextLength
)
{
	ULONG driverContextLength = 0;
	PDRIVER_EVENT_CONTEXT pDriverEventContext = NULL;
	PEVENT_CONTEXT pEventContext = NULL;

	if (EventContextLength < sizeof(EVENT_CONTEXT) ||
		EventContextLength > MAXULONG - sizeof(DRIVER_EVENT_CONTEXT))
    {
        KdPrint(("[PTFS]::AllocateEventContextRaw invalid EventContextLength requested\n"));
        return NULL;
	}

	driverContextLength = EventContextLength - sizeof(EVENT_CONTEXT) + sizeof(DRIVER_EVENT_CONTEXT);

	pDriverEventContext = PTFSAllocateZero(driverContextLength);

    if (pDriverEventContext == NULL)
		return NULL;

	InitializeListHead(&pDriverEventContext->ListEntry);

	pEventContext = &pDriverEventContext->EventContext;
	pEventContext->Length = EventContextLength;

	return pEventContext;
}

PEVENT_CONTEXT
AllocateEventContext(
	IN PPTFSDCB pDcb,
	IN PIRP pIrp,
	IN ULONG EventContextLength,
	IN OPTIONAL PPTFSCCB pCcb
)
{
	PEVENT_CONTEXT pEventContext;
	pEventContext = AllocateEventContextRaw(EventContextLength);

	if (pEventContext == NULL)
		return NULL;

	SetCommonEventContext(pDcb, pEventContext, pIrp, pCcb);
	pEventContext->SerialNumber = InterlockedIncrement((LONG*)&pDcb->ulSerialNumber);

	return pEventContext;
}

VOID
FreeEventContext(
    IN PEVENT_CONTEXT pEventContext
)
{
    PDRIVER_EVENT_CONTEXT pDriverEventContext = CONTAINING_RECORD(pEventContext, DRIVER_EVENT_CONTEXT, EventContext);
    PTFSFree(pDriverEventContext);
}

VOID
EventNotification(
    IN PIRP_LIST pNotifyEventIrpList,
    IN PEVENT_CONTEXT pEventContext
)
{
    PDRIVER_EVENT_CONTEXT pDriverEventContext = CONTAINING_RECORD(pEventContext, DRIVER_EVENT_CONTEXT, EventContext);

    InitializeListHead(&pDriverEventContext->ListEntry);

    ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);

    ExInterlockedInsertTailList(&pNotifyEventIrpList->ListHead, &pDriverEventContext->ListEntry, &pNotifyEventIrpList->ListSpinLock);
    KeSetEvent(&pNotifyEventIrpList->NotEmptyEvent, IO_NO_INCREMENT, FALSE);
}

VOID
MoveIrpList(
	IN PIRP_LIST Source,
	OUT LIST_ENTRY* Dest
)
{
	PLIST_ENTRY listHead = NULL;
	PIRP_ENTRY pIrpEntry = NULL;
	PIRP pIrp = NULL;
	KIRQL oldIrql;

	InitializeListHead(Dest);

	ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);

	KeAcquireSpinLock(&Source->ListSpinLock, &oldIrql);

	while (!IsListEmpty(&Source->ListHead))
	{
		listHead = RemoveHeadList(&Source->ListHead);
		pIrpEntry = CONTAINING_RECORD(listHead, IRP_ENTRY, ListEntry);
		pIrp = pIrpEntry->pIrp;

		if (pIrp == NULL)
		{
			ASSERT(pIrpEntry->bCancelRoutineFreeMemory == FALSE);
			FreeIrpEntry(pIrpEntry);
			continue;
		}

		if (IoSetCancelRoutine(pIrp, NULL) == NULL)
		{
			InitializeListHead(&pIrpEntry->ListEntry);
			pIrpEntry->bCancelRoutineFreeMemory = TRUE;
			continue;
		}

		InsertTailList(Dest, &pIrpEntry->ListEntry);
	}

	KeClearEvent(&Source->NotEmptyEvent);
	KeReleaseSpinLock(&Source->ListSpinLock, oldIrql);
}

VOID
ReleasePendingIrp(
    IN PIRP_LIST PendingIrpList
)
{
    PLIST_ENTRY listHead = NULL;
    LIST_ENTRY completeList;
    PIRP_ENTRY pIrpEntry = NULL;
    PIRP pIrp = NULL;

    MoveIrpList(PendingIrpList, &completeList);

    while (!IsListEmpty(&completeList)) 
    {
        listHead = RemoveHeadList(&completeList);
        pIrpEntry = CONTAINING_RECORD(listHead, IRP_ENTRY, ListEntry);
        pIrp = pIrpEntry->pIrp;
        FreeIrpEntry(pIrpEntry);
        PTFSCompleteIrpRequest(pIrp, STATUS_CANCELLED, 0);
    }
}

VOID
ReleaseNotifyEvent(
    IN PIRP_LIST pNotifyEventIrpList
)
{
    PDRIVER_EVENT_CONTEXT pDriverEventContext = NULL;
    PLIST_ENTRY listHead = NULL;
    KIRQL oldIrql;

    ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);

    KeAcquireSpinLock(&pNotifyEventIrpList->ListSpinLock, &oldIrql);

    while (!IsListEmpty(&pNotifyEventIrpList->ListHead)) 
    {
        listHead = RemoveHeadList(&pNotifyEventIrpList->ListHead);
        pDriverEventContext =CONTAINING_RECORD(listHead, DRIVER_EVENT_CONTEXT, ListEntry);
        PTFSFree(pDriverEventContext);
        pDriverEventContext = NULL;
    }

    KeClearEvent(&pNotifyEventIrpList->NotEmptyEvent);
    KeReleaseSpinLock(&pNotifyEventIrpList->ListSpinLock, oldIrql);
}

VOID
RetryIrps(
    IN PIRP_LIST PendingRetryIrpList
)
{
    PLIST_ENTRY listHead = NULL;
    LIST_ENTRY retryList;
    PIRP_ENTRY pIrpEntry = NULL;
    PIRP pIrp = NULL;
    PDEVICE_OBJECT pDeviceObject = NULL;

    MoveIrpList(PendingRetryIrpList, &retryList);

    while (!IsListEmpty(&retryList)) 
    {
        listHead = RemoveHeadList(&retryList);
        pIrpEntry = CONTAINING_RECORD(listHead, IRP_ENTRY, ListEntry);
        pIrp = pIrpEntry->pIrp;
        pDeviceObject = pIrpEntry->pIrpSp->DeviceObject;
        FreeIrpEntry(pIrpEntry);
        PTFSDispatchRoutine(pDeviceObject, pIrp);
    }
}

VOID
NotificationLoop(
    IN PIRP_LIST PendingIrpList,
    IN PIRP_LIST pNotifyEventIrpList
)
{
    PDRIVER_EVENT_CONTEXT pDriverEventContext = NULL;
    PLIST_ENTRY listHead;
    PIRP_ENTRY pIrpEntry = NULL;
    LIST_ENTRY completeList;
    KIRQL irpIrql, notifyIrql;
    PIRP pIrp = NULL;
    PVOID pBuffer = NULL;
    ULONG eventLen = 0, bufferLen = 0;

    KdPrint(("[PTFS]::NotificationLoop Start\n"));

    InitializeListHead(&completeList);

    ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);
    KeAcquireSpinLock(&PendingIrpList->ListSpinLock, &irpIrql);
    KeAcquireSpinLock(&pNotifyEventIrpList->ListSpinLock, &notifyIrql);

    while (!IsListEmpty(&PendingIrpList->ListHead) && !IsListEmpty(&pNotifyEventIrpList->ListHead)) 
    {
        listHead = RemoveHeadList(&pNotifyEventIrpList->ListHead);
        pDriverEventContext = CONTAINING_RECORD(listHead, DRIVER_EVENT_CONTEXT, ListEntry);
        listHead = RemoveHeadList(&PendingIrpList->ListHead);
        pIrpEntry = CONTAINING_RECORD(listHead, IRP_ENTRY, ListEntry);

        eventLen = pDriverEventContext->EventContext.Length;
        pIrp = pIrpEntry->pIrp;

        if (pIrp == NULL) 
        {
            KdPrint(("[PTFS]::NotificationLoop Irp canceled\n"));
            ASSERT(pIrpEntry->bCancelRoutineFreeMemory == FALSE);
            FreeIrpEntry(pIrpEntry);
            InsertTailList(&pNotifyEventIrpList->ListHead, &pDriverEventContext->ListEntry);
            continue;
        }

        if (IoSetCancelRoutine(pIrp, NULL) == NULL) 
        {
            KdPrint(("[PTFS]::NotificationLoop IoSetCancelRoutine return Null\n"));
            InitializeListHead(&pIrpEntry->ListEntry);
            pIrpEntry->bCancelRoutineFreeMemory = TRUE;
            InsertTailList(&pNotifyEventIrpList->ListHead, &pDriverEventContext->ListEntry);
            continue;
        }

        bufferLen = pIrpEntry->pIrpSp->Parameters.DeviceIoControl.OutputBufferLength;
        pBuffer = pIrp->AssociatedIrp.SystemBuffer;

        if (bufferLen == 0 || pBuffer == NULL || bufferLen < eventLen) 
        {
            KdPrint(("[PTFS]::NotificationLoop STATUS_INSUFFICIENT_RESOURCES BufferLen[%d], EventLen[%d]\n", bufferLen, eventLen));
            InsertTailList(&pNotifyEventIrpList->ListHead, &pDriverEventContext->ListEntry);
            pIrpEntry->ulSerialNumber = 0;
        }
        else 
        {
            RtlCopyMemory(pBuffer, &pDriverEventContext->EventContext, eventLen);
            pIrpEntry->ulSerialNumber = eventLen;

            if (pDriverEventContext->pCompletedEvent)
                KeSetEvent(pDriverEventContext->pCompletedEvent, IO_NO_INCREMENT, FALSE);
            PTFSFree(pDriverEventContext);
        }

        InsertTailList(&completeList, &pIrpEntry->ListEntry);
    }

    KeClearEvent(&pNotifyEventIrpList->NotEmptyEvent);
    KeClearEvent(&PendingIrpList->NotEmptyEvent);

    KeReleaseSpinLock(&pNotifyEventIrpList->ListSpinLock, notifyIrql);
    KeReleaseSpinLock(&PendingIrpList->ListSpinLock, irpIrql);

    while (!IsListEmpty(&completeList)) 
    {
        listHead = RemoveHeadList(&completeList);
        pIrpEntry = CONTAINING_RECORD(listHead, IRP_ENTRY, ListEntry);
        pIrp = pIrpEntry->pIrp;

        if (pIrpEntry->ulSerialNumber == 0)
        {
            pIrp->IoStatus.Information = 0;
            pIrp->IoStatus.Status = STATUS_INSUFFICIENT_RESOURCES;
        }
        else 
        {
            pIrp->IoStatus.Information = pIrpEntry->ulSerialNumber;
            pIrp->IoStatus.Status = STATUS_SUCCESS;
        }

        FreeIrpEntry(pIrpEntry);
        PTFSCompleteIrpRequest(pIrp, pIrp->IoStatus.Status, pIrp->IoStatus.Information);
    }

    KdPrint(("[PTFS]::NotificationLoop End\n"));
}

VOID
NotificationThread(
    IN PVOID pDcbParam
)
{
    PKEVENT events[6];
    PKWAIT_BLOCK waitBlock;
    NTSTATUS status;
    PPTFSDCB pDcb = pDcbParam;

    KdPrint(("[PTFS]::NotificationThread Start\n"));

    waitBlock = PTFSAllocate(sizeof(KWAIT_BLOCK) * 6);
    if (waitBlock == NULL) 
    {
        KdPrint(("[PTFS]::NotificationThread Failed to Allocate\n"));
        return;
    }
    events[0] = &pDcb->ReleaseEvent;
    events[1] = &pDcb->NotifyEventIrpList.NotEmptyEvent;
    events[2] = &pDcb->PendingEventIrpList.NotEmptyEvent;
    events[3] = &pDcb->pPTFSGlobal->PendingServiceIrpList.NotEmptyEvent;
    events[4] = &pDcb->pPTFSGlobal->NotifyServiceIrpList.NotEmptyEvent;
    events[5] = &pDcb->PendingRetryIrpList.NotEmptyEvent;

    do 
    {
        status = KeWaitForMultipleObjects(6, events, WaitAny, Executive, KernelMode, FALSE, NULL, waitBlock);

        if (status != STATUS_WAIT_0) 
        {
            if (status == STATUS_WAIT_1 || status == STATUS_WAIT_2) 
                NotificationLoop(&pDcb->PendingEventIrpList, &pDcb->NotifyEventIrpList);
            else if (status == STATUS_WAIT_0 + 3 || status == STATUS_WAIT_0 + 4) 
                NotificationLoop(&pDcb->pPTFSGlobal->PendingServiceIrpList, &pDcb->pPTFSGlobal->NotifyServiceIrpList);
            else
                RetryIrps(&pDcb->PendingRetryIrpList);
        }

    } while (status != STATUS_WAIT_0);

    PTFSFree(waitBlock);
    KdPrint(("[PTFS]::NotificationThread End\n"));
}

NTSTATUS
StartEventNotificationThread(
    IN PPTFSDCB pDcb
)
{
    NTSTATUS status;
    HANDLE thread;

    KdPrint(("[PTFS]::StartEventNotificationThread Start\n"));

    KeResetEvent(&pDcb->ReleaseEvent);

    status = PsCreateSystemThread(&thread, THREAD_ALL_ACCESS, NULL, NULL, NULL, (PKSTART_ROUTINE)NotificationThread, pDcb);

    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::StartEventNotificationThread Failed to PsCreateSystemThread Status[0x%x]\n", status));
        return status;
    }

    ObReferenceObjectByHandle(thread, THREAD_ALL_ACCESS, NULL, KernelMode, (PVOID*)&pDcb->EventNotificationThread, NULL);

    ZwClose(thread);

    KdPrint(("[PTFS]::StartEventNotificationThread End\n"));

    return STATUS_SUCCESS;
}

VOID
StopEventNotificationThread(
    IN PPTFSDCB pDcb
)
{
    KdPrint(("[PTFS]::StopEventNotificationThread Start\n"));

    if (KeSetEvent(&pDcb->ReleaseEvent, 0, FALSE) > 0 && pDcb->EventNotificationThread) 
    {
        KdPrint(("[PTFS]::StopEventNotificationThread Waiting for Notify thread to terminate.\n"));
        ASSERT(KeGetCurrentIrql() <= APC_LEVEL);

        if (pDcb->EventNotificationThread) 
        {
            KeWaitForSingleObject(pDcb->EventNotificationThread, Executive, KernelMode, FALSE, NULL);
            KdPrint(("[PTFS]::StopEventNotificationThread Notify thread successfully terminated.\n"));
            ObDereferenceObject(pDcb->EventNotificationThread);
            pDcb->EventNotificationThread = NULL;
        }
    }

    KdPrint(("[PTFS]::StopEventNotificationThread End\n"));
}

VOID
CleanupAllChangeNotificationWaiters(
    IN PPTFSVCB pVcb
)
{
    PTFSVcbLock(pVcb, FALSE);
    KdPrint(("[PTFS]::CleanupAllChangeNotificationWaiters\n"));
    FsRtlNotifyCleanupAll(pVcb->NotifySync, &pVcb->DirNotifyList);
    PTFSVcbUnlock(pVcb);
}

VOID
StopFcbGarbageCollectorThread(
    IN PPTFSVCB pVcb
)
{
    if (pVcb->FcbGarbageCollectorThread != NULL) 
    {
        KeWaitForSingleObject(pVcb->FcbGarbageCollectorThread, Executive, KernelMode,FALSE, NULL);
        ObDereferenceObject(pVcb->FcbGarbageCollectorThread);
        pVcb->FcbGarbageCollectorThread = NULL;
    }
}

NTSTATUS
PTFSEventRelease(
    IN PDEVICE_OBJECT pDeviceObject,
    IN PIRP pIrp
)
{
    PPTFSDCB pDcb = NULL;
    PPTFSVCB pVcb = NULL;
    NTSTATUS status = STATUS_SUCCESS;

    if (pDeviceObject == NULL)
        return STATUS_INVALID_PARAMETER;

    KdPrint(("[PTFS]::PTFSEventRelease\n"));

    pVcb = pDeviceObject->DeviceExtension;
    if (GETIDENTIFIERTYPE(pVcb) != VCB) 
    {
        KdPrint(("[PTFS]::PTFSEventRelease wrong identifier type\n"));
        return STATUS_INVALID_PARAMETER;
    }
    
    pDcb = pVcb->pDcb;

    if (IsDeletePending(pDcb->pDeviceObject)) 
    {
        KdPrint(("[PTFS]::PTFSEventRelease Event release is already running for this device.\n"));
        return STATUS_SUCCESS;
    }

    if (IsUnmountPendingVcb(pVcb)) 
    {
        KdPrint(("[PTFS]::PTFSEventRelease Event release is already running for this volume.\n"));
        return STATUS_SUCCESS;
    }

    status = IoAcquireRemoveLock(&pDcb->RemoveLock, pIrp);
    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::PTFSEventRelease Failed to IoAcquireRemoveLock status[0x%x]\n", status));
        return STATUS_DEVICE_REMOVED;
    }

    DeleteMountPoint(pDcb);

    // then mark the device for unmount pending
    SetLongFlag(pVcb->ulFlags, VCB_DISMOUNT_PENDING);
    SetLongFlag(pDcb->ulFlags, DCB_DELETE_PENDING);

    KdPrint(("[PTFS]::PTFSEventRelease Starting unmount for device %wZ\n", pDcb->punstrDiskDeviceName));

    ReleasePendingIrp(&pDcb->PendingIrpList);
    ReleasePendingIrp(&pDcb->PendingEventIrpList);
    ReleasePendingIrp(&pDcb->PendingRetryIrpList);
    StopCheckThread(pDcb);
    StopEventNotificationThread(pDcb);

    StopFcbGarbageCollectorThread(pVcb);
    ClearLongFlag(pVcb->ulFlags, VCB_MOUNTED);

    CleanupAllChangeNotificationWaiters(pVcb);
    IoReleaseRemoveLockAndWait(&pDcb->RemoveLock, pIrp);

    DeleteDeviceObject(pDcb);

    KdPrint(("[PTFS]::PTFSEventRelease Finished event release.\n"));
    return status;
}

ULONG
GetCurrentSessionId(
    IN PIRP pIrp
)
{
    ULONG sessionNumber = 0;
    NTSTATUS status;

    status = IoGetRequestorSessionId(pIrp, &sessionNumber);
    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::GetCurrentSessionId Faile to IoGetRequestorSessionId status[0x%x]\n", status));
        return (ULONG)-1;
    }

    KdPrint(("[PTFS]::GetCurrentSessionId SessionId[%lu]\n", sessionNumber));
    return sessionNumber;
}

NTSTATUS
PTFSGlobalEventRelease(
    IN PDEVICE_OBJECT pDeviceObject,
    IN PIRP pIrp
)
{
    PPTFS_GLOBAL pPTFSGlobal = NULL;
    PUNICODE_STRING_INTERMEDIATE pszMountPoint = NULL;
    PTFS_CONTROL ptfsControl;
    PMOUNT_ENTRY pMountEntry = NULL;

    pPTFSGlobal = pDeviceObject->DeviceExtension;
    if (GETIDENTIFIERTYPE(pPTFSGlobal) != PGL) 
    {
        KdPrint(("[PTFS]::PTFSGlobalEventRelease Invalid Identi Type PGL\n"));
        return STATUS_INVALID_PARAMETER;
    }

    GET_IRP_UNICODE_STRING_INTERMEDIATE_OR_RETURN(pIrp, pszMountPoint)

    RtlZeroMemory(&ptfsControl, sizeof(PTFS_CONTROL));
    RtlStringCchCopyW(ptfsControl.MountPoint, MAXIMUM_FILENAME_LENGTH, L"\\DosDevices\\");
    
    if ((pszMountPoint->Length / sizeof(WCHAR)) < 4) 
    {
        ptfsControl.MountPoint[12] = towupper(pszMountPoint->Buffer[0]);
        ptfsControl.MountPoint[13] = L':';
        ptfsControl.MountPoint[14] = L'\0';
    }
    else 
    {
        if (pszMountPoint->Length > sizeof(ptfsControl.MountPoint) - 12 * sizeof(WCHAR)) 
        {
            KdPrint(("[PTFS]::PTFSGlobalEventRelease Mount point pBuffer has invalid size\n"));
            return STATUS_BUFFER_OVERFLOW;
        }

        RtlCopyMemory(&ptfsControl.MountPoint[12], pszMountPoint->Buffer, pszMountPoint->Length);
    }

    ptfsControl.SessionId = GetCurrentSessionId(pIrp);
    pMountEntry = FindMountEntry(pPTFSGlobal, &ptfsControl, TRUE);
    if (pMountEntry == NULL) 
    {
        ptfsControl.SessionId = (ULONG)-1;
        KdPrint(("[PTFS]::PTFSGlobalEventRelease Cannot found device associated to mount point %ws\n", ptfsControl.MountPoint));
        return STATUS_BUFFER_TOO_SMALL;
    }

    if (IsDeletePending(pMountEntry->MountControl.VolumeDeviceObject)) 
    {
        KdPrint(("[PTFS]::PTFSGlobalEventRelease Device is deleted\n"));
        return STATUS_DEVICE_REMOVED;
    }

    if (!IsMounted(pMountEntry->MountControl.VolumeDeviceObject)) 
    {
        KdPrint(("[PTFS]::PTFSGlobalEventRelease Device is still not mounted, so an unmount not possible at this point\n"));
        return STATUS_DEVICE_BUSY;
    }

    return PTFSEventRelease(pMountEntry->MountControl.VolumeDeviceObject, pIrp);
}


