
#include "mount.h"
#include "init.h"
#include "util.h"

NTSTATUS
SendIoContlToMountManager(
    IN ULONG IoControlCode,
    IN PVOID InputBuffer,
    IN ULONG Length,
    OUT PVOID OutputBuffer,
    IN ULONG OutputLength
)
{
    NTSTATUS status;
    UNICODE_STRING mountManagerName;
    PFILE_OBJECT pMountFileObject = NULL;
    PDEVICE_OBJECT pMountDeviceObject = NULL;
    PIRP pIrp = NULL;
    KEVENT driverEvent;
    IO_STATUS_BLOCK iosb;

    KdPrint(("[PTFS]::SendIoContlToMountManager Start\n"));

    RtlInitUnicodeString(&mountManagerName, MOUNTMGR_DEVICE_NAME);

    status = IoGetDeviceObjectPointer(&mountManagerName, FILE_READ_ATTRIBUTES, &pMountFileObject, &pMountDeviceObject);

    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::SendIoContlToMountManager failed to IoGetDeviceObjectPointer status[0x%x]\n", status));
        return status;
    }

    KeInitializeEvent(&driverEvent, NotificationEvent, FALSE);

    pIrp = IoBuildDeviceIoControlRequest(IoControlCode, pMountDeviceObject,
        InputBuffer, Length, OutputBuffer,
        OutputLength, FALSE, &driverEvent, &iosb);

    if (pIrp == NULL) 
    {
        KdPrint(("[PTFS]::SendIoContlToMountManager failed to IoBuildDeviceIoControlRequest\n"));
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    status = IoCallDriver(pMountDeviceObject, pIrp);

    if (status == STATUS_PENDING) 
    {
        KdPrint(("[PTFS]::SendIoContlToMountManager IoCallDriver pending\n"));
        KeWaitForSingleObject(&driverEvent, Executive, KernelMode, FALSE, NULL);
    }
    else
        KdPrint(("[PTFS]::SendIoContlToMountManager IoCallDriver status[0x%x]\n", status));


    status = iosb.Status;
    ObDereferenceObject(pMountFileObject);

    if (NT_SUCCESS(status)) 
        KdPrint(("[PTFS]::SendIoContlToMountManager successed to IoCallDriver\n"));
    else 
        KdPrint(("[PTFS]::SendIoContlToMountManager Failed to IoCallDriver status[0x%x]\n", status));

    return status;
}

NTSTATUS
SendVolumeMountPoint(
    IN PPTFSDCB pDcb,
    IN BOOLEAN Create
)
{
    NTSTATUS status;
    PMOUNTMGR_VOLUME_MOUNT_POINT pVolumMountPoint = NULL;
    ULONG length = 0;

    KdPrint(("[PTFS]::SendVolumeMountPoint Start\n"));

    length = sizeof(MOUNTMGR_VOLUME_MOUNT_POINT) + pDcb->punstrMountPoint->Length + pDcb->punstrPersistentSymbolicLinkName->Length;
    pVolumMountPoint = PTFSAllocateZero(length);

    if (pVolumMountPoint == NULL) 
    {
        KdPrint(("[PTFS]::SendVolumeMountPoint Failed to allocate\n"));
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    pVolumMountPoint->SourceVolumeNameOffset = sizeof(MOUNTMGR_VOLUME_MOUNT_POINT);
    pVolumMountPoint->SourceVolumeNameLength = pDcb->punstrMountPoint->Length;
    RtlCopyMemory((PCHAR)pVolumMountPoint + pVolumMountPoint->SourceVolumeNameOffset, pDcb->punstrMountPoint->Buffer, pVolumMountPoint->SourceVolumeNameLength);
    pVolumMountPoint->TargetVolumeNameOffset = pVolumMountPoint->SourceVolumeNameOffset + pVolumMountPoint->SourceVolumeNameLength;
    pVolumMountPoint->TargetVolumeNameLength = pDcb->punstrPersistentSymbolicLinkName->Length;
    RtlCopyMemory((PCHAR)pVolumMountPoint + pVolumMountPoint->TargetVolumeNameOffset, pDcb->punstrPersistentSymbolicLinkName->Buffer, pVolumMountPoint->TargetVolumeNameLength);

    status = SendIoContlToMountManager(Create ? IOCTL_MOUNTMGR_VOLUME_MOUNT_POINT_CREATED : IOCTL_MOUNTMGR_VOLUME_MOUNT_POINT_DELETED,
        pVolumMountPoint, length, NULL, 0);

    if (NT_SUCCESS(status)) 
        KdPrint(("[PTFS]::SendVolumeMountPoint successed to IoCallDriver(IOCTL_MOUNTMGR_VOLUME_MOUNT_POINT)\n"));
    else 
        KdPrint(("[PTFS]::SendVolumeMountPoint failed to IoCallDriver(IOCTL_MOUNTMGR_VOLUME_MOUNT_POINT) status[0x%x]\n", status));

    PTFSFree(pVolumMountPoint);
    return status;
}

NTSTATUS
QueryAutoMount(
    OUT BOOLEAN* State
)
{
    NTSTATUS status;
    MOUNTMGR_QUERY_AUTO_MOUNT queryAutoMount;

    KdPrint(("[PTFS]::QueryAutoMount Start\n"));

    status = SendIoContlToMountManager(IOCTL_MOUNTMGR_QUERY_AUTO_MOUNT, NULL, 0, &queryAutoMount, sizeof(queryAutoMount));
    if (NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::QueryAutoMount successed to IoCallDriver(IOCTL_MOUNTMGR_QUERY_AUTO_MOUNT) Current State[%d]\n", queryAutoMount.CurrentState));
        *State = queryAutoMount.CurrentState;
    }
    else
        KdPrint(("[PTFS]::QueryAutoMount failed to IoCallDriver(IOCTL_MOUNTMGR_QUERY_AUTO_MOUNT) status[0x%x]\n", status));

    return status;
}


NTSTATUS
SendAutoMount(
    IN BOOLEAN State
)
{
    NTSTATUS status;
    MOUNTMGR_SET_AUTO_MOUNT setAutoMount;

    setAutoMount.NewState = State;

    KdPrint(("[PTFS]::SendAutoMount Start State[%d]\n", setAutoMount.NewState));

    status = SendIoContlToMountManager(IOCTL_MOUNTMGR_SET_AUTO_MOUNT, &setAutoMount, sizeof(setAutoMount), NULL, 0);
    if (NT_SUCCESS(status)) 
        KdPrint(("[PTFS]::SendAutoMount successed to IoCallDriver(IOCTL_MOUNTMGR_SET_AUTO_MOUNT)\n"));
    else
        KdPrint(("[PTFS]::SendAutoMount failed to IoCallDriver(IOCTL_MOUNTMGR_SET_AUTO_MOUNT) status[0x%x]\n", status));

    return status;
}

NTSTATUS
SendVolumeArrivalNotification(
    IN PUNICODE_STRING punstrDeviceName
)
{
    NTSTATUS status;
    PMOUNTMGR_TARGET_NAME pTargetName = NULL;
    ULONG length = 0;

    KdPrint(("[PTFS]::SendVolumeArrivalNotification Start [%wZ]\n", punstrDeviceName));

    length = sizeof(MOUNTMGR_TARGET_NAME) + punstrDeviceName->Length - 1;
    pTargetName = PTFSAllocateZero(length);

    if (pTargetName == NULL) 
    {
        KdPrint(("[PTFS]::SendVolumeArrivalNotification failed to allocate\n"));
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    pTargetName->DeviceNameLength = punstrDeviceName->Length;
    RtlCopyMemory(pTargetName->DeviceName, punstrDeviceName->Buffer, punstrDeviceName->Length);

    status = SendIoContlToMountManager(IOCTL_MOUNTMGR_VOLUME_ARRIVAL_NOTIFICATION, pTargetName, length, NULL, 0);

    if (NT_SUCCESS(status)) 
        KdPrint(("[PTFS]::SendVolumeArrivalNotification successed to IoCallDriver(IOCTL_MOUNTMGR_VOLUME_ARRIVAL_NOTIFICATION)\n"));
    else 
        KdPrint(("[PTFS]::SendVolumeArrivalNotification failed to IoCallDriver(IOCTL_MOUNTMGR_VOLUME_ARRIVAL_NOTIFICATION) status[0x%x]\n", status));

    PTFSFree(pTargetName);
    return status;
}

VOID
NotifyDirectoryMountPointCreated(
    IN PPTFSDCB pDcb
)
{
    KdPrint(("[PTFS]::NotifyDirectoryMountPointCreated Start\n"));
    SendVolumeMountPoint(pDcb, /*Create*/ TRUE);
    KdPrint(("[PTFS]::NotifyDirectoryMountPointCreated End\n"));
}

VOID
NotifyDirectoryMountPointDeleted(
    IN PPTFSDCB pDcb
)
{
    KdPrint(("[PTFS]::NotifyDirectoryMountPointDeleted Start\n"));
    SendVolumeMountPoint(pDcb, /*Create*/ FALSE);
    KdPrint(("[PTFS]::NotifyDirectoryMountPointDeleted End\n"));
}

NTSTATUS
SendVolumeCreatePoint(
    IN PDRIVER_OBJECT pDriverObject,
    IN PUNICODE_STRING punstrDeviceName,
    IN PUNICODE_STRING punstrMountPoint
)
{
    UNREFERENCED_PARAMETER(pDriverObject);

    NTSTATUS status;
    PMOUNTMGR_CREATE_POINT_INPUT pCreatePoint = NULL;
    ULONG length = 0;

    KdPrint(("[PTFS]::SendVolumeCreatePoint Start CreatePoint[%wZ], device name[%wZ]\n", punstrMountPoint, punstrDeviceName));

    length = sizeof(MOUNTMGR_CREATE_POINT_INPUT) + punstrMountPoint->Length + punstrDeviceName->Length;
    pCreatePoint = PTFSAllocateZero(length);
    if (pCreatePoint == NULL) 
    {
        KdPrint(("[PTFS]::SendVolumeCreatePoint failed to allocate\n"));
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    pCreatePoint->DeviceNameOffset = sizeof(MOUNTMGR_CREATE_POINT_INPUT);
    pCreatePoint->DeviceNameLength = punstrDeviceName->Length;
    RtlCopyMemory((PCHAR)pCreatePoint + pCreatePoint->DeviceNameOffset, punstrDeviceName->Buffer, punstrDeviceName->Length);

    pCreatePoint->SymbolicLinkNameOffset = pCreatePoint->DeviceNameOffset + pCreatePoint->DeviceNameLength;
    pCreatePoint->SymbolicLinkNameLength = punstrMountPoint->Length;
    RtlCopyMemory((PCHAR)pCreatePoint + pCreatePoint->SymbolicLinkNameOffset, punstrMountPoint->Buffer, punstrMountPoint->Length);

    status = SendIoContlToMountManager(IOCTL_MOUNTMGR_CREATE_POINT, pCreatePoint, length, NULL, 0);

    if (NT_SUCCESS(status)) 
        KdPrint(("[PTFS]::SendVolumeCreatePoint successed to IoCallDriver(IOCTL_MOUNTMGR_CREATE_POINT)\n"));
    else
        KdPrint(("[PTFS]::SendVolumeCreatePoint failed to IoCallDriver(IOCTL_MOUNTMGR_CREATE_POINT) status[0x%x]\n", status));

    PTFSFree(pCreatePoint);
    return status;
}

NTSTATUS
SendVolumeDeletePoints(
    IN PUNICODE_STRING punstrMountPoint,
    IN PUNICODE_STRING punstrDeviceName
)
{
    NTSTATUS status;
    PMOUNTMGR_MOUNT_POINT pCreatePoint = NULL;
    PMOUNTMGR_MOUNT_POINTS pDeletedPoints = NULL;
    ULONG length = 0, olength = 0;
    ULONG nextStringOffset = 0;

    KdPrint(("[PTFS]::SendVolumeDeletePoints Start\n"));

    length = sizeof(MOUNTMGR_MOUNT_POINT);
    if (punstrMountPoint != NULL)
        length += punstrMountPoint->Length;
    if (punstrDeviceName != NULL)
        length += punstrDeviceName->Length;

    pCreatePoint = PTFSAllocateZero(length);
    if (pCreatePoint == NULL) 
    {
        KdPrint(("[PTFS]::SendVolumeDeletePoints Failed to allocate\n"));
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    olength = sizeof(MOUNTMGR_MOUNT_POINTS) + 1024;
    pDeletedPoints = PTFSAllocateZero(olength);
    if (pDeletedPoints == NULL) 
    {
        KdPrint(("[PTFS]::SendVolumeDeletePoints Failed to allocate(2)\n"));
        PTFSFree(pCreatePoint);
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    nextStringOffset = sizeof(MOUNTMGR_MOUNT_POINT);
    if (punstrMountPoint != NULL) 
    {
        KdPrint(("[PTFS]::SendVolumeDeletePoints MountPoint[%wZ]\n", punstrMountPoint));
        pCreatePoint->SymbolicLinkNameOffset = nextStringOffset;
        pCreatePoint->SymbolicLinkNameLength = punstrMountPoint->Length;
        nextStringOffset = pCreatePoint->SymbolicLinkNameOffset + pCreatePoint->SymbolicLinkNameLength;
        RtlCopyMemory((PCHAR)pCreatePoint + pCreatePoint->SymbolicLinkNameOffset, punstrMountPoint->Buffer, punstrMountPoint->Length);
    }

    if (punstrDeviceName != NULL) 
    {
        KdPrint(("[PTFS]::SendVolumeDeletePoints DeviceName[%wZ]\n", punstrDeviceName));
        pCreatePoint->DeviceNameOffset = nextStringOffset;
        pCreatePoint->DeviceNameLength = punstrDeviceName->Length;
        RtlCopyMemory((PCHAR)pCreatePoint + pCreatePoint->DeviceNameOffset, punstrDeviceName->Buffer, punstrDeviceName->Length);
    }

    status = SendIoContlToMountManager(IOCTL_MOUNTMGR_DELETE_POINTS, pCreatePoint, length, pDeletedPoints, olength);

    if (NT_SUCCESS(status))
        KdPrint(("[PTFS]::SendVolumeDeletePoints successed to IoCallDriver(IOCTL_MOUNTMGR_DELETE_POINTS) Count[%d]\n", pDeletedPoints->NumberOfMountPoints));
    else
        KdPrint(("[PTFS]::SendVolumeDeletePoints failed to IoCallDriver(IOCTL_MOUNTMGR_DELETE_POINTS) status[0x%x]\n", status));

    PTFSFree(pCreatePoint);
    PTFSFree(pDeletedPoints);
    return status;
}

