#include "PTFileSystem.h"
#include "init.h"
#include "util.h"
#include "mount.h"
#include "Fscontrol.h"
#include <Mountmgr.h>
#include <Ntddstor.h>
#include <Wdm.h>
#include <Wdmsec.h>
#include <ntstrsafe.h>

UNICODE_STRING g_sddl = RTL_CONSTANT_STRING(L"D:P(A;;GA;;;SY)(A;;GRGWGX;;;BA)(A;;GRGWGX;;;WD)(A;;GRGX;;;RC)");

typedef struct _st_DEVOBJ_EXTENSION {

    CSHORT          Type;
    USHORT          Size;

    PDEVICE_OBJECT  pDeviceObject;

    ULONG           PowerFlags;
    struct          _DEVICE_OBJECT_POWER_EXTENSION* Dope;
    ULONG ExtensionFlags;
    PVOID           DeviceNode;
    PDEVICE_OBJECT  AttachedTo;
    __volatile LONG StartIoCount;
    LONG           StartIoKey;
    ULONG          StartIoFlags;
    PVPB           Vpb;
    PVOID DependencyNode;
    PVOID InterruptContext;

    __volatile PVOID VerifierContext;

} UND_DEVOBJ_EXTENSION, * PUND_DEVOBJ_EXTENSION;


VOID 
InitializeIrpList(
    IN PIRP_LIST pIrpList
) 
{
    InitializeListHead(&pIrpList->ListHead);
    KeInitializeSpinLock(&pIrpList->ListSpinLock);
    KeInitializeEvent(&pIrpList->NotEmptyEvent, NotificationEvent, FALSE);
}

VOID 
FreeDcbNames(
    IN PPTFSDCB pDcb
) 
{
    if (pDcb->punstrMountPoint != NULL) 
    {
        PTFSFreeUnicodeString(pDcb->punstrMountPoint);
        pDcb->punstrMountPoint = NULL;
    }

    if (pDcb->punstrSymbolicLinkName != NULL) 
    {
        PTFSFreeUnicodeString(pDcb->punstrSymbolicLinkName);
        pDcb->punstrSymbolicLinkName = NULL;
    }

    if (pDcb->punstrDiskDeviceName != NULL) 
    {
        PTFSFreeUnicodeString(pDcb->punstrDiskDeviceName);
        pDcb->punstrDiskDeviceName = NULL;
    }

    if (pDcb->punstrUNCName != NULL) 
    {
        PTFSFreeUnicodeString(pDcb->punstrUNCName);
        pDcb->punstrUNCName = NULL;
    }

    if (pDcb->punstrPersistentSymbolicLinkName != NULL) 
    {
        PTFSFreeUnicodeString(pDcb->punstrPersistentSymbolicLinkName);
        pDcb->punstrPersistentSymbolicLinkName = NULL;
    }
}

BOOLEAN 
IsMounted(
    IN PDEVICE_OBJECT pDeviceObject
) 
{
    PPTFSVCB pVcb = NULL;
    PPTFSDCB pDcb = NULL;

    if (pDeviceObject == NULL)
        return FALSE;

    pDcb = pDeviceObject->DeviceExtension;

    if (GETIDENTIFIERTYPE(pDcb) == DCB)
        pVcb = pDcb->pVcb;
    else 
        pVcb = pDeviceObject->DeviceExtension;

    if (pVcb == NULL)
        return FALSE;

    if (GETIDENTIFIERTYPE(pVcb) == VCB) 
    {
        pDcb = pVcb->pDcb;

        if (IsFlagOn(pVcb->ulFlags, VCB_MOUNTED)) 
        {
            if (pDcb->pDeviceObject->Vpb) 
                return IsFlagOn(pDcb->pDeviceObject->Vpb->Flags, VPB_MOUNTED);

            return TRUE;
        }
    }

    return FALSE;
}

BOOLEAN 
IsDeletePending (
    IN PDEVICE_OBJECT pDeviceObject
) 
{
    PPTFSDCB pDcb = NULL;

    if (pDeviceObject == NULL)
        return TRUE;
  
    if (pDeviceObject->Vpb == NULL && pDeviceObject->DeviceExtension == NULL && pDeviceObject->Characteristics == 0) 
        return TRUE;

    pDcb = pDeviceObject->DeviceExtension;

    if (pDcb == NULL)
        return TRUE;

    if (GETIDENTIFIERTYPE(pDcb) == DCB) 
    {
        if (IsFlagOn(pDcb->ulFlags, DCB_DELETE_PENDING)) 
            return TRUE;
    }

    if (pDeviceObject->DeviceObjectExtension)
    {
        PUND_DEVOBJ_EXTENSION pDevExtension = (PUND_DEVOBJ_EXTENSION)pDeviceObject->DeviceObjectExtension;
        if(pDevExtension->ExtensionFlags & (DOE_UNLOAD_PENDING | DOE_DELETE_PENDING | DOE_REMOVE_PENDING | DOE_REMOVE_PROCESSED))
                return TRUE;
    }

    return FALSE;
}

BOOLEAN 
IsUnmountPendingVcb(
    IN PPTFSVCB pVcb
) 
{
    if (pVcb == NULL)
        return FALSE;
   
    if (GETIDENTIFIERTYPE(pVcb) == VCB)
    {
        if (IsFlagOn(pVcb->ulFlags, VCB_DISMOUNT_PENDING)) 
            return TRUE;

        return IsDeletePending(pVcb->pDcb->pDeviceObject);
    }

    return FALSE;
}

BOOLEAN
IsUnmountPending(
    IN PDEVICE_OBJECT pDeviceObject
)
{
    PPTFSVCB pVcb = NULL;
    PPTFSDCB pDcb = NULL;

    if (pDeviceObject == NULL)
        return FALSE;

    pDcb = pDeviceObject->DeviceExtension;

    if (pDcb == NULL)
        return FALSE;

    if (GETIDENTIFIERTYPE(pDcb) == DCB)
        pVcb = pDcb->pVcb;
    else
        pVcb = pDeviceObject->DeviceExtension;

    return IsDeletePending(pDeviceObject) || IsUnmountPendingVcb(pVcb);
}

VOID 
DeleteDeviceDelayed(
    IN PPTFS_GLOBAL pPTFSGlobal
    )
{
    PDEVICE_ENTRY pDeviceEntry = NULL;
    LIST_ENTRY completeList;
    PLIST_ENTRY listHead = &pPTFSGlobal->DeviceDeleteList;
    PLIST_ENTRY entry;
    PLIST_ENTRY nextEntry;
    ULONG ulTotalCount = 0;
    PPTFSDCB pDcb = NULL;
    NTSTATUS status;

    InitializeListHead(&completeList);

    if (!ExAcquireResourceExclusiveLite(&pPTFSGlobal->Resource, FALSE)) 
    {
        KdPrint(("[PTFS]::DeleteDeviceDelayed Failed ExAcquireResourceExclusiveLite(Global Resource)\n"));
        return;
    }

    if (IsListEmpty(&pPTFSGlobal->DeviceDeleteList)) 
    {
        ExReleaseResourceLite(&pPTFSGlobal->Resource);
        return;
    }

    for (entry = listHead->Flink, nextEntry = entry->Flink; entry != listHead; entry = nextEntry, nextEntry = entry->Flink, ulTotalCount++) 
    {
        pDeviceEntry = CONTAINING_RECORD(entry, DEVICE_ENTRY, ListEntry);
        if (pDeviceEntry) 
        {
            BOOLEAN bCanDeleteDiskDevice = FALSE;
        
            KdPrint(("[PTFS]::DeleteDeviceDelayed delayed delete Counter %d\n", pDeviceEntry->ulCounter));

            pDcb = NULL;
            if (pDeviceEntry->pDiskDeviceObject) 
            {
                bCanDeleteDiskDevice = pDeviceEntry->pDiskDeviceObject->ReferenceCount == 0;
                pDcb = pDeviceEntry->pDiskDeviceObject->DeviceExtension;
                InterlockedIncrement((LONG*)&pDeviceEntry->ulCounter);
                KdPrint(("[PTFS]::DeleteDeviceDelayed DiskDevice ReferenceCount[%d]\n", pDeviceEntry->pDiskDeviceObject->ReferenceCount));
            }

			if (pDeviceEntry->pVolumeDeviceObject)
			{
				KdPrint(("[PTFS]::DeleteDeviceDelayed limit Count %d\n", pDeviceEntry->pVolumeDeviceObject->ReferenceCount));
				if (bCanDeleteDiskDevice)
				{
					KdPrint(("[PTFS]::DeleteDeviceDelayed Delete Symbolic Name[%wZ]\n", pDcb->punstrSymbolicLinkName));
					status = IoDeleteSymbolicLink(pDcb->punstrSymbolicLinkName);

					if (!NT_SUCCESS(status))
						KdPrint(("[PTFS]::DeleteDeviceDelayed Failed IoDeleteSymbolicLink status[0x%x]\n", status));

					FreeDcbNames(pDcb);

					KdPrint(("[PTFS]::DeleteDeviceDelayed Delete volume device ReferenceCount[%d]\n", pDeviceEntry->pVolumeDeviceObject->ReferenceCount));
					IoDeleteDevice(pDeviceEntry->pVolumeDeviceObject);
					pDeviceEntry->pVolumeDeviceObject = NULL;
				}
				else
				{
					KdPrint(("[PTFS]::DeleteDeviceDelayed Disk device still references Refcount[%d]\n", pDeviceEntry->pDiskDeviceObject->ReferenceCount));
				}
			}

			if (pDeviceEntry->pDiskDeviceObject && bCanDeleteDiskDevice)
			{
				KdPrint(("[PTFS]::DeleteDeviceDelayed  delete the disk device Refcount[%d]\n", pDeviceEntry->pDiskDeviceObject->ReferenceCount));
				IoDeleteDevice(pDeviceEntry->pDiskDeviceObject);
				if (pDeviceEntry->pDiskDeviceObject->Vpb)
				{
					KdPrint(("[PTFS]::DeleteDeviceDelayed  volume deviceObject NULL\n"));
					pDeviceEntry->pDiskDeviceObject->Vpb->DeviceObject = NULL;
				}
				pDeviceEntry->pDiskDeviceObject = NULL;
			}

            if (pDeviceEntry->pVolumeDeviceObject == NULL && pDeviceEntry->pDiskDeviceObject == NULL)
            {
                if (pDeviceEntry->ulSessionId == -1) 
                {
                    RemoveEntryList(&pDeviceEntry->ListEntry);
                    InitializeListHead(&pDeviceEntry->ListEntry);
                }
                else {
                    KdPrint(("[PTFS]::DeleteDeviceDelayed  sessionId\n"));
                }
            }
        }
    }

    KdPrint(("[PTFS]::DeleteDeviceDelayed  Total devices to delete  Count[%d]\n", ulTotalCount));
    ExReleaseResourceLite(&pPTFSGlobal->Resource);
}

VOID 
PTFSDeleteDeviceThread(
    IN PVOID pThreadParam
)
{
    NTSTATUS status;
    KTIMER      timer;
    PVOID           pollevents[3];
    LARGE_INTEGER timeout = { 0 };
    BOOLEAN           bWaitObj = TRUE;
    PPTFS_GLOBAL pPTFSGlobal = (PPTFS_GLOBAL)pThreadParam;

    KdPrint(("[PTFS]::PTFSDeleteDeviceThread Begin\n"));

    KeInitializeTimerEx(&timer, SynchronizationTimer);

    pollevents[0] = (PVOID)&pPTFSGlobal->KillDeleteDeviceEvent;
    pollevents[1] = (PVOID)&timer;
    pollevents[2] = (PVOID)&pPTFSGlobal->DeleteDeviceEvent;

    KeSetTimerEx(&timer, timeout, PTFS_DELETE_DEVICE_CHECK_INTERVAL, NULL);

    while (bWaitObj)
    {
        status = KeWaitForMultipleObjects(3, pollevents, WaitAny, Executive,
            KernelMode, FALSE, NULL, NULL);

        if (!NT_SUCCESS(status) || status == STATUS_WAIT_0) 
        {
            KdPrint(("[PTFS]::PTFSDeleteDeviceThread catced KillEvent\n"));
            bWaitObj = FALSE;
        }
        else 
            DeleteDeviceDelayed(pPTFSGlobal);
    }

    KeCancelTimer(&timer);
    KdPrint(("[PTFS]::PTFSDeleteDeviceThread End\n"));
    PsTerminateSystemThread(STATUS_SUCCESS);
}

NTSTATUS
StartDeleteDeviceThread (
    IN PPTFS_GLOBAL pPTFSGlobal
)
{
    NTSTATUS    status;
    HANDLE         hThread = NULL;

    KdPrint(("[PTFS]::StartDeleteDeviceThread Begin\n"));

    status = PsCreateSystemThread(&hThread, THREAD_ALL_ACCESS, NULL, NULL, NULL,
        (PKSTART_ROUTINE)PTFSDeleteDeviceThread,
        pPTFSGlobal);

    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::StartDeleteDeviceThread Failed PsCreateSystemThread status[0x%x]\n", status));
        return status;
    }

    ObReferenceObjectByHandle(
        hThread, THREAD_ALL_ACCESS, NULL, 
        KernelMode,
        (PVOID*)&pPTFSGlobal->DeviceDeleteThread, 
        NULL);

    ZwClose(hThread);

    KdPrint(("[PTFS]::StartDeleteDeviceThread End\n"));

    return STATUS_SUCCESS;
}

NTSTATUS
PTFSGetMountPointList(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp,
	IN PPTFS_GLOBAL pPTFSGlobal
)
{
	PIO_STACK_LOCATION pIrpSp = NULL;
	NTSTATUS status = STATUS_INVALID_PARAMETER;
	PLIST_ENTRY listEntry = NULL;
	PMOUNT_ENTRY pMountEntry = NULL;
	PPTFS_CONTROL pPTFSControl = NULL;
	int i = 0;

	UNREFERENCED_PARAMETER(pDeviceObject);

	KdPrint(("[PTFS]::PTFSGetMountPointList Start\n"));

	pIrpSp = IoGetCurrentIrpStackLocation(pIrp);

	try
	{
		ExAcquireResourceExclusiveLite(&pPTFSGlobal->Resource, TRUE);
		pIrp->IoStatus.Information = 0;
		pPTFSControl = (PPTFS_CONTROL)pIrp->AssociatedIrp.SystemBuffer;

		for (listEntry = pPTFSGlobal->MountPointList.Flink; listEntry != &pPTFSGlobal->MountPointList; listEntry = listEntry->Flink, ++i)
		{
			if (pIrpSp->Parameters.DeviceIoControl.OutputBufferLength < (sizeof(PTFS_CONTROL) * (i + 1)))
			{
				status = STATUS_BUFFER_OVERFLOW;
				__leave;
			}

			pIrp->IoStatus.Information = sizeof(PTFS_CONTROL) * (i + 1);
			pMountEntry = CONTAINING_RECORD(listEntry, MOUNT_ENTRY, ListEntry);
			RtlCopyMemory(&pPTFSControl[i], &pMountEntry->MountControl, sizeof(PTFS_CONTROL));
//			pPTFSControl[i].VolumeDeviceObject = NULL;
		}
		status = STATUS_SUCCESS;
	}
	finally
	{
		ExReleaseResourceLite(&pPTFSGlobal->Resource);
	}

	KdPrint(("[PTFS]::PTFSGetMountPointList End\n"));
	return status;
}

NTSTATUS
PTFSCreateGlobalDiskDevice(
    IN PDRIVER_OBJECT pDriverObject,
    OUT PPTFS_GLOBAL* ppPTFSGlobal
)
{
    NTSTATUS status;
    UNICODE_STRING unstrDeviceName;
    UNICODE_STRING unstrSymbolicLinkName;
    UNICODE_STRING unstrFsDiskDeviceName;
    PDEVICE_OBJECT pDeviceObject = NULL;
    PDEVICE_OBJECT pFsDiskDeviceObject = NULL;
    PPTFS_GLOBAL     pPTFSGlobal = NULL;

    RtlInitUnicodeString(&unstrDeviceName, PTFS_GLOBAL_DEVICE_NAME);
    RtlInitUnicodeString(&unstrSymbolicLinkName, PTFS_GLOBAL_SYMBOLIC_LINK_NAME);
    RtlInitUnicodeString(&unstrFsDiskDeviceName, PTFS_GLOBAL_FS_DISK_DEVICE_NAME);

    status = IoCreateDeviceSecure(
        pDriverObject,
        sizeof(PTFS_GLOBAL),
        &unstrDeviceName,
        FILE_DEVICE_UNKNOWN,
        0,
        FALSE,
        &g_sddl,
        NULL,
        &pDeviceObject);

    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::PTFSCreateGlobalDiskDevice Failed IoCreateDeviceSecure status[0x%x]\n", status));
        return status;
    }

    KdPrint(("[PTFS]::PTFSCreateGlobalDiskDevice Create Device name[%wZ]\n", unstrDeviceName));

    status = IoCreateDeviceSecure(
        pDriverObject,
        0,
        &unstrFsDiskDeviceName,
        FILE_DEVICE_DISK_FILE_SYSTEM,
        0,
        FALSE,
        &g_sddl,
        NULL,
        &pFsDiskDeviceObject);

    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::PTFSCreateGlobalDiskDevice Failed IoCreateDeviceSecure(Disk File System) status[0x%x]\n", status));
        IoDeleteDevice(pDeviceObject);
        return status;
    }

    KdPrint(("[PTFS]::PTFSCreateGlobalDiskDevice Create File system disk Device name[%wZ]\n", unstrFsDiskDeviceName));

    ObReferenceObject(pDeviceObject);

    status = IoCreateSymbolicLink(&unstrSymbolicLinkName, &unstrDeviceName);

    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::PTFSCreateGlobalDiskDevice Failed IoCreateSymbolicLink status[0x%x]\n", status));
        IoDeleteDevice(pFsDiskDeviceObject);
        IoDeleteDevice(pDeviceObject);
        return status;
    }

    KdPrint(("[PTFS]::PTFSCreateGlobalDiskDevice Symbolic Link [%wZ -> %wZ]\n", &unstrDeviceName, &unstrSymbolicLinkName));

    pPTFSGlobal = (PPTFS_GLOBAL)pDeviceObject->DeviceExtension;
    pPTFSGlobal->pDeviceObject = pDeviceObject;
    pPTFSGlobal->pFsDiskDeviceObject = pFsDiskDeviceObject;
    pPTFSGlobal->ulMountId = 0;

    InitializeIrpList(&pPTFSGlobal->PendingServiceIrpList);
    InitializeIrpList(&pPTFSGlobal->NotifyServiceIrpList);
    InitializeListHead(&pPTFSGlobal->MountPointList);
    InitializeListHead(&pPTFSGlobal->DeviceDeleteList);
    ExInitializeResourceLite(&pPTFSGlobal->Resource);
    ExInitializeResourceLite(&pPTFSGlobal->MountManagerLock);

    pPTFSGlobal->Identifier.FsdIdType = PGL;
    pPTFSGlobal->Identifier.ulSize = sizeof(PTFS_GLOBAL);

    KeInitializeEvent(&pPTFSGlobal->KillDeleteDeviceEvent, NotificationEvent, FALSE);
    KeInitializeEvent(&pPTFSGlobal->DeleteDeviceEvent, SynchronizationEvent, FALSE);

    StartDeleteDeviceThread(pPTFSGlobal);

    SetFlag(pFsDiskDeviceObject->Flags, DO_DIRECT_IO);
    SetFlag(pFsDiskDeviceObject->Flags, DO_LOW_PRIORITY_FILESYSTEM);

    ClearFlag(pFsDiskDeviceObject->Flags, DO_DEVICE_INITIALIZING);

    IoRegisterFileSystem(pFsDiskDeviceObject);

    ObReferenceObject(pFsDiskDeviceObject);

    *ppPTFSGlobal = pPTFSGlobal;

    return STATUS_SUCCESS;
}

VOID 
RegisterUncProvider(
    IN PVOID pDcbParam
)
{
    NTSTATUS status;
    PPTFSDCB pDcb = pDcbParam;

    if (pDcb->punstrUNCName != NULL && pDcb->punstrUNCName->Length > 0) 
    {
        status = FsRtlRegisterUncProvider(&(pDcb->hMupHandle), pDcb->punstrDiskDeviceName, FALSE);

        if (NT_SUCCESS(status)) 
            KdPrint(("[PTFS]::RegisterUncProvider successed to FsRtlRegisterUncProvider\n"));
        else 
        {
            KdPrint(("[PTFS]::RegisterUncProvider Failed to FsRtlRegisterUncProvider status[0x%x]\n", status));
            pDcb->hMupHandle = 0;
        }
    }
    PsTerminateSystemThread(STATUS_SUCCESS);
}

VOID 
DeregisterUncProvider(
    IN PVOID pDcbParam
)
{
    PPTFSDCB pDcb = pDcbParam;
    if (pDcb->hMupHandle) 
    {
        FsRtlDeregisterUncProvider(pDcb->hMupHandle);
        pDcb->hMupHandle = 0;
    }
    PsTerminateSystemThread(STATUS_SUCCESS);
}

VOID 
CreateMountPointSysProc(
    IN PVOID pDcbParam
) 
{
    NTSTATUS status;
    PPTFSDCB pDcb = pDcbParam;

    KdPrint(("[PTFS]::CreateMountPointSysProc start\n"));

    if (IsUnmountPendingVcb(pDcb->pVcb)) 
    {
        KdPrint(("[PTFS]::CreateMountPointSysProc Device was in meantime deleted\n"));
        return;
    }
    status = IoCreateSymbolicLink(pDcb->punstrMountPoint, pDcb->punstrDiskDeviceName);

    if (!NT_SUCCESS(status)) 
        KdPrint(("[PTFS]::CreateMountPointSysProc Failed to IoCreateSymbolicLink  mount point[%wZ] status[0x%x] \n", pDcb->punstrMountPoint, status));

    KdPrint(("[PTFS]::CreateMountPointSysProc end\n"));
}


VOID 
PTFSCreateMountPoint(
    IN PPTFSDCB pDcb
)
{
	if (pDcb->punstrMountPoint != NULL && pDcb->punstrMountPoint->Length > 0) 
    {
		if (pDcb->usUseMountManager) 
			SendVolumeCreatePoint(pDcb->pDriverObject, pDcb->punstrDiskDeviceName, pDcb->punstrMountPoint);
		else 
        {
            KdPrint(("[PTFS]::PTFSCreateMountPoint Not using Mount Manager.\n"));
/*
			if (pDcb->usMountGlobally)
				RunAsSystem(CreateMountPointSysProc, pDcb);
			else 
*/
				CreateMountPointSysProc(pDcb);
		}
	}
	else 
        KdPrint(("[PTFS]::PTFSCreateMountPoint Mount point string is empty.\n"));
}

PDEVICE_ENTRY 
FindDeviceForDeleteBySessionId(
    PPTFS_GLOBAL pPTFSGlobal,
	ULONG sessionId) 
{
	PLIST_ENTRY listHead = &pPTFSGlobal->DeviceDeleteList;
	PLIST_ENTRY entry;
	PLIST_ENTRY nextEntry;

	if (sessionId == -1)
		return NULL;

	if (!ExAcquireResourceExclusiveLite(&pPTFSGlobal->Resource, TRUE)) 
    {
        KdPrint(("[PTFS]::FindDeviceForDeleteBySessionId Failed to acquire Global Resource\n"));
        return NULL;
	}

	if (IsListEmpty(&pPTFSGlobal->DeviceDeleteList)) 
    {
		ExReleaseResourceLite(&pPTFSGlobal->Resource);
		return NULL;
	}

	for (entry = listHead->Flink, nextEntry = entry->Flink; entry != listHead; entry = nextEntry, nextEntry = entry->Flink) 
    {
		PDEVICE_ENTRY pDeviceEntry = CONTAINING_RECORD(entry, DEVICE_ENTRY, ListEntry);

		if (pDeviceEntry) 
        {
			if (pDeviceEntry->ulSessionId == sessionId) 
            {
				ExReleaseResourceLite(&pPTFSGlobal->Resource);
                KdPrint(("[PTFS]::FindDeviceForDeleteBySessionId found for a specific session Device\n"));
                return pDeviceEntry;
			}
		}
	}

	ExReleaseResourceLite(&pPTFSGlobal->Resource);
	return NULL;
}

BOOLEAN 
DeleteMountPointSymbolicLink(
    IN PUNICODE_STRING MountPoint) 
{
    NTSTATUS status;

    if (MountPoint == NULL || MountPoint->Length <= 0) 
    {
        KdPrint(("[PTFS]::DeleteMountPointSymbolicLink Null MountPoint\n"));
        return FALSE;
    }

    KdPrint(("[PTFS]::DeleteMountPointSymbolicLink Symbolic Name[%wZ]\n", MountPoint));
    
    status = IoDeleteSymbolicLink(MountPoint);
    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::DeleteMountPointSymbolicLink Failed to IoDeleteSymbolicLink[0x%x]\n", status));
        return FALSE;
    }
    return TRUE;
}

PMOUNT_ENTRY
InsertMountEntry(
    IN PPTFS_GLOBAL pPTFSGlobal, 
    IN PPTFS_CONTROL pPTFSControl,
    IN BOOLEAN lockGlobal) 
{
    PMOUNT_ENTRY pMountEntry = NULL;
    pMountEntry = PTFSAllocateZero(sizeof(MOUNT_ENTRY));

    if (pMountEntry == NULL) 
    {
        KdPrint(("[PTFS]::InsertMountEntry allocation failed\n"));
        return NULL;
    }

    RtlCopyMemory(&pMountEntry->MountControl, pPTFSControl, sizeof(PTFS_CONTROL));

    InitializeListHead(&pMountEntry->ListEntry);

    if (lockGlobal) 
        ExAcquireResourceExclusiveLite(&pPTFSGlobal->Resource, TRUE);

    InsertTailList(&pPTFSGlobal->MountPointList, &pMountEntry->ListEntry);

    if (lockGlobal)
        ExReleaseResourceLite(&pPTFSGlobal->Resource);

    return pMountEntry;
}


PMOUNT_ENTRY
FindMountEntry(
	IN PPTFS_GLOBAL pPTFSGlobal,
	IN PPTFS_CONTROL pPTFSControl,
	IN BOOLEAN lockGlobal)
{
	PLIST_ENTRY listEntry;
	PMOUNT_ENTRY pMountEntry = NULL;
	BOOLEAN useMountPoint = (pPTFSControl->MountPoint[0] != L'\0');
	BOOLEAN found = FALSE;
	BOOLEAN isSessionIdMatch = FALSE;

	KdPrint(("[PTFS]::FindMountEntry Finding mount entry lockGlobal[%d], mount point[%ws].\n", lockGlobal, pPTFSControl->MountPoint));

	if (lockGlobal)
		ExAcquireResourceExclusiveLite(&pPTFSGlobal->Resource, TRUE);

	for (listEntry = pPTFSGlobal->MountPointList.Flink; listEntry != &pPTFSGlobal->MountPointList; listEntry = listEntry->Flink)
	{
		pMountEntry = CONTAINING_RECORD(listEntry, MOUNT_ENTRY, ListEntry);
		if (useMountPoint)
		{
			isSessionIdMatch = pPTFSControl->SessionId == pMountEntry->MountControl.SessionId;
			if ((wcscmp(pPTFSControl->MountPoint, pMountEntry->MountControl.MountPoint) == 0) &&
				(isSessionIdMatch || pMountEntry->MountControl.SessionId == (ULONG)-1))
			{
				KdPrint(("[PTFS]::FindMountEntry Found entry with matching mount point.\n"));
				found = TRUE;
				break;
			}
			else
				KdPrint(("[PTFS]::FindMountEntry Skipping entry with non-matching mount point[%ws]\n", pMountEntry->MountControl.MountPoint));
		}
		else
		{
			if (wcscmp(pPTFSControl->DeviceName, pMountEntry->MountControl.DeviceName) == 0)
			{
				KdPrint(("[PTFS]::FindMountEntry Found entry with matching device name[%ws]\n", pMountEntry->MountControl.DeviceName));
				found = TRUE;
				break;
			}
		}
	}

	if (lockGlobal)
		ExReleaseResourceLite(&pPTFSGlobal->Resource);

	if (found)
	{
        KdPrint(("[PTFS]::FindMountEntry Mount entry found[%ws]->[%ws]\n", pMountEntry->MountControl.MountPoint, pMountEntry->MountControl.DeviceName));
		return pMountEntry;
	}
	else
	{
        KdPrint(("[PTFS]::FindMountEntry No mount entry found.\n"));
		return NULL;
	}
}

PMOUNT_ENTRY 
FindMountEntryByName(
    IN PPTFS_GLOBAL pPTFSGlobal,
    IN PUNICODE_STRING DiskDeviceName,
    IN PUNICODE_STRING UNCName,
    IN BOOLEAN LockGlobal
) 
{
    PMOUNT_ENTRY pMountEntry = NULL;
    PPTFS_CONTROL pPTFSControl = NULL;

    if (DiskDeviceName == NULL)
        return NULL;
    pPTFSControl = PTFSAllocateZero(sizeof(PTFS_CONTROL));

    if (pPTFSControl == NULL)
        return NULL;

    RtlCopyMemory(pPTFSControl->DeviceName, DiskDeviceName->Buffer, DiskDeviceName->Length);
    if (UNCName->Buffer != NULL && UNCName->Length > 0) 
        RtlCopyMemory(pPTFSControl->UNCName, UNCName->Buffer, UNCName->Length);

    pMountEntry = FindMountEntry(pPTFSGlobal, pPTFSControl, LockGlobal);
    PTFSFree(pPTFSControl);
    return pMountEntry;
}

VOID 
RemoveMountEntry(
    IN PPTFS_GLOBAL pPTFSGlobal, 
    IN PMOUNT_ENTRY pMountEntry) 
{
    ExAcquireResourceExclusiveLite(&pPTFSGlobal->Resource, TRUE);

    RemoveEntryList(&pMountEntry->ListEntry);
    InitializeListHead(&pMountEntry->ListEntry);

    ExReleaseResourceLite(&pPTFSGlobal->Resource);
    PTFSFree(pMountEntry);
}

VOID 
PTFSInitIrpList(
    IN PIRP_LIST pIrpList) 
{
    InitializeListHead(&pIrpList->ListHead);
    KeInitializeSpinLock(&pIrpList->ListSpinLock);
    KeInitializeEvent(&pIrpList->NotEmptyEvent, NotificationEvent, FALSE);
}

BOOLEAN
PTFSNoOpAcquire(
    IN PVOID pFcb, 
    IN BOOLEAN Wait
) 
{
    UNREFERENCED_PARAMETER(pFcb);
    UNREFERENCED_PARAMETER(Wait);
    ASSERT(IoGetTopLevelIrp() == NULL);
    IoSetTopLevelIrp((PIRP)FSRTL_CACHE_TOP_LEVEL_IRP);
    return TRUE;
}

VOID 
PTFSNoOpRelease(
    IN PVOID pFcb
) 
{
    UNREFERENCED_PARAMETER(pFcb);
    ASSERT(IoGetTopLevelIrp() == (PIRP)FSRTL_CACHE_TOP_LEVEL_IRP);
    IoSetTopLevelIrp(NULL);
}

NTSTATUS
PTFSCreateDiskDevice(
    IN PDRIVER_OBJECT pDriverObject,
    IN ULONG MountId,
    IN PWCHAR MountPoint,
    IN PWCHAR UNCName,
    IN ULONG SessionId,
    IN PWCHAR BaseGuid,
    IN PPTFS_GLOBAL pPTFSGlobal,
    IN DEVICE_TYPE DeviceType,
    IN ULONG DeviceCharacteristics,
    IN BOOLEAN MountGlobally,
    IN BOOLEAN UseMountManager,
    OUT PPTFSDCB* ppDcb)
{
    WCHAR* diskDeviceNameBuf = NULL;
    WCHAR* symbolicLinkNameBuf = NULL;
    WCHAR* mountPointBuf = NULL;
    PDEVICE_OBJECT pDiskDeviceObject = NULL;
    PPTFSDCB pDcb = NULL;
    UNICODE_STRING diskDeviceName;
    BOOLEAN isNetworkFileSystem = (DeviceType == FILE_DEVICE_NETWORK_FILE_SYSTEM);
    PPTFS_CONTROL pPTFSControl = NULL;
    NTSTATUS status = STATUS_SUCCESS;

    __try 
    {
        KdPrint(("[PTFS]::PTFSCreateDiskDevice Creating disk device mount point[%ws],mount ID[%ul]\n", MountPoint, MountId));

        diskDeviceNameBuf = PTFSAllocateZero(MAXIMUM_FILENAME_LENGTH * sizeof(WCHAR));
        symbolicLinkNameBuf = PTFSAllocateZero(MAXIMUM_FILENAME_LENGTH * sizeof(WCHAR));
        mountPointBuf = PTFSAllocateZero(MAXIMUM_FILENAME_LENGTH * sizeof(WCHAR));
        pPTFSControl = PTFSAllocateZero(sizeof(PTFS_CONTROL));

        if (diskDeviceNameBuf == NULL || symbolicLinkNameBuf == NULL || mountPointBuf == NULL || pPTFSControl == NULL) 
        {
            status = STATUS_INSUFFICIENT_RESOURCES;
            KdPrint(("[PTFS]::PTFSCreateDiskDevice Failed to allocate buffers while creating disk device\n"));
            __leave;
        }

        if (isNetworkFileSystem) {
            RtlStringCchCopyW(diskDeviceNameBuf, MAXIMUM_FILENAME_LENGTH, PTFS_NET_DEVICE_NAME);
            RtlStringCchCatW(diskDeviceNameBuf, MAXIMUM_FILENAME_LENGTH, BaseGuid);
            RtlStringCchCopyW(symbolicLinkNameBuf, MAXIMUM_FILENAME_LENGTH, PTFS_NET_SYMBOLIC_LINK_NAME);
            RtlStringCchCatW(symbolicLinkNameBuf, MAXIMUM_FILENAME_LENGTH, BaseGuid);
        }
        else 
        {
            RtlStringCchCopyW(diskDeviceNameBuf, MAXIMUM_FILENAME_LENGTH, PTFS_DISK_DEVICE_NAME);
            RtlStringCchCatW(diskDeviceNameBuf, MAXIMUM_FILENAME_LENGTH, BaseGuid);
            RtlStringCchCopyW(symbolicLinkNameBuf, MAXIMUM_FILENAME_LENGTH, PTFS_SYMBOLIC_LINK_NAME);
            RtlStringCchCatW(symbolicLinkNameBuf, MAXIMUM_FILENAME_LENGTH, BaseGuid);
        }

        RtlInitUnicodeString(&diskDeviceName, diskDeviceNameBuf);

        if (!isNetworkFileSystem) 
        {
            status = IoCreateDeviceSecure(pDriverObject,
                    sizeof(PTFSDCB),
                    &diskDeviceName,
                    FILE_DEVICE_DISK,
                    DeviceCharacteristics,
                    FALSE,
                    &g_sddl,
                    NULL,
                    &pDiskDeviceObject);
        }
        else 
        {
            status = IoCreateDevice(pDriverObject,
                sizeof(PTFSDCB),
                NULL,
                FILE_DEVICE_DISK,
                DeviceCharacteristics,
                FALSE,
                &pDiskDeviceObject);
        }

        if (!NT_SUCCESS(status)) 
        {
            KdPrint(("[PTFS]::PTFSCreateDiskDevice Failed IoCreateDevice[%d]\n", isNetworkFileSystem));
            __leave;
        }

        pDcb = pDiskDeviceObject->DeviceExtension;
        *ppDcb = pDcb;
        pDcb->pDeviceObject = pDiskDeviceObject;
        pDcb->pPTFSGlobal = pPTFSGlobal;

        pDcb->Identifier.FsdIdType = DCB;
        pDcb->Identifier.ulSize = sizeof(PTFSDCB);

        pDcb->ulMountId = MountId;
        pDcb->VolumeDeviceType = DeviceType;
        pDcb->DeviceType = FILE_DEVICE_DISK;
        pDcb->ulDeviceCharacteristics = DeviceCharacteristics;
        pDcb->ulSessionId = SessionId;
        KeInitializeEvent(&pDcb->KillEvent, NotificationEvent, FALSE);
        KeInitializeEvent(&pDcb->ForceTimeoutEvent, NotificationEvent, FALSE);
        IoInitializeRemoveLock(&pDcb->RemoveLock, TAG, 1, 100);
        pDiskDeviceObject->Flags |= DO_DIRECT_IO;

        PTFSInitIrpList(&pDcb->PendingIrpList);
        PTFSInitIrpList(&pDcb->PendingEventIrpList);
        PTFSInitIrpList(&pDcb->NotifyEventIrpList);
        PTFSInitIrpList(&pDcb->PendingRetryIrpList);

        KeInitializeEvent(&pDcb->ReleaseEvent, NotificationEvent, FALSE);
        ExInitializeResourceLite(&pDcb->Resource);

        pDcb->CacheManagerNoOpCallbacks.AcquireForLazyWrite = &PTFSNoOpAcquire;
        pDcb->CacheManagerNoOpCallbacks.ReleaseFromLazyWrite = &PTFSNoOpRelease;
        pDcb->CacheManagerNoOpCallbacks.AcquireForReadAhead = &PTFSNoOpAcquire;
        pDcb->CacheManagerNoOpCallbacks.ReleaseFromReadAhead = &PTFSNoOpRelease;

        pDcb->usMountGlobally = MountGlobally;
        pDcb->usUseMountManager = UseMountManager;

        if (wcscmp(MountPoint, L"") != 0) 
        {
            RtlStringCchCopyW(mountPointBuf, MAXIMUM_FILENAME_LENGTH, L"\\DosDevices\\");
            if (wcslen(MountPoint) < 4) 
            {
                mountPointBuf[12] = towupper(MountPoint[0]);
                mountPointBuf[13] = L':';
                mountPointBuf[14] = L'\0';
            }
            else 
                RtlStringCchCatW(mountPointBuf, MAXIMUM_FILENAME_LENGTH, MountPoint);

            if (isNetworkFileSystem)
                pDcb->usUseMountManager = FALSE;
        }
        else
            RtlStringCchCopyW(mountPointBuf, MAXIMUM_FILENAME_LENGTH, L"");

        pDcb->punstrDiskDeviceName = PTFSAllocateUnicodeString(diskDeviceNameBuf);
        pDcb->punstrSymbolicLinkName = PTFSAllocateUnicodeString(symbolicLinkNameBuf);
        pDcb->punstrMountPoint = PTFSAllocateUnicodeString(mountPointBuf);
        if (UNCName != NULL)
            pDcb->punstrUNCName = PTFSAllocateUnicodeString(UNCName);

        if (pDcb->punstrDiskDeviceName == NULL || pDcb->punstrSymbolicLinkName == NULL ||
            pDcb->punstrMountPoint == NULL || (pDcb->punstrUNCName == NULL && UNCName != NULL))
        {
            KdPrint(("[PTFS]::PTFSCreateDiskDevice Failed to allocate memory for device naming\n"));
            FreeDcbNames(pDcb);
            ExDeleteResourceLite(&pDcb->Resource);
            IoDeleteDevice(pDiskDeviceObject);
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        KdPrint(("[PTFS]::PTFSCreateDiskDevice disk device name[%wZ], symbolic link name[%wZ], mount point[%wZ], type[%d]\n"
            ,pDcb->punstrDiskDeviceName, pDcb->punstrSymbolicLinkName, pDcb->punstrMountPoint, DeviceType));

        status = IoCreateSymbolicLink(pDcb->punstrSymbolicLinkName, pDcb->punstrDiskDeviceName);

        if (!NT_SUCCESS(status)) 
        {
            KdPrint(("[PTFS]::PTFSCreateDiskDevice Failed to IoCreateSymbolicLink Status[0x%x]\n", status));
            ExDeleteResourceLite(&pDcb->Resource);
            IoDeleteDevice(pDiskDeviceObject);
            FreeDcbNames(pDcb);
            __leave;
        }

        KdPrint(("[PTFS]::PTFSCreateDiskDevice SymbolicLink[%wZ -> %wZ]created\n", pDcb->punstrSymbolicLinkName, pDcb->punstrDiskDeviceName));
        pDiskDeviceObject->Flags &= ~DO_DEVICE_INITIALIZING;

        ObReferenceObject(pDiskDeviceObject);

        RtlZeroMemory(pPTFSControl, sizeof(PTFS_CONTROL));

        RtlStringCchCopyW(pPTFSControl->DeviceName, sizeof(pPTFSControl->DeviceName) / sizeof(WCHAR), diskDeviceNameBuf);
        RtlStringCchCopyW(pPTFSControl->MountPoint, sizeof(pPTFSControl->MountPoint) / sizeof(WCHAR), mountPointBuf);
        if (UNCName != NULL)
            RtlStringCchCopyW(pPTFSControl->UNCName, sizeof(pPTFSControl->UNCName) / sizeof(WCHAR), UNCName);
 
        pPTFSControl->Type = DeviceType;
        pPTFSControl->SessionId = pDcb->ulSessionId;

        InsertMountEntry(pPTFSGlobal, pPTFSControl, FALSE);
    }
    __finally 
    {
        if (diskDeviceNameBuf)
            PTFSFree(diskDeviceNameBuf);
        if (symbolicLinkNameBuf)
            PTFSFree(symbolicLinkNameBuf);
        if (mountPointBuf)
            PTFSFree(mountPointBuf);
        if (pPTFSControl)
            PTFSFree(pPTFSControl);
    }
    return status;
}

VOID 
PTFSDeleteMountPointSysProc(
    IN PVOID pDcbParam
) 
{
    PPTFSDCB pDcb = (PPTFSDCB)pDcbParam;
    KdPrint(("[PTFS]::PTFSDeleteMountPointSysProc Start\n"));

    if (DeleteMountPointSymbolicLink(pDcb->punstrMountPoint)) 
        SetLongFlag(pDcb->ulFlags, DCB_MOUNTPOINT_DELETED);

    KdPrint(("[PTFS]::PTFSDeleteMountPointSysProc End\n"));
}


VOID DeleteMountPoint(
    IN PPTFSDCB pDcb) 
{
    if (pDcb->punstrMountPoint != NULL && pDcb->punstrMountPoint->Length > 0)
    {
        if (pDcb->usUseMountManager) 
        {
            pDcb->usUseMountManager = FALSE; 
            if (IsMountPointDriveLetter(pDcb->punstrMountPoint))
            {
                KdPrint(("[PTFS]::DeleteMountPoint Issuing a clean mount manager delete for device[%wZ]\n",  pDcb->punstrDiskDeviceName));
                SendVolumeDeletePoints(NULL, pDcb->punstrDiskDeviceName);
            }
            else if (pDcb->punstrPersistentSymbolicLinkName)
            {
                ULONG removeReparseInputlength = 0;
                PCHAR removeReparseInput = CreateRemoveReparsePointRequest(&removeReparseInputlength);

                if (removeReparseInput) 
                {
                    SendDirectoryFsctl(pDcb->pDeviceObject, pDcb->punstrMountPoint, FSCTL_DELETE_REPARSE_POINT, removeReparseInput, removeReparseInputlength);
                    PTFSFree(removeReparseInput);
                }

                NotifyDirectoryMountPointDeleted(pDcb);
                SendVolumeDeletePoints(NULL, pDcb->punstrDiskDeviceName);
            }
        }
        else 
        {
/*
            if (pDcb->usMountGlobally)
                RunAsSystem(PTFSDeleteMountPointSysProc, pDcb);
            else
*/
            {
                KdPrint(("[PTFS]::DeleteMountPoint Device mounted for current session only\n"));
                PTFSDeleteMountPointSysProc(pDcb);
            }
        }
    }
}

KSTART_ROUTINE PTFSDeregisterUncProvider;
VOID 
PTFSDeregisterUncProvider(
    IN PVOID pDcbParam) 
{
    PPTFSDCB pDcb = pDcbParam;

    if (pDcb->hMupHandle) 
    {
        FsRtlDeregisterUncProvider(pDcb->hMupHandle);
        pDcb->hMupHandle = 0;
    }

    PsTerminateSystemThread(STATUS_SUCCESS);
}

PDEVICE_ENTRY
InsertDeviceToDelete(
    IN PPTFS_GLOBAL pPTFSGlobal, 
    IN PDEVICE_OBJECT pDiskDeviceObject,
    IN PDEVICE_OBJECT pVolumeDeviceObject, 
    IN BOOLEAN lockGlobal,
    IN ULONG SessionId, 
    IN PUNICODE_STRING punstrMountPoint) 
{
    PDEVICE_ENTRY pDeviceEntry = NULL;

    ASSERT(pDiskDeviceObject != NULL);

    pDeviceEntry = PTFSAllocateZero(sizeof(DEVICE_ENTRY));
    if (pDeviceEntry == NULL) 
    {
        KdPrint(("[PTFS]::InsertDeviceToDelete Failed to allocate\n"));
        return NULL;
    }
    pDeviceEntry->pDiskDeviceObject = pDiskDeviceObject;
    pDeviceEntry->pVolumeDeviceObject = pVolumeDeviceObject;
    pDeviceEntry->ulSessionId = SessionId;

    if (SessionId != -1) 
    {
        pDeviceEntry->unstrMountPoint.Buffer = PTFSAllocate(punstrMountPoint->MaximumLength);
        if (pDeviceEntry->unstrMountPoint.Buffer == NULL) 
        {
            KdPrint(("[PTFS]::InsertDeviceToDelete Failed to allocate(2)\n"));
            PTFSFree(pDeviceEntry);
            return NULL;
        }
        pDeviceEntry->unstrMountPoint.MaximumLength = punstrMountPoint->MaximumLength;
        RtlUnicodeStringCopy(&pDeviceEntry->unstrMountPoint, punstrMountPoint);
    }

    InitializeListHead(&pDeviceEntry->ListEntry);

    if (lockGlobal) 
        ExAcquireResourceExclusiveLite(&pPTFSGlobal->Resource, TRUE);

    InsertTailList(&pPTFSGlobal->DeviceDeleteList, &pDeviceEntry->ListEntry);

    if (lockGlobal)
        ExReleaseResourceLite(&pPTFSGlobal->Resource);

    return pDeviceEntry;
}

PDEVICE_ENTRY
InsertDcbToDelete(
    IN PPTFSDCB pDcb, 
    IN PDEVICE_OBJECT pVolumeDeviceObject,
    BOOLEAN lockGlobal) 
{
    if (IsFlagOn(pDcb->ulFlags, DCB_MOUNTPOINT_DELETED)) 
    {
        KdPrint(("[PTFS]::InsertDcbToDelete MountPoint has been deleted so reset the sessionid\n"));
        pDcb->ulSessionId = (ULONG)-1;
    }
    return InsertDeviceToDelete(pDcb->pPTFSGlobal, pDcb->pDeviceObject, pVolumeDeviceObject, lockGlobal, pDcb->ulSessionId, pDcb->punstrMountPoint);
}


VOID 
DeleteDeviceObject(
    IN PPTFSDCB pDcb) 
{
    PPTFSVCB pVcb = NULL;
    PTFS_CONTROL ptfsControl;
    PMOUNT_ENTRY pMountEntry = NULL;
    PDEVICE_OBJECT pVolumeDeviceObject = NULL;

    PAGED_CODE();

    ASSERT(GETIDENTIFIERTYPE(pDcb) == DCB);

    pVcb = pDcb->pVcb;

    if (pDcb->punstrSymbolicLinkName == NULL)
    {
        KdPrint(("[PTFS]::DeleteDeviceObject Symbolic name already deleted.\n"));
        return;
    }

    KdPrint(("[PTFS]::DeleteDeviceObject Start\n"));

    RtlZeroMemory(&ptfsControl, sizeof(PTFS_CONTROL));
    RtlCopyMemory(ptfsControl.DeviceName, pDcb->punstrDiskDeviceName->Buffer, pDcb->punstrDiskDeviceName->Length);

    ptfsControl.SessionId = pDcb->ulSessionId;
    pMountEntry = FindMountEntry(pDcb->pPTFSGlobal, &ptfsControl, TRUE);
    if (pMountEntry != NULL)
    {
        if (pMountEntry->MountControl.Type == FILE_DEVICE_NETWORK_FILE_SYSTEM) 
            RunAsSystem(PTFSDeregisterUncProvider, pDcb);

        KdPrint(("[PTFS]::DeleteDeviceObject Removing mount entry.\n"));
        RemoveMountEntry(pDcb->pPTFSGlobal, pMountEntry);
    }
    else
        KdPrint(("[PTFS]::DeleteDeviceObject failed to find mount entry.\n"));

    if (pDcb->unstrMountedDeviceInterfaceName.Buffer != NULL) 
    {
        KdPrint(("[PTFS]::DeleteDeviceObject Changing interface state to false for mounted device [%wZ]\n", &pDcb->unstrMountedDeviceInterfaceName));
        IoSetDeviceInterfaceState(&pDcb->unstrMountedDeviceInterfaceName, FALSE);
        RtlFreeUnicodeString(&pDcb->unstrMountedDeviceInterfaceName);
        RtlInitUnicodeString(&pDcb->unstrMountedDeviceInterfaceName, NULL);
    }
    if (pDcb->unstrDiskDeviceInterfaceName.Buffer != NULL) 
    {
        KdPrint(("[PTFS]::DeleteDeviceObject Changing interface state to false for disk device [%wZ]\n", &pDcb->unstrDiskDeviceInterfaceName));
        IoSetDeviceInterfaceState(&pDcb->unstrDiskDeviceInterfaceName, FALSE);
        RtlFreeUnicodeString(&pDcb->unstrDiskDeviceInterfaceName);
        RtlInitUnicodeString(&pDcb->unstrDiskDeviceInterfaceName, NULL);
    }

    if (pVcb != NULL) 
    {
        KdPrint(("[PTFS]::DeleteDeviceObject Deleting volume device object.\n"));
        pVolumeDeviceObject = pVcb->pDeviceObject;
    }

    if(InsertDcbToDelete(pDcb, pVolumeDeviceObject, FALSE))
        KeSetEvent(&pDcb->pPTFSGlobal->DeleteDeviceEvent, 0, FALSE);

    KdPrint(("[PTFS]::DeleteDeviceObject End\n"));
}

