#include "FsControl.h"
#include "init.h"
#include "util.h"
#include "notification.h"
#include "event.h"
#include "fcb.h"
#include "irp_buffer_helper.h"
#include "timeout.h"
#include "mount.h"
#include <wdmsec.h>
#include <ntstrsafe.h>

#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSOplockRequest)
#pragma alloc_text (PAGE, PTFSDispatchFileSystemControl)
#endif


NTSTATUS 
PTFSOplockRequest(
    IN PIRP* ppIrp
) 
{
    NTSTATUS Status = STATUS_SUCCESS;
    PIRP pIrp = *ppIrp;
    PIO_STACK_LOCATION pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
    ULONG FsControlCode;
    PPTFSDCB pDcb = NULL;
    PPTFSVCB pVcb = NULL;
    PPTFSFCB pFcb = NULL;
    PPTFSCCB pCcb = NULL;
    PFILE_OBJECT pFileObject = NULL;
    ULONG OplockCount = 0;
    BOOLEAN AcquiredVcb = FALSE;
    BOOLEAN AcquiredFcb = FALSE;
    PREQUEST_OPLOCK_INPUT_BUFFER pInputBuffer = NULL;
    ULONG OutputBufferLength = 0;

    PAGED_CODE();

    FsControlCode = pIrpSp->Parameters.FileSystemControl.FsControlCode;
    pFileObject = pIrpSp->FileObject;

    pCcb = pFileObject->FsContext2;
    if (pCcb == NULL || pCcb->Identifier.FsdIdType != CCB) 
    {
        KdPrint(("[PTFS]::PTFSOplockRequest STATUS_INVALID_PARAMETER(CCB)"));
        return STATUS_INVALID_PARAMETER;
    }

    pFcb = pCcb->pFcb;
    if (pFcb == NULL || pFcb->Identifier.FsdIdType != FCB) 
    {
        KdPrint(("[PTFS]::PTFSOplockRequest STATUS_INVALID_PARAMETER(FCB)"));
        return STATUS_INVALID_PARAMETER;
    }

    pVcb = pFcb->pVcb;
    if (pVcb == NULL || pVcb->Identifier.FsdIdType != VCB) {
        KdPrint(("[PTFS]::PTFSOplockRequest STATUS_INVALID_PARAMETER(VCB)"));
        return STATUS_INVALID_PARAMETER;
    }

    pDcb = pVcb->pDcb;
    if (pDcb == NULL || pDcb->Identifier.FsdIdType != DCB) 
    {
        KdPrint(("[PTFS]::PTFSOplockRequest STATUS_INVALID_PARAMETER(DCB)"));
        return STATUS_INVALID_PARAMETER;
    }

    if (pDcb->bOplocksDisabled) 
    {
        KdPrint(("[PTFS]::PTFSOplockRequest Oplocks Disabled"));
        return STATUS_NOT_SUPPORTED;
    }

    if (FsControlCode == FSCTL_REQUEST_OPLOCK) 
    {
        OutputBufferLength = pIrpSp->Parameters.FileSystemControl.OutputBufferLength;
        GET_IRP_BUFFER_OR_RETURN(pIrp, pInputBuffer)
        if (OutputBufferLength < sizeof(REQUEST_OPLOCK_OUTPUT_BUFFER)) 
        {
            KdPrint(("[PTFS]::PTFSOplockRequest STATUS_BUFFER_TOO_SMALL"));
            return STATUS_BUFFER_TOO_SMALL;
        }
    }

    if ((PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY)) && 
        ((FsControlCode != FSCTL_REQUEST_OPLOCK) || !FsRtlOplockIsSharedRequest(pIrp))) 
    {
        KdPrint(("[PTFS]::PTFSOplockRequest STATUS_INVALID_PARAMETER"));
        return STATUS_INVALID_PARAMETER;
    }

    try 
    {
        ULONG level = 0;
        ULONG flags = 0;

        if ((FsControlCode == FSCTL_REQUEST_OPLOCK_LEVEL_1) ||
            (FsControlCode == FSCTL_REQUEST_BATCH_OPLOCK) ||
            (FsControlCode == FSCTL_REQUEST_FILTER_OPLOCK) ||
            (FsControlCode == FSCTL_REQUEST_OPLOCK_LEVEL_2) ||
            ((FsControlCode == FSCTL_REQUEST_OPLOCK) && FlagOn(pInputBuffer->Flags, REQUEST_OPLOCK_INPUT_FLAG_REQUEST))) 
        {
            PTFSVcbLock(pFcb->pVcb, TRUE);
            AcquiredVcb = TRUE;
            PTFSFcbLock(pFcb, FALSE);
            AcquiredFcb = TRUE;

            if (!pDcb->usFileLockInUserMode) 
            {
                if (FsRtlOplockIsSharedRequest(pIrp)) 
                {
                    if (!PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY))
                    {
                        if (g_pfn_FsRtlCheckLockForOplockRequest) // Win8+
                            OplockCount = (ULONG)!g_pfn_FsRtlCheckLockForOplockRequest(&pFcb->FileLock, &pFcb->AdvancedFCBHeader.AllocationSize);
                        else
                            OplockCount = (ULONG)FsRtlAreThereCurrentOrInProgressFileLocks(&pFcb->FileLock);
                    }
                }
                else 
                    OplockCount = pFcb->lFileCount;
            }
        }
        else if ((FsControlCode == FSCTL_OPLOCK_BREAK_ACKNOWLEDGE) ||
            (FsControlCode == FSCTL_OPBATCH_ACK_CLOSE_PENDING) ||
            (FsControlCode == FSCTL_OPLOCK_BREAK_NOTIFY) ||
            (FsControlCode == FSCTL_OPLOCK_BREAK_ACK_NO_2) ||
            ((FsControlCode == FSCTL_REQUEST_OPLOCK) && FlagOn(pInputBuffer->Flags, REQUEST_OPLOCK_INPUT_FLAG_ACK))) 
        {
            PTFSFcbLock(pFcb, TRUE);
            AcquiredFcb = TRUE;
        }
        else if (FsControlCode == FSCTL_REQUEST_OPLOCK)
        {
            KdPrint(("[PTFS]::PTFSOplockRequest STATUS_INVALID_PARAMETER(FSCTL_REQUEST_OPLOCK)"));
            Status = STATUS_INVALID_PARAMETER;
            __leave;
        }
        else 
        {
            KdPrint(("[PTFS]::PTFSOplockRequest STATUS_INVALID_PARAMETER(1)"));
            Status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        if (((FsControlCode == FSCTL_REQUEST_FILTER_OPLOCK) ||
            (FsControlCode == FSCTL_REQUEST_BATCH_OPLOCK) ||
            ((FsControlCode == FSCTL_REQUEST_OPLOCK) && FlagOn(pInputBuffer->RequestedOplockLevel, OPLOCK_LEVEL_CACHE_HANDLE)))  && 
            PTFSFCBFlagsIsSet(pFcb, PTFS_DELETE_ON_CLOSE)) 
        {
            KdPrint(("[PTFS]::PTFSOplockRequest STATUS_DELETE_PENDING"));
            Status = STATUS_DELETE_PENDING;
            __leave;
        }

        if (FsControlCode == FSCTL_REQUEST_OPLOCK) 
        {
            level = pInputBuffer->RequestedOplockLevel;
            flags = pInputBuffer->Flags;
        }

        Status = FsRtlOplockFsctrl(PTFSGetFcbOplock(pFcb), pIrp, OplockCount);
        *ppIrp = NULL;
    }

    finally 
    {
        if (AcquiredFcb) 
            PTFSFcbUnlock(pFcb);

        if (AcquiredVcb)
            PTFSVcbUnlock(pFcb->pVcb);

        KdPrint(("[PTFS]::PTFSOplockRequest status[0x%x]", Status));
    }

    return Status;
}


PCHAR 
CreateSetReparsePointRequest(
    IN PUNICODE_STRING SymbolicLinkName,
    OUT PULONG Length) 
{
    PREPARSE_DATA_BUFFER reparseData = NULL;
    USHORT mountPointReparsePathLength = SymbolicLinkName->Length + sizeof(WCHAR);
    *Length = FIELD_OFFSET(REPARSE_DATA_BUFFER, MountPointReparseBuffer.PathBuffer) + mountPointReparsePathLength + sizeof(WCHAR) + sizeof(WCHAR);
    reparseData = PTFSAllocateZero(*Length);
    if (!reparseData) 
    {
        KdPrint(("[PTFS]::CreateSetReparsePointRequest Failed to allocate reparseData buffer\n"));
        *Length = 0;
        return NULL;
    }

    reparseData->ReparseTag = IO_REPARSE_TAG_MOUNT_POINT;
    reparseData->ReparseDataLength = (USHORT)(*Length) - REPARSE_DATA_BUFFER_HEADER_SIZE;
    reparseData->MountPointReparseBuffer.SubstituteNameOffset = 0;
    reparseData->MountPointReparseBuffer.SubstituteNameLength = mountPointReparsePathLength;
    reparseData->MountPointReparseBuffer.PrintNameOffset = reparseData->MountPointReparseBuffer.SubstituteNameLength + sizeof(WCHAR);
    reparseData->MountPointReparseBuffer.PrintNameLength = 0;
    RtlCopyMemory(reparseData->MountPointReparseBuffer.PathBuffer, SymbolicLinkName->Buffer, SymbolicLinkName->Length);
    reparseData->MountPointReparseBuffer.PathBuffer[mountPointReparsePathLength / sizeof(WCHAR) - 1] = L'\\';

    return (PCHAR)reparseData;
}

NTSTATUS 
SendDirectoryFsctl(
    IN PDEVICE_OBJECT pDeviceObject, 
    IN PUNICODE_STRING Path,
    IN ULONG Code, 
    IN PCHAR Input, 
    IN ULONG Length) 
{
    UNREFERENCED_PARAMETER(pDeviceObject);

    HANDLE handle = NULL;
    PUNICODE_STRING directoryStr = NULL;

    KdPrint(("[PTFS]::SendDirectoryFsctl Start\n"));

    __try 
    {
        IO_STATUS_BLOCK ioStatusBlock;
        OBJECT_ATTRIBUTES objectAttributes;
        NTSTATUS result;

        directoryStr = ChangePrefix(Path, &g_DosDevicesPrefix, TRUE , &g_ObjectManagerPrefix);
        if (!directoryStr) 
        {
            KdPrint(("[PTFS]::SendDirectoryFsctl Failed to change prefix for [%wZ]\n", Path));
            return STATUS_INVALID_PARAMETER;
       }

        InitializeObjectAttributes(&objectAttributes, directoryStr, OBJ_CASE_INSENSITIVE | OBJ_KERNEL_HANDLE, NULL, NULL);
        KdPrint(("[PTFS]::SendDirectoryFsctl Open directory[%wZ]\n", directoryStr));

        result = ZwOpenFile(&handle, FILE_WRITE_ATTRIBUTES, &objectAttributes, &ioStatusBlock,
            FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE,
            FILE_OPEN_REPARSE_POINT | FILE_OPEN_FOR_BACKUP_INTENT);

        if (!NT_SUCCESS(result))
        {
            KdPrint(("[PTFS]::SendDirectoryFsctl Failed to ZwOpenFile[0x%x]\n", result));
            return result;
        }

        result = ZwFsControlFile(handle, NULL, NULL, NULL, &ioStatusBlock, Code, Input, Length, NULL, 0);
        if (!NT_SUCCESS(result)) 
        {
            KdPrint(("[PTFS]::SendDirectoryFsctl Failed to ZwFsControlFile[0x%x]\n", result));
            return result;
        }
    }
    __finally 
    {
        if (directoryStr) 
            PTFSFreeUnicodeString(directoryStr);

        if (handle)
            ZwClose(handle);
    }

    KdPrint(("[PTFS]::SendDirectoryFsctl End\n"));
    return STATUS_SUCCESS;
}

PCHAR 
CreateRemoveReparsePointRequest(
    PULONG pLength) 
{
    PREPARSE_DATA_BUFFER pReparseData = NULL;

    KdPrint(("[PTFS]::CreateRemoveReparsePointRequest start\n"));
    *pLength = REPARSE_GUID_DATA_BUFFER_HEADER_SIZE;

    pReparseData = PTFSAllocateZero(sizeof(REPARSE_DATA_BUFFER));
    if (NULL == pReparseData) 
    {
        KdPrint(("[PTFS]::CreateRemoveReparsePointRequest Failed to allocate reparseGuidData buffe\n"));
        *pLength = 0;
        return NULL;
    }

    pReparseData->ReparseTag = IO_REPARSE_TAG_MOUNT_POINT;
    KdPrint(("[PTFS]::CreateRemoveReparsePointRequest end\n"));
    return (PCHAR)pReparseData;
}

VOID 
InitVpb(
    IN PVPB pVpb, 
    IN PDEVICE_OBJECT pVolumeDevice
) 
{
    if (pVpb != NULL) 
    {
        pVpb->DeviceObject = pVolumeDevice;
        pVpb->VolumeLabelLength = (USHORT)wcslen(VOLUME_LABEL) * sizeof(WCHAR);
        RtlStringCchCopyW(pVpb->VolumeLabel,sizeof(pVpb->VolumeLabel) / sizeof(WCHAR), VOLUME_LABEL);
        pVpb->SerialNumber = PTFS_VOLUME_SERIAL_NUMBER;
    }
}

NTSTATUS 
PTFSMountVolume(
    IN PDEVICE_OBJECT pDiskDeviceObject, 
    IN PIRP pIrp
)
{
    PPTFSDCB pDcb = NULL;
    PPTFSVCB pVcb = NULL;
    PVPB pVpb = NULL;
    PTFS_CONTROL ptfsControl;
    PMOUNT_ENTRY pMountEntry = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    PDEVICE_OBJECT pVolumeDeviceObject = NULL;
    PDRIVER_OBJECT pDriverObject = pDiskDeviceObject->DriverObject;
    NTSTATUS status = STATUS_UNRECOGNIZED_VOLUME;
    BOOLEAN isNetworkFileSystem = FALSE, isDriveLetter = FALSE;

    KdPrint(("[PTFS]::PTFSMountVolume Start\n"));

    pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
    pDcb = pIrpSp->Parameters.MountVolume.DeviceObject->DeviceExtension;

    if (!pDcb) 
    {
        KdPrint(("[PTFS]::PTFSMountVolume no device extension\n"));
        return status;
    }

    if (GETIDENTIFIERTYPE(pDcb) != DCB)
    {
        KdPrint(("[PTFS]::PTFSMountVolume Invalid identifier type\n"));
        return status;
    }

    if (IsDeletePending(pDcb->pDeviceObject)) 
    {
        KdPrint(("[PTFS]::PTFSMountVolume deleted pending\n"));
        return STATUS_DEVICE_REMOVED;
    }

    isNetworkFileSystem = (pDcb->VolumeDeviceType == FILE_DEVICE_NETWORK_FILE_SYSTEM);

    KdPrint(("[PTFS]::PTFSMountVolume MountPoint[%wZ], Device[%wZ]\n", pDcb->punstrMountPoint, pDcb->punstrDiskDeviceName));

    if (!isNetworkFileSystem) 
    {
        status = IoCreateDevice(pDriverObject,
            sizeof(PTFSVCB),
            NULL,
            pDcb->VolumeDeviceType,
            pDcb->ulDeviceCharacteristics,
            FALSE, 
            &pVolumeDeviceObject);
    }
    else
    {
        status = IoCreateDeviceSecure(
            pDriverObject,
            sizeof(PTFSVCB),
            pDcb->punstrDiskDeviceName, 
            pDcb->VolumeDeviceType, 
            pDcb->ulDeviceCharacteristics,
            FALSE,
            &g_sddl,
            NULL, 
            &pVolumeDeviceObject);
    }

    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::PTFSMountVolume failed to IoCreateDevice status[0x%x]\n", status));
        return status;
    }

    pVcb = pVolumeDeviceObject->DeviceExtension;
    pVcb->Identifier.FsdIdType = VCB;
    pVcb->Identifier.ulSize = sizeof(PTFSVCB);

    pVcb->pDeviceObject = pVolumeDeviceObject;
    pVcb->pDcb = pDcb;
    pVcb->ValidFcbMask = 0xffffffffffffffff;
    pDcb->pVcb = pVcb;

    if (pVcb->pDcb->ulFcbGarbageCollectionIntervalMs != 0) 
    {
        InitializeListHead(&pVcb->FcbGarbageList);
        KeInitializeEvent(&pVcb->FcbGarbageListNotEmpty, SynchronizationEvent,FALSE);
        StartFcbGarbageCollector(pVcb);
    }

    InitializeListHead(&pVcb->NextFCB);
    InitializeListHead(&pVcb->DirNotifyList);
    FsRtlNotifyInitializeSync(&pVcb->NotifySync);
    ExInitializeFastMutex(&pVcb->AdvancedFCBHeaderMutex);
    FsRtlSetupAdvancedHeader(&pVcb->VolumeFileHeader, &pVcb->AdvancedFCBHeaderMutex);

    pVpb = pIrpSp->Parameters.MountVolume.Vpb;
    InitVpb(pVpb, pVcb->pDeviceObject);

    SetLongFlag(pVolumeDeviceObject->Flags, DO_DIRECT_IO);
    ClearLongFlag(pVolumeDeviceObject->Flags, DO_DEVICE_INITIALIZING);
    SetLongFlag(pVcb->ulFlags, VCB_MOUNTED);

    ObReferenceObject(pVolumeDeviceObject);

     ExAcquireResourceExclusiveLite(&pDcb->Resource, TRUE);

    RtlZeroMemory(&ptfsControl, sizeof(PTFS_CONTROL));
    RtlCopyMemory(ptfsControl.DeviceName, pDcb->punstrDiskDeviceName->Buffer, pDcb->punstrDiskDeviceName->Length);
    if (pDcb->punstrUNCName->Buffer != NULL && pDcb->punstrUNCName->Length > 0) 
        RtlCopyMemory(ptfsControl.UNCName, pDcb->punstrUNCName->Buffer, pDcb->punstrUNCName->Length);
    ptfsControl.SessionId = pDcb->ulSessionId;
    pMountEntry = FindMountEntry(pDcb->pPTFSGlobal, &ptfsControl, TRUE);
    if (pMountEntry != NULL) 
        pMountEntry->MountControl.VolumeDeviceObject = pVolumeDeviceObject;
    else 
    {
        ExReleaseResourceLite(&pDcb->Resource);
        KdPrint(("[PTFS]::PTFSMountVolume MountEntry not found\n"));
        return STATUS_DEVICE_REMOVED;
    }

    ExReleaseResourceLite(&pDcb->Resource);

    ExAcquireResourceExclusiveLite(&pDcb->Resource, TRUE);
    UpdateTimeout(&pDcb->liTickCount, PTFS_KEEPALIVE_TIMEOUT_DEFAULT * 3);
    ExReleaseResourceLite(&pDcb->Resource);
    StartCheckThread(pDcb);

    isDriveLetter = IsMountPointDriveLetter(pDcb->punstrMountPoint);
    if (pDcb->usUseMountManager) 
    {
        BOOLEAN autoMountStateBackup = TRUE;
        if (!isDriveLetter) 
        {
            ExAcquireResourceExclusiveLite(&pDcb->pPTFSGlobal->MountManagerLock, TRUE);
            QueryAutoMount(&autoMountStateBackup);
            if (autoMountStateBackup) 
                SendAutoMount(FALSE);
        }
        status = SendVolumeArrivalNotification(pDcb->punstrDiskDeviceName);
        if (!NT_SUCCESS(status)) 
            KdPrint(("[PTFS]::PTFSMountVolume failed to SendVolumeArrivalNotification[0x%x]\n", status));

        if (!isDriveLetter) 
        {
            if (autoMountStateBackup)
                SendAutoMount(TRUE);
            ExReleaseResourceLite(&pDcb->pPTFSGlobal->MountManagerLock);
        }
    }

    if (isDriveLetter) 
        PTFSCreateMountPoint(pDcb);

    if (isNetworkFileSystem) 
        RunAsSystem(RegisterUncProvider, pDcb);

    KdPrint(("[PTFS]::PTFSMountVolume Mounting successfully done.\n"));
    return STATUS_SUCCESS;
}

NTSTATUS
PTFSUserFsRequest(
    IN PDEVICE_OBJECT pDeviceObject, 
    IN PIRP* ppIrp
) 
{
    UNREFERENCED_PARAMETER(pDeviceObject);

    NTSTATUS status = STATUS_INVALID_DEVICE_REQUEST;
    PIO_STACK_LOCATION pIrpSp = NULL;
    PFILE_OBJECT pFileObject = NULL;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;

    KdPrint(("[PTFS]::PTFSUserFsRequest Start\n"));

    pIrpSp = IoGetCurrentIrpStackLocation(*ppIrp);

    switch (pIrpSp->Parameters.FileSystemControl.FsControlCode) 
    {
    case FSCTL_ACTIVATE_KEEPALIVE:
        pFileObject = pIrpSp->FileObject;
        if (pFileObject == NULL)
        {
            KdPrint(("[PTFS]::PTFSUserFsRequest FileObject is null\n"));
            return STATUS_INVALID_PARAMETER;
        }
        pCcb = pFileObject->FsContext2;
        if (pCcb == NULL || pCcb->Identifier.FsdIdType != CCB) 
        {
            KdPrint(("[PTFS]::PTFSUserFsRequest no CCB\n"));
            return STATUS_INVALID_PARAMETER;
        }

        pFcb = pCcb->pFcb;
        if (pFcb == NULL || pFcb->Identifier.FsdIdType != FCB) 
        {
            KdPrint(("[PTFS]::PTFSUserFsRequest no FCB\n"));
            return STATUS_INVALID_PARAMETER;
        }

        if (!pFcb->bIsKeepalive) 
        {
            KdPrint(("[PTFS]::PTFSUserFsRequest FSCTL_ACTIVATE_KEEPALIVE for wrong file: %wZ\n", pFcb->unstrFileName));
            return STATUS_INVALID_PARAMETER;
        }

        if (pFcb->pVcb->bIsKeepaliveActive && !pCcb->bIsKeepaliveActive)
        {
            KdPrint(("[PTFS]::PTFSUserFsRequest FSCTL_ACTIVATE_KEEPALIVE when a different keepalive handle\n"));
            return STATUS_INVALID_PARAMETER;
        }

        KdPrint(("[PTFS]::PTFSUserFsRequest Activating keepalive handle from process[%lu]\n", IoGetRequestorProcessId(*ppIrp)));
        PTFSFcbLock(pFcb, FALSE);
        pCcb->bIsKeepaliveActive = TRUE;
        pFcb->pVcb->bIsKeepaliveActive = TRUE;
        PTFSFcbUnlock(pFcb);
        status = STATUS_SUCCESS;
        break;

    case FSCTL_NOTIFY_PATH: 
    {
        PNOTIFY_PATH_INTERMEDIATE pNotifyPath = NULL;
        UNICODE_STRING receivedBuffer;

        GET_IRP_NOTIFY_PATH_INTERMEDIATE_OR_RETURN(*ppIrp, pNotifyPath)
        pIrpSp = IoGetCurrentIrpStackLocation(*ppIrp);

        pFileObject = pIrpSp->FileObject;
        if (pFileObject == NULL)
        {
            KdPrint(("[PTFS]::PTFSUserFsRequest FileObject is null(FSCTL_NOTIFY_PATH)\n"));
            return STATUS_INVALID_PARAMETER;
        }

        pCcb = pFileObject->FsContext2;
        if (pCcb == NULL || pCcb->Identifier.FsdIdType != CCB) 
        {
            KdPrint(("[PTFS]::PTFSUserFsRequest no CCB(FSCTL_NOTIFY_PATH)\n"));
            return STATUS_INVALID_PARAMETER;
        }

        pFcb = pCcb->pFcb;
        if (pFcb == NULL || pFcb->Identifier.FsdIdType != FCB) 
        {
            KdPrint(("[PTFS]::PTFSUserFsRequest no FCB(FSCTL_NOTIFY_PATH)\n"));
            return STATUS_INVALID_PARAMETER;
        }

        receivedBuffer.Length = pNotifyPath->Length;
        receivedBuffer.MaximumLength = pNotifyPath->Length;
        receivedBuffer.Buffer = pNotifyPath->Buffer;

        KdPrint(("[PTFS]::PTFSUserFsRequest Received FSCTL_NOTIFY_PATH, CompletionFilter: %lu, Action: %lu, Length: %i, Path: %wZ\n",
            pNotifyPath->CompletionFilter, pNotifyPath->Action, receivedBuffer.Length, &receivedBuffer));

 
        PTFSFcbLock(pFcb, TRUE);
        status = NotifyReportChange0( pFcb, &receivedBuffer, pNotifyPath->CompletionFilter, pNotifyPath->Action);
        PTFSFcbUnlock(pFcb);

        if (status == STATUS_OBJECT_NAME_INVALID) 
            CleanupAllChangeNotificationWaiters(pFcb->pVcb);
    }
    break;

    case FSCTL_REQUEST_OPLOCK_LEVEL_1:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_REQUEST_OPLOCK_LEVEL_1\n"));
        status = PTFSOplockRequest(ppIrp);
        break;

    case FSCTL_REQUEST_OPLOCK_LEVEL_2:
        KdPrint(("[PTFS]::PTFSUserFsRequest FileObject FSCTL_REQUEST_OPLOCK_LEVEL_2\n"));
        status = PTFSOplockRequest(ppIrp);
        break;

    case FSCTL_REQUEST_BATCH_OPLOCK:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_REQUEST_BATCH_OPLOCK\n"));
        status = PTFSOplockRequest(ppIrp);
        break;

    case FSCTL_OPLOCK_BREAK_ACKNOWLEDGE:
        KdPrint(("[PTFS]::PTFSUserFsRequest FileObject FSCTL_OPLOCK_BREAK_ACKNOWLEDGE\n"));
        status = PTFSOplockRequest(ppIrp);
        break;

    case FSCTL_OPBATCH_ACK_CLOSE_PENDING:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_OPBATCH_ACK_CLOSE_PENDING\n"));
        status = PTFSOplockRequest(ppIrp);
        break;

    case FSCTL_OPLOCK_BREAK_NOTIFY:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_OPLOCK_BREAK_NOTIFY\n"));
        status = PTFSOplockRequest(ppIrp);
        break;

    case FSCTL_OPLOCK_BREAK_ACK_NO_2:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_OPLOCK_BREAK_ACK_NO_2\n"));
        status = PTFSOplockRequest(ppIrp);
        break;

    case FSCTL_REQUEST_FILTER_OPLOCK:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_REQUEST_FILTER_OPLOCK\n"));
        status = PTFSOplockRequest(ppIrp);
        break;

    case FSCTL_REQUEST_OPLOCK:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_REQUEST_OPLOCK\n"));
        status = PTFSOplockRequest(ppIrp);
        break;

    case FSCTL_LOCK_VOLUME:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_LOCK_VOLUME\n"));
        status = STATUS_SUCCESS;
        break;

    case FSCTL_UNLOCK_VOLUME:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_UNLOCK_VOLUME\n"));
        status = STATUS_SUCCESS;
        break;

    case FSCTL_DISMOUNT_VOLUME:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_DISMOUNT_VOLUME\n"));
        break;

    case FSCTL_IS_VOLUME_MOUNTED:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_IS_VOLUME_MOUNTED\n"));
        status = STATUS_SUCCESS;
        break;

    case FSCTL_IS_PATHNAME_VALID:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_IS_PATHNAME_VALID\n"));
        break;

    case FSCTL_MARK_VOLUME_DIRTY:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_MARK_VOLUME_DIRTY\n"));
        break;

    case FSCTL_QUERY_RETRIEVAL_POINTERS:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_QUERY_RETRIEVAL_POINTERS\n"));
        break;

    case FSCTL_GET_COMPRESSION:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_GET_COMPRESSION\n"));
        break;

    case FSCTL_SET_COMPRESSION:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SET_COMPRESSION\n"));
        break;

    case FSCTL_MARK_AS_SYSTEM_HIVE:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_MARK_AS_SYSTEM_HIVE\n"));
        break;

    case FSCTL_INVALIDATE_VOLUMES:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_INVALIDATE_VOLUMES\n"));
        break;

    case FSCTL_QUERY_FAT_BPB:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_QUERY_FAT_BPB\n"));
        break;

    case FSCTL_FILESYSTEM_GET_STATISTICS:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_FILESYSTEM_GET_STATISTICS\n"));
        break;

    case FSCTL_GET_NTFS_VOLUME_DATA:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_GET_NTFS_VOLUME_DATA\n"));
        break;

    case FSCTL_GET_NTFS_FILE_RECORD:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_GET_NTFS_FILE_RECORD\n"));
        break;

    case FSCTL_GET_VOLUME_BITMAP:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_GET_VOLUME_BITMAP\n"));
        break;

    case FSCTL_GET_RETRIEVAL_POINTERS:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_GET_RETRIEVAL_POINTERS\n"));
        break;

    case FSCTL_MOVE_FILE:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_MOVE_FILE\n"));
        break;

    case FSCTL_IS_VOLUME_DIRTY:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_IS_VOLUME_DIRTY\n"));
        break;

    case FSCTL_ALLOW_EXTENDED_DASD_IO:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_ALLOW_EXTENDED_DASD_IO\n"));
        break;

    case FSCTL_FIND_FILES_BY_SID:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_FIND_FILES_BY_SID\n"));
        break;

    case FSCTL_SET_OBJECT_ID:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SET_OBJECT_ID\n"));
        break;

    case FSCTL_GET_OBJECT_ID:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_GET_OBJECT_ID\n"));
        break;

    case FSCTL_DELETE_OBJECT_ID:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_DELETE_OBJECT_ID\n"));
        break;

    case FSCTL_SET_REPARSE_POINT:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SET_REPARSE_POINT\n"));
        break;

    case FSCTL_GET_REPARSE_POINT:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_GET_REPARSE_POINT\n"));
        status = STATUS_NOT_A_REPARSE_POINT;
        break;

    case FSCTL_DELETE_REPARSE_POINT:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_DELETE_REPARSE_POINT\n"));
        break;

    case FSCTL_ENUM_USN_DATA:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_ENUM_USN_DATA\n"));
        break;

    case FSCTL_SECURITY_ID_CHECK:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SECURITY_ID_CHECK\n"));
        break;

    case FSCTL_READ_USN_JOURNAL:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_READ_USN_JOURNAL\n"));
        break;

    case FSCTL_SET_OBJECT_ID_EXTENDED:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SET_OBJECT_ID_EXTENDED\n"));
        break;

    case FSCTL_CREATE_OR_GET_OBJECT_ID:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_CREATE_OR_GET_OBJECT_ID\n"));
        break;

    case FSCTL_SET_SPARSE:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SET_SPARSE\n"));
        break;

    case FSCTL_SET_ZERO_DATA:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SET_ZERO_DATA\n"));
        break;

    case FSCTL_QUERY_ALLOCATED_RANGES:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_QUERY_ALLOCATED_RANGES\n"));
        break;

    case FSCTL_SET_ENCRYPTION:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SET_ENCRYPTION\n"));
        break;

    case FSCTL_ENCRYPTION_FSCTL_IO:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_ENCRYPTION_FSCTL_IO\n"));
        break;

    case FSCTL_WRITE_RAW_ENCRYPTED:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_WRITE_RAW_ENCRYPTED\n"));
        break;

    case FSCTL_READ_RAW_ENCRYPTED:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_READ_RAW_ENCRYPTED\n"));
        break;

    case FSCTL_CREATE_USN_JOURNAL:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_CREATE_USN_JOURNAL\n"));
        break;

    case FSCTL_READ_FILE_USN_DATA:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_READ_FILE_USN_DATA\n"));
        break;

    case FSCTL_WRITE_USN_CLOSE_RECORD:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_WRITE_USN_CLOSE_RECORD\n"));
        break;

    case FSCTL_EXTEND_VOLUME:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_EXTEND_VOLUME\n"));
        break;

    case FSCTL_QUERY_USN_JOURNAL:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_QUERY_USN_JOURNAL\n"));
        break;

    case FSCTL_DELETE_USN_JOURNAL:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_DELETE_USN_JOURNAL\n"));
        break;

    case FSCTL_MARK_HANDLE:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_MARK_HANDLE\n"));
        break;

    case FSCTL_SIS_COPYFILE:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SIS_COPYFILE\n"));
        break;

    case FSCTL_SIS_LINK_FILES:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SIS_LINK_FILES\n"));
        break;

    case FSCTL_RECALL_FILE:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_RECALL_FILE\n"));
        break;

    case FSCTL_SET_ZERO_ON_DEALLOCATION:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_SET_ZERO_ON_DEALLOCATION\n"));
        break;

    case FSCTL_CSC_INTERNAL:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_CSC_INTERNAL\n"));
        break;

    case FSCTL_QUERY_ON_DISK_VOLUME_INFO:
        KdPrint(("[PTFS]::PTFSUserFsRequest  FSCTL_QUERY_ON_DISK_VOLUME_INFO\n"));
        break;

    default:
        KdPrint(("[PTFS]::PTFSUserFsRequest  Unknown FSCTL[%d]\n", (pIrpSp->Parameters.FileSystemControl.FsControlCode >> 2) & 0xFFF));
        status = STATUS_INVALID_DEVICE_REQUEST;
    }

    return status;
}

NTSTATUS
PTFSDispatchFileSystemControl(
    IN PDEVICE_OBJECT pDeviceObject,
    IN PIRP pIrp
)
{
    NTSTATUS status = STATUS_INVALID_DEVICE_REQUEST;
    PIO_STACK_LOCATION pIrpSp = NULL;

    __try 
    {
        KdPrint(("[PTFS]::PTFSDispatchFileSystemControl start PID[%lu]\n", IoGetRequestorProcessId(pIrp)));
        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);

        switch (pIrpSp->MinorFunction)
        {
        case IRP_MN_KERNEL_CALL:
            KdPrint(("[PTFS]::PTFSDispatchFileSystemControl IRP_MN_KERNEL_CALL\n"));
            break;

        case IRP_MN_LOAD_FILE_SYSTEM:
            KdPrint(("[PTFS]::PTFSDispatchFileSystemControl IRP_MN_LOAD_FILE_SYSTEM\n"));
            break;

        case IRP_MN_MOUNT_VOLUME: 
            KdPrint(("[PTFS]::PTFSDispatchFileSystemControl IRP_MN_MOUNT_VOLUME\n"));
            status = PTFSMountVolume(pDeviceObject, pIrp);
            break;

        case IRP_MN_USER_FS_REQUEST:
            KdPrint(("[PTFS]::PTFSDispatchFileSystemControl IRP_MN_USER_FS_REQUEST\n"));
            status = PTFSUserFsRequest(pDeviceObject, &pIrp);
            break;

        case IRP_MN_VERIFY_VOLUME:
            KdPrint(("[PTFS]::PTFSDispatchFileSystemControl IRP_MN_VERIFY_VOLUME\n"));
            break;

        default:
            KdPrint(("[PTFS]::PTFSDispatchFileSystemControl unknown[%d]\n", pIrpSp->MinorFunction));
            status = STATUS_INVALID_DEVICE_REQUEST;
            break;
        }
    }
    __finally 
    {
        PTFSCompleteIrpRequest(pIrp, status, 0);
        KdPrint(("[PTFS]::PTFSDispatchFileSystemControl end\n"));
    }

    return status;
}
