#include "fileInfo.h"
#include "init.h"
#include "util.h"
#include "event.h"
#include "except.h"
#include "notification.h"
#include "irp_buffer_helper.h"
#include <ntstrsafe.h>


#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSDispatchQueryInformation)
#pragma alloc_text (PAGE, PTFSDispatchSetInformation)
#endif

LONG 
GetParentDirectoryEndingIndex(
    IN PUNICODE_STRING punstrFileName
) 
{
    USHORT lastIndex = 0;
    LONG index = 0;

    if (punstrFileName->Length == 0) 
        return -1;

    lastIndex = punstrFileName->Length / sizeof(WCHAR) - 1;
    if (punstrFileName->Buffer[lastIndex] == L'\\') 
        lastIndex--;

    for(index = lastIndex; index >= 0; index--)
    {
        if (punstrFileName->Buffer[index] == L'\\')
            return index;
    }

    return -1;
}

BOOLEAN 
IsInSameDirectory(
    IN PUNICODE_STRING punstrFileName1,
    IN PUNICODE_STRING punstrFileName2
) 
{
    LONG i = 0;
    LONG parentEndingIndex = GetParentDirectoryEndingIndex(punstrFileName1);

    if (parentEndingIndex != GetParentDirectoryEndingIndex(punstrFileName2)) 
        return FALSE;

    for (i = 0; i < parentEndingIndex; i++) 
    {
        if (punstrFileName1->Buffer[i] != punstrFileName2->Buffer[i])
            return FALSE;
    }

    return TRUE;
}

VOID
FlushFcb(
    IN PPTFSFCB pFcb,
    IN OPTIONAL PFILE_OBJECT pFileObject
)
{
    if (pFcb == NULL)
        return;

    if (pFcb->SectionObjectPointers.ImageSectionObject != NULL)
    {
        KdPrint(("[PTFS]::FlushFcb MmFlushImageSection FileName[%wZ] FileCount[%lu]\n", &pFcb->unstrFileName, pFcb->lFileCount));
        MmFlushImageSection(&pFcb->SectionObjectPointers, MmFlushForWrite);
        KdPrint(("[PTFS]::FlushFcb MmFlushImageSection done FileName[%wZ] FileCount[%lu]\n", &pFcb->unstrFileName, pFcb->lFileCount));
    }

    if (pFcb->SectionObjectPointers.DataSectionObject != NULL)
    {
        KdPrint(("[PTFS]::FlushFcb CcFlushCache FileName[%wZ] FileCount[%lu]\n", &pFcb->unstrFileName, pFcb->lFileCount));

        CcFlushCache(&pFcb->SectionObjectPointers, NULL, 0, NULL);

        PTFSPagingIoLock(pFcb, FALSE);
        PTFSPagingIoUnlock(pFcb);

        CcPurgeCacheSection(&pFcb->SectionObjectPointers, NULL, 0, FALSE);

        if (pFileObject != NULL)
            CcUninitializeCacheMap(pFileObject, NULL, NULL);

        KdPrint(("[PTFS]::FlushFcb CcFlushCache done FileName[%wZ] FileCount[%lu]\n", &pFcb->unstrFileName, pFcb->lFileCount));
    }
}

VOID 
FlushAllCachedFcb(
    IN PPTFSFCB pFcbRelatedTo,
    IN OPTIONAL PFILE_OBJECT pFileObject
) 
{
    PLIST_ENTRY thisEntry, nextEntry, listHead;
    PPTFSFCB pFcb = NULL;

    if (pFcbRelatedTo == NULL)
        return;

    KdPrint(("[PTFS]::FlushAllCachedFcb Start\n"));

    if (!PTFSFCBFlagsIsSet(pFcbRelatedTo, PTFS_FILE_DIRECTORY)) 
    {
        KdPrint(("[PTFS]::FlushAllCachedFcb Flush File Only[%wZ]\n", &pFcbRelatedTo->unstrFileName));
        FlushFcb(pFcbRelatedTo, pFileObject);
        return;
    }

    PTFSVcbLock(pFcbRelatedTo->pVcb, FALSE);

    listHead = &pFcbRelatedTo->pVcb->NextFCB;

    for (thisEntry = listHead->Flink; thisEntry != listHead; thisEntry = nextEntry) 
    {
        nextEntry = thisEntry->Flink;
        pFcb = CONTAINING_RECORD(thisEntry, PTFSFCB, NextFCB);
        if (PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY)) 
        {
            KdPrint(("[PTFS]::FlushAllCachedFcb Skip file[%wZ]\n", &pFcb->unstrFileName));
            continue;
        }

        KdPrint(("[PTFS]::FlushAllCachedFcb check[%wZ] related [%wZ]\n", &pFcb->unstrFileName, &pFcbRelatedTo->unstrFileName));

        if (StartsWith(&pFcb->unstrFileName, &pFcbRelatedTo->unstrFileName)) 
        {
            KdPrint(("[PTFS]::FlushAllCachedFcb flush possible [%wZ]\n", &pFcb->unstrFileName));
            FlushFcb(pFcb, NULL);
        }

        pFcb = NULL;
    }

    PTFSVcbUnlock(pFcbRelatedTo->pVcb);
    KdPrint(("[PTFS]::FlushAllCachedFcb End\n"));
}

NTSTATUS
PTFSDispatchQueryInformation(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
    NTSTATUS status = STATUS_NOT_IMPLEMENTED;
    PIO_STACK_LOCATION pIrpSp = NULL;
    PFILE_OBJECT pFileObject = NULL;
    FILE_INFORMATION_CLASS infoClass;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;
    PPTFSVCB pVcb = NULL;
    ULONG eventLength = 0;
    PEVENT_CONTEXT pEventContext = NULL;

    __try 
    {
        pIrp->IoStatus.Information = 0;

        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
        pFileObject = pIrpSp->FileObject;
        infoClass = pIrpSp->Parameters.QueryFile.FileInformationClass;

        KdPrint(("[PTFS]::PTFSDispatchQueryInformation Start FileInfoClass[%d], PID[%lu]\n", infoClass, IoGetRequestorProcessId(pIrp)));

        if (pFileObject == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileObject is null\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }


        pVcb = pDeviceObject->DeviceExtension;
        if (GETIDENTIFIERTYPE(pVcb) != VCB || !PTFSCheckCCB(pVcb->pDcb, pFileObject->FsContext2))
        {
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation Invalid identifier type\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        pCcb = (PPTFSCCB)pFileObject->FsContext2;
        ASSERT(pCcb != NULL);

        pFcb = pCcb->pFcb;
        ASSERT(pFcb != NULL);

        PTFSFcbLock(pFcb, TRUE);
        switch (infoClass) 
        {
        case FileBasicInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileBasicInformation\n"));
            break;
        case FileInternalInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileInternalInformation\n"));
            break;
        case FileEaInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileEaInformation\n"));
            break;
        case FileStandardInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileStandardInformation\n"));
            break;
        case FileAllInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileAllInformation\n"));
            break;
        case FileAlternateNameInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileAlternateNameInformation\n"));
            break;
        case FileAttributeTagInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileAttributeTagInformation\n"));
            break;
        case FileCompressionInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileCompressionInformation\n"));
            break;
        case FileNormalizedNameInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileNormalizedNameInformation\n"));
        case FileNameInformation: 
        {
            PFILE_NAME_INFORMATION pNameInfo = NULL;
            PUNICODE_STRING punstrFileName = &pFcb->unstrFileName;
            PCHAR dest = NULL;
            BOOLEAN isNetworkDevice = FALSE;

            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileNameInformation\n"));
            if (!PREPARE_OUTPUT(pIrp, pNameInfo, FALSE)) 
            {
                KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileNameInformation Invalid output pBuffer\n"));
                status = STATUS_BUFFER_OVERFLOW;
                __leave;
            }

            dest = (PCHAR)&pNameInfo->FileName;
            pNameInfo->FileNameLength = punstrFileName->Length;

            isNetworkDevice = (pVcb->pDcb->VolumeDeviceType == FILE_DEVICE_NETWORK_FILE_SYSTEM);

            if (isNetworkDevice) 
            {
                PUNICODE_STRING devicePath = pVcb->pDcb->punstrUNCName->Length ? pVcb->pDcb->punstrUNCName : pVcb->pDcb->punstrDiskDeviceName;
                pNameInfo->FileNameLength += devicePath->Length;

                if (!AppendVarSizeOutputString(pIrp, dest, devicePath,FALSE,TRUE)) 
                {
                    status = STATUS_BUFFER_OVERFLOW;
                    __leave;
                }
                dest += devicePath->Length;
            }

            if (!AppendVarSizeOutputString(pIrp, dest, punstrFileName, FALSE, TRUE)) 
            {
                status = STATUS_BUFFER_OVERFLOW;
                __leave;
            }
            status = STATUS_SUCCESS;
            __leave;
        } 
        break;

        case FileNetworkOpenInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileNetworkOpenInformation\n"));
            break;
        case FilePositionInformation:
        {
            PFILE_POSITION_INFORMATION pFilePosInfo = NULL;

            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FilePositionInformation\n"));
            if (!PREPARE_OUTPUT(pIrp, pFilePosInfo,FALSE)) 
            {
                KdPrint(("[PTFS]::PTFSDispatchQueryInformation FilePositionInformation Invalid output pBuffer\n"));
                status = STATUS_INFO_LENGTH_MISMATCH;
                __leave;
            }

            if (pFileObject->CurrentByteOffset.QuadPart < 0)
            {
                KdPrint(("[PTFS]::PTFSDispatchQueryInformation FilePositionInformation Invalid CurrentByteOffset\n"));
                status = STATUS_INVALID_PARAMETER;
                __leave;
            }

            pFilePosInfo->CurrentByteOffset = pFileObject->CurrentByteOffset;
            status = STATUS_SUCCESS;
            __leave;
        } 
        break;

        case FileStreamInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileStreamInformation\n"));
            if (!pVcb->pDcb->usUseAltStream) 
            {
                KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileStreamInformation alternate stream disabled\n"));
                status = STATUS_NOT_IMPLEMENTED;
                __leave;
            }
            break;
        case FileStandardLinkInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileStandardLinkInformation\n"));
            break;
        case FileNetworkPhysicalNameInformation:
        {
            PFILE_NETWORK_PHYSICAL_NAME_INFORMATION pNetInfo = NULL;
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileNetworkPhysicalNameInformation\n"));

            if (!PREPARE_OUTPUT(pIrp, pNetInfo,FALSE)) 
            {
                status = STATUS_BUFFER_OVERFLOW;
                __leave;
            }

            if (!AppendVarSizeOutputString(pIrp, &pNetInfo->FileName, &pFcb->unstrFileName,FALSE, FALSE)) 
            {
                status = STATUS_BUFFER_OVERFLOW;
                __leave;
            }

            status = STATUS_SUCCESS;
            __leave;
        }
        case FileRemoteProtocolInformation:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation FileRemoteProtocolInformation\n"));
            break;

        default:
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation unknown type[%d]\n", infoClass));
            break;
        }

        if (pFcb->bBlockUserModeDispatch)
        {
            status = STATUS_SUCCESS;
            __leave;
        }

        eventLength = sizeof(EVENT_CONTEXT) + pFcb->unstrFileName.Length;
        pEventContext = AllocateEventContext(pVcb->pDcb, pIrp, eventLength, pCcb);
        if (pEventContext == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchQueryInformation Failed to AllocateEventContext\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        pEventContext->Context = pCcb->UserContext;
        pEventContext->Operation.File.FileInformationClass = infoClass;
        pEventContext->Operation.File.BufferLength = pIrpSp->Parameters.QueryFile.Length;
        pEventContext->Operation.File.FileNameLength = pFcb->unstrFileName.Length;
        RtlCopyMemory(pEventContext->Operation.File.FileName, pFcb->unstrFileName.Buffer, pFcb->unstrFileName.Length);

        status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, 0);
    }
    __finally 
    {
        if (pFcb)
            PTFSFcbUnlock(pFcb);
        PTFSCompleteDispatchRoutine(pIrp, status);
        KdPrint(("[PTFS]::PTFSDispatchQueryInformation End\n"));
    }

    return status;
}

NTSTATUS
PTFSDispatchSetInformation(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
    NTSTATUS status = STATUS_NOT_IMPLEMENTED;
    PIO_STACK_LOCATION pIrpSp = NULL;
    PFILE_OBJECT pFileObject = NULL;
    PVOID pBuffer = NULL;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;
    PPTFSVCB pVcb = NULL;
    ULONG eventLength = 0;
    PFILE_OBJECT pTargetFileObject = NULL;
    PEVENT_CONTEXT pEventContext = NULL;
    BOOLEAN isPagingIo = FALSE;
    BOOLEAN fcbLocked = FALSE;
    BOOLEAN isRenameOrLink = FALSE;

    pVcb = pDeviceObject->DeviceExtension;

    __try 
    {
        KdPrint(("[PTFS]::PTFSDispatchSetInformation start\n"));

        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
        pFileObject = pIrpSp->FileObject;

        if (pFileObject == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileObject is null\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        if (GETIDENTIFIERTYPE(pVcb) != VCB || !PTFSCheckCCB(pVcb->pDcb, pFileObject->FsContext2)) 
        {
            KdPrint(("[PTFS]::PTFSDispatchSetInformation invalid identifier Type\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        pCcb = (PPTFSCCB)pFileObject->FsContext2;
        ASSERT(pCcb != NULL);

        KdPrint(("[PTFS]::PTFSDispatchSetInformation ProcessId[%lu]\n", IoGetRequestorProcessId(pIrp)));

        pBuffer = pIrp->AssociatedIrp.SystemBuffer;
        isPagingIo = (pIrp->Flags & IRP_PAGING_IO);

        pFcb = pCcb->pFcb;
        ASSERT(pFcb != NULL);

        switch (pIrpSp->Parameters.SetFile.FileInformationClass) 
        {
        case FileAllocationInformation: 
        {
            if ((pFileObject->SectionObjectPointer != NULL) &&(pFileObject->SectionObjectPointer->DataSectionObject != NULL)) 
            {
                LARGE_INTEGER AllocationSize = ((PFILE_ALLOCATION_INFORMATION)pBuffer)->AllocationSize;
                if (AllocationSize.QuadPart < pFcb->AdvancedFCBHeader.AllocationSize.QuadPart &&
                    !MmCanFileBeTruncated(pFileObject->SectionObjectPointer,&AllocationSize)) 
                {
                    status = STATUS_USER_MAPPED_FILE;
                    __leave;
                }
            }
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileAllocationInformation[%lld]\n", ((PFILE_ALLOCATION_INFORMATION)pBuffer)->AllocationSize.QuadPart));
        } 
        break;
        case FileBasicInformation:
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileBasicInformation\n"));
            break;
        case FileDispositionInformation:
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileDispositionInformation\n"));
        case FileDispositionInformationEx:
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileDispositionInformationEx\n"));
            break;
        case FileEndOfFileInformation: 
        {
            if ((pFileObject->SectionObjectPointer != NULL) &&(pFileObject->SectionObjectPointer->DataSectionObject != NULL)) 
            {
                PFILE_END_OF_FILE_INFORMATION pInfoEoF = (PFILE_END_OF_FILE_INFORMATION)pBuffer;
                if (pInfoEoF->EndOfFile.QuadPart < pFcb->AdvancedFCBHeader.FileSize.QuadPart &&
                    !MmCanFileBeTruncated(pFileObject->SectionObjectPointer,&pInfoEoF->EndOfFile)) 
                {
                    status = STATUS_USER_MAPPED_FILE;
                    __leave;
                }

                if (!isPagingIo) 
                {
                    CcFlushCache(&pFcb->SectionObjectPointers, NULL, 0, NULL);
                    PTFSPagingIoLock(pFcb, FALSE);
                    PTFSPagingIoUnlock(pFcb);
                    CcPurgeCacheSection(&pFcb->SectionObjectPointers, NULL, 0, FALSE);
                }
            }
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileEndOfFileInformation[%lld]\n", ((PFILE_END_OF_FILE_INFORMATION)pBuffer)->EndOfFile.QuadPart));
        } 
        break;
        case FileLinkInformation:
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileLinkInformation\n"));
            break;
        case FilePositionInformation: 
        {
            PFILE_POSITION_INFORMATION pFilePosInfo = (PFILE_POSITION_INFORMATION)pIrp->AssociatedIrp.SystemBuffer;

            ASSERT(pFilePosInfo != NULL);

            KdPrint(("[PTFS]::PTFSDispatchSetInformation FilePositionInformation[%lld]\n", pFilePosInfo->CurrentByteOffset.QuadPart));
            pFileObject->CurrentByteOffset = pFilePosInfo->CurrentByteOffset;
            status = STATUS_SUCCESS;
            __leave;
        }
        break;
        case FileRenameInformation:
        case FileRenameInformationEx:
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileRenameInformation\n"));
            pTargetFileObject = pIrpSp->Parameters.SetFile.FileObject;
            if (pTargetFileObject)
            {
                PPTFSCCB pTargetCcb = (PPTFSCCB)pTargetFileObject->FsContext2;
                KdPrint(("[PTFS]::PTFSDispatchSetInformation TargetFileObject specified so perform flush\n"));
                ASSERT(pTargetCcb != NULL);
                FlushAllCachedFcb(pTargetCcb->pFcb, pTargetFileObject);
            }
            FlushAllCachedFcb(pFcb, pFileObject);
            break;
        case FileValidDataLengthInformation:
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileValidDataLengthInformation\n"));
            break;
        default:
            KdPrint(("[PTFS]::PTFSDispatchSetInformation unknown type[%d]\n", pIrpSp->Parameters.SetFile.FileInformationClass));
            break;
        }

        PTFSFcbLock(pFcb, FALSE);
        fcbLocked = TRUE;
        eventLength = sizeof(EVENT_CONTEXT) + pFcb->unstrFileName.Length;
        if (pIrpSp->Parameters.SetFile.Length > MAXULONG - eventLength) 
        {
            KdPrint(("[PTFS]::PTFSDispatchSetInformation Invalid SetFile Length received\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }
        eventLength += pIrpSp->Parameters.SetFile.Length;
        pTargetFileObject = pIrpSp->Parameters.SetFile.FileObject;

        if (pTargetFileObject) 
        {
            KdPrint(("[PTFS]::PTFSDispatchSetInformation FileObject Specified %wZ\n", &pTargetFileObject->FileName));
            if (pTargetFileObject->FileName.Length > MAXULONG - eventLength) 
            {
                KdPrint(("[PTFS]::PTFSDispatchSetInformation Invalid FileObject FileName Length received\n"));
                status = STATUS_INSUFFICIENT_RESOURCES;
                __leave;
            }
            eventLength += pTargetFileObject->FileName.Length;
        }

        pEventContext = AllocateEventContext(pVcb->pDcb, pIrp, eventLength, pCcb);
        if (pEventContext == NULL)
        {
            KdPrint(("[PTFS]::PTFSDispatchSetInformation Failed to AllocateEventContext\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        pEventContext->Context = pCcb->UserContext;
        pEventContext->Operation.SetFile.FileInformationClass = pIrpSp->Parameters.SetFile.FileInformationClass;
        pEventContext->Operation.SetFile.BufferLength = pIrpSp->Parameters.SetFile.Length;
        pEventContext->Operation.SetFile.BufferOffset = FIELD_OFFSET(EVENT_CONTEXT, Operation.SetFile.FileName[0]) + pFcb->unstrFileName.Length + sizeof(WCHAR);

        isRenameOrLink = pIrpSp->Parameters.SetFile.FileInformationClass == FileRenameInformation || 
            pIrpSp->Parameters.SetFile.FileInformationClass == FileLinkInformation || 
            pIrpSp->Parameters.SetFile.FileInformationClass == FileRenameInformationEx;

        if (!isRenameOrLink) 
            RtlCopyMemory((PCHAR)pEventContext + pEventContext->Operation.SetFile.BufferOffset, pIrp->AssociatedIrp.SystemBuffer, pIrpSp->Parameters.SetFile.Length);

        if (isRenameOrLink) 
        {
            PFILE_RENAME_INFORMATION pRenameInfo = (PFILE_RENAME_INFORMATION)pIrp->AssociatedIrp.SystemBuffer;
            PPTFS_RENAME_INFORMATION pRenameContext = (PPTFS_RENAME_INFORMATION)((PCHAR)pEventContext + pEventContext->Operation.SetFile.BufferOffset);

            ASSERT(sizeof(FILE_RENAME_INFORMATION) == sizeof(FILE_LINK_INFORMATION));

            pRenameContext->ReplaceIfExists = pRenameInfo->ReplaceIfExists;
            pRenameContext->FileNameLength = pRenameInfo->FileNameLength;
            RtlCopyMemory(pRenameContext->FileName, pRenameInfo->FileName, pRenameInfo->FileNameLength);

            if (pTargetFileObject != NULL) 
            {
                PFILE_OBJECT pParentFileObject = pTargetFileObject->RelatedFileObject;

                KdPrint(("[PTFS]::PTFSDispatchSetInformation RenameContext Length[%d], FileName[%ws]\n", pRenameContext->FileNameLength, pRenameContext->FileName));
                RtlZeroMemory(pRenameContext->FileName, pRenameContext->FileNameLength);
                if (pParentFileObject != NULL) 
                {
                    RtlCopyMemory(pRenameContext->FileName, pParentFileObject->FileName.Buffer, pParentFileObject->FileName.Length);
                    RtlStringCchCatW(pRenameContext->FileName, NTSTRSAFE_MAX_CCH, L"\\");
                    RtlStringCchCatW(pRenameContext->FileName, NTSTRSAFE_MAX_CCH, pTargetFileObject->FileName.Buffer);
                    pRenameContext->FileNameLength = pTargetFileObject->FileName.Length + pParentFileObject->FileName.Length + sizeof(WCHAR);
                }
                else 
                {
                    RtlCopyMemory(pRenameContext->FileName, pTargetFileObject->FileName.Buffer, pTargetFileObject->FileName.Length);
                    pRenameContext->FileNameLength = pTargetFileObject->FileName.Length;
                }
            }

            if (pIrpSp->Parameters.SetFile.FileInformationClass == FileRenameInformation|| 
                pIrpSp->Parameters.SetFile.FileInformationClass == FileRenameInformationEx) 
            {
                KdPrint(("[PTFS]::PTFSDispatchSetInformation Rename[%wZ ==> %ws], FileCount[%u]\n", pFcb->unstrFileName, pRenameContext->FileName, (ULONG)pFcb->lFileCount));
            }
        }

        pEventContext->Operation.SetFile.FileNameLength = pFcb->unstrFileName.Length;
        RtlCopyMemory(pEventContext->Operation.SetFile.FileName, pFcb->unstrFileName.Buffer, pFcb->unstrFileName.Length);

        status = PTFSCheckOplock(pFcb, pIrp, pEventContext, PTFSOplockComplete, PTFSPrePostIrp);
        if (status != STATUS_SUCCESS) 
        {
            if (status == STATUS_PENDING) 
                KdPrint(("[PTFS]::PTFSDispatchSetInformation FsRtlCheckOplock returned STATUS_PENDING\n"));
            else 
                FreeEventContext(pEventContext);

            __leave;
        }

        status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, 0);
    }
    __finally 
    {
        if (fcbLocked)
            PTFSFcbUnlock(pFcb);

        PTFSCompleteIrpRequest(pIrp, status, 0);
        KdPrint(("[PTFS]::PTFSDispatchSetInformation end\n"));
    }

    return status;
}

VOID
PTFSCompleteQueryInformation(
	IN PIRP_ENTRY pIrpEntry,
	IN PEVENT_INFORMATION pEventInfo
)
{
    PIRP pIrp = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status = STATUS_SUCCESS;
    ULONG info = 0;
    ULONG bufferLen = 0;
    PVOID pBuffer = NULL;
    PPTFSCCB pCcb = NULL;

    KdPrint(("[PTFS]::PTFSCompleteQueryInformation start\n"));

    pIrp = pIrpEntry->pIrp;
    pIrpSp = pIrpEntry->pIrpSp;

    pCcb = pIrpEntry->pFileObject->FsContext2;
    ASSERT(pCcb != NULL);

    pCcb->UserContext = pEventInfo->Context;
    pBuffer = pIrp->AssociatedIrp.SystemBuffer;
    bufferLen = pIrpSp->Parameters.QueryFile.Length;

    if (bufferLen == 0 || pBuffer == NULL || bufferLen < pEventInfo->BufferLength) 
    {
        info = 0;
        status = STATUS_INSUFFICIENT_RESOURCES;
    }
    else 
    {
        ASSERT(pBuffer != NULL);

        RtlZeroMemory(pBuffer, bufferLen);
        RtlCopyMemory(pBuffer, pEventInfo->Buffer, pEventInfo->BufferLength);

        info = pEventInfo->BufferLength;
        status = pEventInfo->Status;

        if (NT_SUCCESS(status) &&
            (pIrpSp->Parameters.QueryFile.FileInformationClass == FileAllInformation ||
            pIrpSp->Parameters.QueryFile.FileInformationClass == FileStandardInformation ||
            pIrpSp->Parameters.QueryFile.FileInformationClass == FileNetworkOpenInformation))
        {
            FSRTL_ADVANCED_FCB_HEADER* pFcbHeader = pIrpEntry->pFileObject->FsContext;
            LONGLONG allocationSize = 0;
            LONGLONG fileSize = 0;

            ASSERT(pFcbHeader != NULL);

            if (pIrpSp->Parameters.QueryFile.FileInformationClass == FileAllInformation) 
            {
                PFILE_ALL_INFORMATION pAllInfo = (PFILE_ALL_INFORMATION)pBuffer;
                allocationSize = pAllInfo->StandardInformation.AllocationSize.QuadPart;
                fileSize = pAllInfo->StandardInformation.EndOfFile.QuadPart;
                pAllInfo->PositionInformation.CurrentByteOffset = pIrpEntry->pFileObject->CurrentByteOffset;
            }
            else if (pIrpSp->Parameters.QueryFile.FileInformationClass == FileStandardInformation) 
            {
                PFILE_STANDARD_INFORMATION pStandardInfo = (PFILE_STANDARD_INFORMATION)pBuffer;
                allocationSize = pStandardInfo->AllocationSize.QuadPart;
                fileSize = pStandardInfo->EndOfFile.QuadPart;
            }
            else if (pIrpSp->Parameters.QueryFile.FileInformationClass == FileNetworkOpenInformation) 
            {
                PFILE_NETWORK_OPEN_INFORMATION pNetworkInfo = (PFILE_NETWORK_OPEN_INFORMATION)pBuffer;
                allocationSize = pNetworkInfo->AllocationSize.QuadPart;
                fileSize = pNetworkInfo->EndOfFile.QuadPart;
            }

            InterlockedExchange64(&pFcbHeader->AllocationSize.QuadPart, allocationSize);
            InterlockedExchange64(&pFcbHeader->FileSize.QuadPart, fileSize);

            KdPrint(("[PTFS]::PTFSCompleteQueryInformation AllocationSize[%llu], EndOfFile[%llu]\n", allocationSize, fileSize));
        }
    }

    PTFSCompleteIrpRequest(pIrp, status, info);

    KdPrint(("[PTFS]::PTFSCompleteQueryInformation end\n"));
}

VOID
PTFSCompleteSetInformation(
	IN PIRP_ENTRY pIrpEntry,
	IN PEVENT_INFORMATION pEventInfo
)
{
    PIRP pIrp = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status;
    ULONG info = 0;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;
    UNICODE_STRING oldFileName;
    BOOLEAN fcbLocked = FALSE;
    BOOLEAN vcbLocked = FALSE;
    FILE_INFORMATION_CLASS infoClass;

    pIrp = pIrpEntry->pIrp;
    status = pEventInfo->Status;

    __try 
    {
        KdPrint(("[PTFS]::PTFSCompleteSetInformation start\n"));

        pIrpSp = pIrpEntry->pIrpSp;
        pCcb = pIrpEntry->pFileObject->FsContext2;
        ASSERT(pCcb != NULL);

        KeEnterCriticalRegion();
        ExAcquireResourceExclusiveLite(&pCcb->Resource, TRUE);

        pFcb = pCcb->pFcb;
        ASSERT(pFcb != NULL);

        info = pEventInfo->BufferLength;
        infoClass = pIrpSp->Parameters.SetFile.FileInformationClass;

        if (!(pIrp->Flags & IRP_PAGING_IO)) 
        {
            if (NT_SUCCESS(status) && infoClass == FileRenameInformation) 
            {
                PTFSVcbLock(pFcb->pVcb, FALSE);
                vcbLocked = TRUE;
            }
            PTFSFcbLock(pFcb, FALSE);
            fcbLocked = TRUE;
        }

        pCcb->UserContext = pEventInfo->Context;
        RtlZeroMemory(&oldFileName, sizeof(UNICODE_STRING));

        if (NT_SUCCESS(status)) 
        {
            if (infoClass == FileDispositionInformation || infoClass == FileDispositionInformationEx) 
            {
                if (pEventInfo->Operation.Delete.DeleteOnClose) 
                {
                    if (!MmFlushImageSection(&pFcb->SectionObjectPointers,MmFlushForDelete)) 
                    {
                        KdPrint(("[PTFS]::PTFSCompleteSetInformation Cannot delete user mapped image\n"));
                        status = STATUS_CANNOT_DELETE;
                    }
                    else 
                    {
                        PTFSCCBFlagsSetBit(pCcb, PTFS_DELETE_ON_CLOSE);
                        PTFSFCBFlagsSetBit(pFcb, PTFS_DELETE_ON_CLOSE);
                        KdPrint(("[PTFS]::PTFSCompleteSetInformation FileObject is DeletePending\n"));
                        pIrpEntry->pFileObject->DeletePending = TRUE;
                    }
                }
                else 
                {
                    PTFSCCBFlagsClearBit(pCcb, PTFS_DELETE_ON_CLOSE);
                    PTFSFCBFlagsClearBit(pFcb, PTFS_DELETE_ON_CLOSE);
                    KdPrint(("[PTFS]::PTFSCompleteSetInformation FileObject is not DeletePending\n"));
                    pIrpEntry->pFileObject->DeletePending = FALSE;
                }
            }

            if (infoClass == FileRenameInformation || infoClass == FileRenameInformationEx) 
            {
                PVOID pBuffer = NULL;

                oldFileName.Buffer = pFcb->unstrFileName.Buffer;
                oldFileName.Length = (USHORT)pFcb->unstrFileName.Length;
                oldFileName.MaximumLength = (USHORT)pFcb->unstrFileName.Length;

                pBuffer = PTFSAllocateZero(pEventInfo->BufferLength + sizeof(WCHAR));
                if (pBuffer == NULL) 
                {
                    status = STATUS_INSUFFICIENT_RESOURCES;
                    ExReleaseResourceLite(&pCcb->Resource);
                    KeLeaveCriticalRegion();
                    __leave;
                }

                pFcb->unstrFileName.Buffer = pBuffer;
                ASSERT(pFcb->unstrFileName.Buffer != NULL);
                RtlCopyMemory(pFcb->unstrFileName.Buffer, pEventInfo->Buffer, pEventInfo->BufferLength);
                pFcb->unstrFileName.Length = (USHORT)pEventInfo->BufferLength;
                pFcb->unstrFileName.MaximumLength = (USHORT)pEventInfo->BufferLength;
                KdPrint(("[PTFS]::PTFSCompleteSetInformation rename also done on Fcb[%wZ]\n", &pFcb->unstrFileName));
            }
        }

        ExReleaseResourceLite(&pCcb->Resource);
        KeLeaveCriticalRegion();

        if (NT_SUCCESS(status)) 
        {
            switch (pIrpSp->Parameters.SetFile.FileInformationClass) 
            {
            case FileAllocationInformation:
                NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_SIZE, FILE_ACTION_MODIFIED);
                break;
            case FileBasicInformation:
                NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_ATTRIBUTES | FILE_NOTIFY_CHANGE_LAST_WRITE | FILE_NOTIFY_CHANGE_LAST_ACCESS | FILE_NOTIFY_CHANGE_CREATION, FILE_ACTION_MODIFIED);
                break;
            case FileDispositionInformation:
            case FileDispositionInformationEx:
                if (pIrpEntry->pFileObject->DeletePending) 
                {
                    if (PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY)) 
                        NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_DIR_NAME, FILE_ACTION_REMOVED);
                    else 
                        NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_FILE_NAME, FILE_ACTION_REMOVED);
                }
                break;
            case FileEndOfFileInformation:
                NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_SIZE, FILE_ACTION_MODIFIED);
                break;
            case FileLinkInformation:
                break;
            case FilePositionInformation:
                break;
            case FileRenameInformationEx:
            case FileRenameInformation:
            {
                KdPrint(("[PTFS]::PTFSCompleteSetInformation FileRenameInformation\n"));

                if (IsInSameDirectory(&oldFileName, &pFcb->unstrFileName)) 
                {
                    if (PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY))
                    {
                        NotifyReportChange0(pFcb, &oldFileName, PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY) ? FILE_NOTIFY_CHANGE_DIR_NAME : FILE_NOTIFY_CHANGE_FILE_NAME, FILE_ACTION_RENAMED_OLD_NAME);
                        NotifyReportChange(pFcb, PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY) ? FILE_NOTIFY_CHANGE_DIR_NAME : FILE_NOTIFY_CHANGE_FILE_NAME, FILE_ACTION_RENAMED_NEW_NAME);
                    }
                }
                else 
                {
                    NotifyReportChange0(pFcb, &oldFileName, PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY) ? FILE_NOTIFY_CHANGE_DIR_NAME : FILE_NOTIFY_CHANGE_FILE_NAME, FILE_ACTION_REMOVED);
                    NotifyReportChange(pFcb, PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY) ? FILE_NOTIFY_CHANGE_DIR_NAME : FILE_NOTIFY_CHANGE_FILE_NAME, FILE_ACTION_ADDED);
                }
                PTFSFree(oldFileName.Buffer);
            } 
            break;
            case FileValidDataLengthInformation:
                NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_SIZE, FILE_ACTION_MODIFIED);
                break;
            default:
                KdPrint(("[PTFS]::PTFSCompleteSetInformation unknown type[%d]\n", pIrpSp->Parameters.SetFile.FileInformationClass));
                break;
            }
        }
    }
    __finally 
    {
        if (fcbLocked) 
            PTFSFcbUnlock(pFcb);

        if (vcbLocked) 
            PTFSVcbUnlock(pFcb->pVcb);

        PTFSCompleteIrpRequest(pIrp, status, info);
        KdPrint(("[PTFS]::PTFSCompleteSetInformation end\n"));
    }
}

