#include "Fcb.h"
#include "Util.h"

const UNICODE_STRING g_KeepAliveFileName = RTL_CONSTANT_STRING(KEEPALIVE_FILE_NAME);
const UNICODE_STRING g_NotificationFileName = RTL_CONSTANT_STRING(NOTIFICATION_FILE_NAME);

PPTFSFCB
AllocateFCB(
	IN PPTFSVCB pVcb,
	IN PWCHAR pwszFileName,
	IN ULONG ulFileNameLength)
{
	PPTFSFCB pFcb = ExAllocateFromLookasideListEx(&g_PTFSFCBLookasideList);

	if (pFcb == NULL && ForceFcbGarbageCollection(pVcb))
		pFcb = ExAllocateFromLookasideListEx(&g_PTFSFCBLookasideList);

	if (pFcb == NULL)
		return NULL;

	ASSERT(pVcb != NULL);

	RtlZeroMemory(pFcb, sizeof(PTFSFCB));

	pFcb->AdvancedFCBHeader.Resource = ExAllocateFromLookasideListEx(&g_PTFSEResourceLookasideList);

	if (pFcb->AdvancedFCBHeader.Resource == NULL)
	{
		ExFreeToLookasideListEx(&g_PTFSFCBLookasideList, pFcb);
		return NULL;
	}

	pFcb->Identifier.FsdIdType = FCB;
	pFcb->Identifier.ulSize = sizeof(PTFSFCB);

	pFcb->pVcb = pVcb;

	ExInitializeResourceLite(&pFcb->PagingIoResource);
	ExInitializeResourceLite(pFcb->AdvancedFCBHeader.Resource);
	ExInitializeFastMutex(&pFcb->AdvancedFCBHeaderMutex);
	FsRtlSetupAdvancedHeader(&pFcb->AdvancedFCBHeader, &pFcb->AdvancedFCBHeaderMutex);

	pFcb->AdvancedFCBHeader.ValidDataLength.QuadPart = MAXLONGLONG;

	pFcb->AdvancedFCBHeader.PagingIoResource = &pFcb->PagingIoResource;

	pFcb->AdvancedFCBHeader.AllocationSize.QuadPart = 4096;
	pFcb->AdvancedFCBHeader.FileSize.QuadPart = 4096;

	pFcb->AdvancedFCBHeader.IsFastIoPossible = FastIoIsNotPossible;
	FsRtlInitializeOplock(PTFSGetFcbOplock(pFcb));

	pFcb->unstrFileName.Buffer = pwszFileName;
	pFcb->unstrFileName.Length = (USHORT)ulFileNameLength;
	pFcb->unstrFileName.MaximumLength = (USHORT)ulFileNameLength;

	InitializeListHead(&pFcb->NextCCB);
	InsertTailList(&pVcb->NextFCB, &pFcb->NextFCB);

	InterlockedIncrement(&pVcb->FcbAllocated);
	InterlockedAnd64(&pVcb->ValidFcbMask, (LONG64)pFcb);
	++pVcb->VolumeMetrics.FcbAllocations;
	return pFcb;
}

PPTFSFCB
GetFCB(
	IN PPTFSVCB pVcb,
	IN PWCHAR pwszFileName,
	IN ULONG ulFileNameLength,
	BOOLEAN bCaseSensitive)
{
	PLIST_ENTRY thisEntry, nextEntry, listHead;
	PPTFSFCB pFcb = NULL;
	UNICODE_STRING fn;

	fn.Length = (USHORT)ulFileNameLength;
	fn.MaximumLength = fn.Length + sizeof(WCHAR);
	fn.Buffer = pwszFileName;

	PTFSVcbLock(pVcb, FALSE);

	listHead = &pVcb->NextFCB;

	for (thisEntry = listHead->Flink; thisEntry != listHead; thisEntry = nextEntry)
	{
		nextEntry = thisEntry->Flink;
		pFcb = CONTAINING_RECORD(thisEntry, PTFSFCB, NextFCB);

        KdPrint(("[PTFS]::GetFCB has entry FileNam[%wZ] FileCount[%d]\n", &pFcb->unstrFileName, pFcb->lFileCount));
		if (pFcb->unstrFileName.Length == ulFileNameLength && RtlEqualUnicodeString(&fn, &pFcb->unstrFileName, !bCaseSensitive))
		{
            KdPrint(("[PTFS]::GetFCB Find existing FCB FileName[%ws]\n", pwszFileName));
            break;
		}

		pFcb = NULL;
	}

	if (pFcb == NULL) 
    {
        KdPrint(("[PTFS]::GetFCB Allocate FCB FileName[%ws]\n", pwszFileName));
		pFcb = AllocateFCB(pVcb, pwszFileName, ulFileNameLength);
		if (pFcb == NULL) 
        {
            KdPrint(("[PTFS]::GetFCB Failed to Allocate FCB FileName[%ws]\n", pwszFileName));
            PTFSFree(pwszFileName);
			PTFSVcbUnlock(pVcb);
			return NULL;
		}

		ASSERT(pFcb != NULL);

		if (RtlEqualUnicodeString(&pFcb->unstrFileName, &g_KeepAliveFileName, FALSE)) 
        {
			pFcb->bIsKeepalive = TRUE;
			pFcb->bBlockUserModeDispatch = TRUE;
		}

		if (RtlEqualUnicodeString(&pFcb->unstrFileName, &g_NotificationFileName, FALSE)) 
			pFcb->bBlockUserModeDispatch = TRUE;
	}
	else 
    {
		CancelFcbGarbageCollection(pFcb);
		PTFSFree(pwszFileName);
	}

	InterlockedIncrement(&pFcb->lFileCount);
	PTFSVcbUnlock(pVcb);
	return pFcb;
}

NTSTATUS
FreeFCB(
	IN PPTFSVCB pVcb,
	IN PPTFSFCB pFcb)
{
	LONG64 validFcbMask = 0;

	ASSERT(pVcb != NULL);
	ASSERT(pFcb != NULL);

	if (GETIDENTIFIERTYPE(pVcb) != VCB)
	{
		KdPrint(("[PTFS]::FreeFCB Invalid VCB[%d]\n", GETIDENTIFIERTYPE(pFcb)));
		return STATUS_INVALID_PARAMETER;
	}

	validFcbMask = pVcb->ValidFcbMask;
	if ((validFcbMask & (LONG64)pFcb) != validFcbMask)
	{
		KdPrint(("[PTFS]::FreeFCB  not match mask %I64x %I64x\n", pFcb, validFcbMask));
		return STATUS_INVALID_PARAMETER;
	}

	if (GETIDENTIFIERTYPE(pFcb) != FCB)
	{
		KdPrint(("[PTFS]::FreeFCB Invalid FCB[%d]\n", GETIDENTIFIERTYPE(pFcb)));
		return STATUS_INVALID_PARAMETER;
	}

	ASSERT(pFcb->pVcb == pVcb);

	PTFSVcbLock(pVcb, FALSE);
	PTFSFcbLock(pFcb, FALSE);

	if (InterlockedDecrement(&pFcb->lFileCount) == 0 && !ScheduleFcbForGarbageCollection(pVcb, pFcb))
		DeleteFcb(pVcb, pFcb);
	else
		PTFSFcbUnlock(pFcb);

	PTFSVcbUnlock(pVcb);
	return STATUS_SUCCESS;
}

VOID
DeleteFcb(
	IN PPTFSVCB pVcb,
	IN PPTFSFCB pFcb)
{
	KdPrint(("[PTFS]::DeleteFcb Invalid FCB[%p]\n", pFcb));

	++pVcb->VolumeMetrics.FcbDeletions;
	RemoveEntryList(&pFcb->NextFCB);
	InitializeListHead(&pFcb->NextCCB);

	PTFSFree(pFcb->unstrFileName.Buffer);

	pFcb->unstrFileName.Buffer = NULL;
	pFcb->unstrFileName.Length = 0;
	pFcb->unstrFileName.MaximumLength = 0;

	FsRtlUninitializeOplock(PTFSGetFcbOplock(pFcb));

	FsRtlTeardownPerStreamContexts(&pFcb->AdvancedFCBHeader);

	pFcb->Identifier.FsdIdType = FREED_FCB;
	PTFSFcbUnlock(pFcb);
	ExDeleteResourceLite(pFcb->AdvancedFCBHeader.Resource);
	ExFreeToLookasideListEx(&g_PTFSEResourceLookasideList, pFcb->AdvancedFCBHeader.Resource);
	ExDeleteResourceLite(&pFcb->PagingIoResource);

	InterlockedIncrement(&pVcb->FcbFreed);
	ExFreeToLookasideListEx(&g_PTFSFCBLookasideList, pFcb);
}

BOOLEAN
ScheduleFcbForGarbageCollection(
	IN PPTFSVCB pVcb,
	IN PPTFSFCB pFcb)
{
	if (pVcb->FcbGarbageCollectorThread == NULL)
		return FALSE;

	if (pFcb->NextGarbageCollectableFcb.Flink != NULL)
		return TRUE;

	pFcb->bGarbageCollectionGracePeriodPassed = FALSE;
	InsertTailList(&pVcb->FcbGarbageList, &pFcb->NextGarbageCollectableFcb);
	KeSetEvent(&pVcb->FcbGarbageListNotEmpty, IO_NO_INCREMENT, FALSE);
	return TRUE;
}

VOID
CancelFcbGarbageCollection(
	IN PPTFSFCB pFcb)
{
	if (pFcb->NextGarbageCollectableFcb.Flink != NULL)
	{
		++pFcb->pVcb->VolumeMetrics.FcbGarbageCollectionCancellations;
		RemoveEntryList(&pFcb->NextGarbageCollectableFcb);
		pFcb->NextGarbageCollectableFcb.Flink = NULL;
		pFcb->bGarbageCollectionGracePeriodPassed = FALSE;
		PTFSFCBFlagsClearBit(pFcb, PTFS_DELETE_ON_CLOSE);
		PTFSFCBFlagsClearBit(pFcb, PTFS_FILE_DIRECTORY);
	}
}

ULONG
DeleteFcbGarbageAndGetRemainingCount(
	IN PPTFSVCB pVcb,
	IN BOOLEAN bForce)
{
	ULONG remainingCount = 0;
	PLIST_ENTRY thisEntry = NULL, nextEntry = NULL;
	PPTFSFCB nextFcb = NULL;

	for (thisEntry = pVcb->FcbGarbageList.Flink; thisEntry != &pVcb->FcbGarbageList; thisEntry = nextEntry)
	{
		nextEntry = thisEntry->Flink;
		nextFcb = CONTAINING_RECORD(thisEntry, PTFSFCB, NextGarbageCollectableFcb);
		if (bForce || nextFcb->bGarbageCollectionGracePeriodPassed)
		{
			RemoveEntryList(thisEntry);
			PTFSFcbLock(nextFcb, FALSE);
			DeleteFcb(pVcb, nextFcb);
		}
		else
		{
			nextFcb->bGarbageCollectionGracePeriodPassed = TRUE;
			++remainingCount;
		}
	}
	ASSERT(!bForce || remainingCount == 0);

	if (remainingCount == 0)
		KeClearEvent(&pVcb->FcbGarbageListNotEmpty);

	return remainingCount;
}

BOOLEAN
ForceFcbGarbageCollection(
	IN PPTFSVCB pVcb)
{
	if (pVcb->FcbGarbageCollectorThread == NULL || IsListEmpty(&pVcb->FcbGarbageList))
		return FALSE;

	++pVcb->VolumeMetrics.ForcedFcbGarbageCollectionPasses;
	DeleteFcbGarbageAndGetRemainingCount(pVcb, TRUE);
	return TRUE;
}


NTSTATUS
WaitForNewFcbGarbage(
	IN PPTFSVCB pVcb)
{
	PVOID events[2];
	NTSTATUS status;
	events[0] = &pVcb->pDcb->ReleaseEvent;
	events[1] = &pVcb->FcbGarbageListNotEmpty;
	status = KeWaitForMultipleObjects(2, events, WaitAny, Executive, KernelMode, FALSE, NULL, NULL);
	return status == STATUS_WAIT_1 ? STATUS_SUCCESS : STATUS_CANCELLED;
}

NTSTATUS
AgeAndDeleteFcbGarbage(
	IN PPTFSVCB pVcb,
	IN PKTIMER Timer)
{
	NTSTATUS status = STATUS_INVALID_PARAMETER;
	ULONG pendingCount = 0;
	PVOID events[2];
	BOOLEAN waited = FALSE;

	events[0] = &pVcb->pDcb->ReleaseEvent;
	events[1] = Timer;
	++pVcb->VolumeMetrics.NormalFcbGarbageCollectionCycles;

	for (;;)
	{
		PTFSVcbLock(pVcb, FALSE);
		++pVcb->VolumeMetrics.NormalFcbGarbageCollectionPasses;
		pendingCount = DeleteFcbGarbageAndGetRemainingCount(pVcb, FALSE);
		PTFSVcbUnlock(pVcb);
		
		if (pendingCount == 0 && waited)
		{
			status = STATUS_SUCCESS;
			break;
		}

		status = KeWaitForMultipleObjects(2, events, WaitAny, Executive, KernelMode, FALSE, NULL, NULL);
		waited = TRUE;

		if (status != STATUS_WAIT_1) 
		{
			status = STATUS_CANCELLED;
			break;
		}
	}
	return status;
}

VOID 
FcbGarbageCollectorThread(
	IN PVOID pVcbParam)
{
	KTIMER timer;
	NTSTATUS status = STATUS_INVALID_PARAMETER;
	LARGE_INTEGER timeout = { 0 };
	PPTFSVCB pVcb = pVcbParam;

	KeInitializeTimerEx(&timer, SynchronizationTimer);
	KeSetTimerEx(&timer, timeout, pVcb->pDcb->ulFcbGarbageCollectionIntervalMs, NULL);

	for (;;) 
	{
		status = WaitForNewFcbGarbage(pVcb);
		if (status != STATUS_SUCCESS) 
			break;

		status = AgeAndDeleteFcbGarbage(pVcb, &timer);

		if (status != STATUS_SUCCESS)
			break;
	}
	KeCancelTimer(&timer);
}

void
StartFcbGarbageCollector(
	IN PPTFSVCB pVcb)
{
	NTSTATUS status = STATUS_INVALID_PARAMETER;
	HANDLE thread = NULL;
	pVcb->FcbGarbageCollectorThread = NULL;

	if (pVcb->pDcb->ulFcbGarbageCollectionIntervalMs == 0) {
		return;
	}
	status = PsCreateSystemThread(&thread, THREAD_ALL_ACCESS, NULL, NULL, NULL,
			(PKSTART_ROUTINE)FcbGarbageCollectorThread, pVcb);

	if (!NT_SUCCESS(status))
		return;

	ObReferenceObjectByHandle(thread, THREAD_ALL_ACCESS, NULL, KernelMode, (PVOID*)&pVcb->FcbGarbageCollectorThread, NULL);
	ZwClose(thread);
}