#include "event.h"
#include <Ntstrsafe.h>
#include "init.h"
#include "util.h"
#include "notification.h"
#include "timeout.h"
#include "create.h"
#include "Directory.h"
#include "read.h"
#include "write.h"
#include "fileinfo.h"
#include "volume.h"
#include "cleanup.h"
#include "lock.h"
#include "flush.h"
#include "security.h"
#include "volume.h"
#include "Fscontrol.h"
#include "mount.h"
#include "irp_buffer_helper.h"

#ifdef ALLOC_PRAGMA
#pragma alloc_text(PAGE, PTFSOplockComplete)
#endif

VOID
PTFSCreateIrpCancelRoutine(
	IN OUT PDEVICE_OBJECT pDeviceObject,
	IN OUT PIRP pIrp
)
{
	PIRP_ENTRY pIrpEntry = pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_IRP_ENTRY];
	IoReleaseCancelSpinLock(pIrp->CancelIrql);
	if (pIrpEntry != NULL) 
    {
        PPTFSVCB pVcb = pDeviceObject->DeviceExtension;
        PPTFSDCB pDcb = pVcb->pDcb;
        pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_EVENT] = NULL;
		InterlockedAnd64(&pIrpEntry->liTickCount.QuadPart, 0);
		pIrpEntry->AsyncStatus = STATUS_CANCELLED;
		KeSetEvent(&pDcb->ForceTimeoutEvent, 0, FALSE);
	}
}

VOID
PTFSIrpCancelRoutine(
	IN OUT PDEVICE_OBJECT pDeviceObject,
	IN OUT PIRP pIrp
)
{
	KIRQL oldIrql;
	PIRP_ENTRY pIrpEntry = NULL;
	ULONG ulSerialNumber = 0;
	PIO_STACK_LOCATION pIrpSp = NULL;

	UNREFERENCED_PARAMETER(pDeviceObject);

    KdPrint(("[PTFS]::PTFSIrpCancelRoutine Start\n"));

	IoReleaseCancelSpinLock(pIrp->CancelIrql);

	pIrpEntry = pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_IRP_ENTRY];

	if (pIrpEntry != NULL) 
    {
		PKSPIN_LOCK lock = &pIrpEntry->pIrpList->ListSpinLock;
		
        ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);
        KeAcquireSpinLock(lock, &oldIrql);
		pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
		ASSERT(pIrpSp != NULL);

		ulSerialNumber = pIrpEntry->ulSerialNumber;

		RemoveEntryList(&pIrpEntry->ListEntry);
		InitializeListHead(&pIrpEntry->ListEntry);

		if (pIrpSp->MajorFunction == IRP_MJ_WRITE) 
        {
			PVOID pEventContext = pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_EVENT];

			if (pEventContext != NULL)
				FreeEventContext(pEventContext);

			pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_EVENT] = NULL;
		}

		if (IsListEmpty(&pIrpEntry->pIrpList->ListHead)) 
			KeClearEvent(&pIrpEntry->pIrpList->NotEmptyEvent);

		pIrpEntry->pIrp = NULL;

		if (pIrpEntry->bCancelRoutineFreeMemory == FALSE)
			InitializeListHead(&pIrpEntry->ListEntry);
		else 
        {
			FreeIrpEntry(pIrpEntry);
			pIrpEntry = NULL;
		}

		pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_IRP_ENTRY] = NULL;
		KeReleaseSpinLock(lock, oldIrql);
	}

    KdPrint(("[PTFS]::PTFSIrpCancelRoutine canceled IRP SerialNumber[%d]\n", ulSerialNumber));

	PTFSCompleteIrpRequest(pIrp, STATUS_CANCELLED, 0);
    KdPrint(("[PTFS]::PTFSIrpCancelRoutine End\n"));
}

VOID
PTFSOplockComplete(
	IN PVOID pContext,
	IN PIRP pIrp
)
{
	PIO_STACK_LOCATION pIrpSp = NULL;

	KdPrint(("[PTFS]::PTFSOplockComplete Start\n"));

	PAGED_CODE();

	pIrpSp = IoGetCurrentIrpStackLocation(pIrp);

	if (pIrp->IoStatus.Status == STATUS_SUCCESS)
		RegisterPendingIrp(pIrpSp->DeviceObject, pIrp, (PEVENT_CONTEXT)pContext, 0);
	else
		PTFSCompleteIrpRequest(pIrp, pIrp->IoStatus.Status, 0);

	KdPrint(("[PTFS]::PTFSOplockComplete End\n"));
}

VOID
PTFSPrePostIrp(
	IN PVOID pContext,
	IN PIRP pIrp
)
{
	KdPrint(("[PTFS]::PTFSPrePostIrp Start\n"));
	
    UNREFERENCED_PARAMETER(pContext);
	IoMarkIrpPending(pIrp);

    KdPrint(("[PTFS]::PTFSPrePostIrp End\n"));
}

NTSTATUS
RegisterPendingIrpMain(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp,
	IN ULONG SerialNumber,
	IN PIRP_LIST pIrpList,
	IN ULONG Flags,
	IN ULONG CheckMount,
	IN NTSTATUS CurrentStatus
)
{
	PIRP_ENTRY pIrpEntry = NULL;
	PIO_STACK_LOCATION pIrpSp = NULL;
	PPTFSVCB pVcb = NULL;
	KIRQL oldIrql;

	KdPrint(("[PTFS]::RegisterPendingIrpMain Start\n"));

	if (GETIDENTIFIERTYPE(pDeviceObject->DeviceExtension) == VCB)
	{
		pVcb = pDeviceObject->DeviceExtension;
		if (CheckMount && IsUnmountPendingVcb(pVcb))
		{
			KdPrint(("[PTFS]::RegisterPendingIrpMain device is not mounted\n"));
			return STATUS_NO_SUCH_DEVICE;
		}
	}

	pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
	pIrpEntry = AllocateIrpEntry();

	if (NULL == pIrpEntry)
	{
		KdPrint(("[PTFS]::RegisterPendingIrpMain failed to allocate IRP_ENTRY\n"));
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	RtlZeroMemory(pIrpEntry, sizeof(IRP_ENTRY));

	InitializeListHead(&pIrpEntry->ListEntry);

	pIrpEntry->ulSerialNumber = SerialNumber;
	pIrpEntry->pFileObject = pIrpSp->FileObject;
	pIrpEntry->pIrp = pIrp;
	pIrpEntry->pIrpSp = pIrpSp;
	pIrpEntry->pIrpList = pIrpList;
	pIrpEntry->ulFlags = Flags;
	pIrpEntry->AsyncStatus = CurrentStatus;

	if (pVcb)
	{
		ExAcquireResourceExclusiveLite(&pVcb->pDcb->Resource, TRUE);
		UpdateTimeout(&pIrpEntry->liTickCount, pVcb->pDcb->ulIrpTimeout);
		ExReleaseResourceLite(&pVcb->pDcb->Resource);
	}
	else
		UpdateTimeout(&pIrpEntry->liTickCount, PTFS_IRP_PENDING_TIMEOUT);

	ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);
	KeAcquireSpinLock(&pIrpList->ListSpinLock, &oldIrql);

	if (pIrpSp->MajorFunction == IRP_MJ_CREATE)
		IoSetCancelRoutine(pIrp, PTFSCreateIrpCancelRoutine);
	else
		IoSetCancelRoutine(pIrp, PTFSIrpCancelRoutine);

	if (pIrp->Cancel)
	{
		if (IoSetCancelRoutine(pIrp, NULL) != NULL)
		{
			KeReleaseSpinLock(&pIrpList->ListSpinLock, oldIrql);
			FreeIrpEntry(pIrpEntry);
			return STATUS_CANCELLED;
		}
	}

	IoMarkIrpPending(pIrp);

	InsertTailList(&pIrpList->ListHead, &pIrpEntry->ListEntry);
	pIrpEntry->bCancelRoutineFreeMemory = FALSE;
	pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_IRP_ENTRY] = pIrpEntry;
	KeSetEvent(&pIrpList->NotEmptyEvent, IO_NO_INCREMENT, FALSE);
	KeReleaseSpinLock(&pIrpList->ListSpinLock, oldIrql);

	KdPrint(("[PTFS]::RegisterPendingIrpMain End\n"));
	return STATUS_PENDING;
}

NTSTATUS
RegisterPendingIrp(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp,
	IN PEVENT_CONTEXT pEventContext,
	IN ULONG Flags
)
{
	PPTFSVCB pVcb = pDeviceObject->DeviceExtension;
	NTSTATUS status;

    KdPrint(("[PTFS]::RegisterPendingIrp Start\n"));

	if (GETIDENTIFIERTYPE(pVcb) != VCB) 
    {
        KdPrint(("[PTFS]::RegisterPendingIrp IdentifierType is not VCB\n"));
        return STATUS_INVALID_PARAMETER;
	}

	status = RegisterPendingIrpMain(pDeviceObject, pIrp, pEventContext->SerialNumber, &pVcb->pDcb->PendingIrpList, Flags, TRUE, STATUS_SUCCESS);

	if (status == STATUS_PENDING)
		EventNotification(&pVcb->pDcb->NotifyEventIrpList, pEventContext);
	else
		FreeEventContext(pEventContext);

    KdPrint(("[PTFS]::RegisterPendingIrp End\n"));
    return status;
}

VOID
RegisterPendingRetryIrp(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
    PIO_STACK_LOCATION pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
    PPTFSVCB pVcb = pDeviceObject->DeviceExtension;
	
    if (GETIDENTIFIERTYPE(pVcb) != VCB)
		return;

	if (pIrpSp->MajorFunction == IRP_MJ_CREATE) 
    {
		PPTFSCCB pCcb = pIrpSp->FileObject->FsContext2;
		ASSERT(pCcb != NULL);
		PTFSCCBFlagsSetBit(pCcb, PTFS_RETRY_CREATE);
	}

	RegisterPendingIrpMain(pDeviceObject, pIrp, 0, &pVcb->pDcb->PendingRetryIrpList, 0, TRUE, STATUS_SUCCESS);
}

VOID
RegisterAsyncCreateFailure(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp,
	IN NTSTATUS Status
)
{
	PPTFSVCB pVcb = pDeviceObject->DeviceExtension;

    if (GETIDENTIFIERTYPE(pVcb) != VCB)
    {
        KdPrint(("[PTFS]::RegisterAsyncCreateFailure IdentifierType is not VCB\n"));
        return;
    }

	RegisterPendingIrpMain(pDeviceObject, pIrp, 0, &pVcb->pDcb->PendingIrpList, 0, TRUE, Status);
	KeSetEvent(&pVcb->pDcb->ForceTimeoutEvent, 0, FALSE);
}

NTSTATUS
RegisterPendingIrpForEvent(
	IN PDEVICE_OBJECT pDeviceObject,
	IN OUT PIRP pIrp
)
{
	PPTFSVCB pVcb = pDeviceObject->DeviceExtension;

	if (GETIDENTIFIERTYPE(pVcb) != VCB)
	{
		KdPrint(("[PTFS]::RegisterPendingIrpForEvent IdentifierType is not VCB\n"));
		return STATUS_INVALID_PARAMETER;
	}

	if (IsUnmountPendingVcb(pVcb))
	{
		KdPrint(("[PTFS]::RegisterPendingIrpForEvent Volume is dismounted\n"));
		return STATUS_NO_SUCH_DEVICE;
	}

	pVcb->bHasEventWait = TRUE;

	return RegisterPendingIrpMain(pDeviceObject, pIrp, 0, &pVcb->pDcb->PendingEventIrpList, 0, TRUE, STATUS_SUCCESS);
}

NTSTATUS
RegisterPendingIrpForService(
	IN PDEVICE_OBJECT pDeviceObject,
	IN OUT PIRP pIrp
)
{
	PPTFS_GLOBAL pPTFSGlobal = NULL;

    KdPrint(("[PTFS]::RegisterPendingIrpForService Start\n"));
    pPTFSGlobal = pDeviceObject->DeviceExtension;

	if (GETIDENTIFIERTYPE(pPTFSGlobal) != PGL)
    {
        KdPrint(("[PTFS]::RegisterPendingIrpForEvent IdentifierType is not DGL\n"));
        return STATUS_INVALID_PARAMETER;
	}

	return RegisterPendingIrpMain(pDeviceObject, pIrp, 0, &pPTFSGlobal->PendingServiceIrpList, 0, FALSE, STATUS_SUCCESS);
}

NTSTATUS
PTFSCompleteIrp(
	IN PDEVICE_OBJECT pDeviceObject,
	IN OUT PIRP pIrp
)
{
	KIRQL oldIrql;
	PLIST_ENTRY thisEntry, nextEntry, listHead;
	PIRP_ENTRY pIrpEntry = NULL;
	PPTFSVCB pVcb = NULL;
	PEVENT_INFORMATION pEventInfo = NULL;

	pEventInfo = (PEVENT_INFORMATION)pIrp->AssociatedIrp.SystemBuffer;
	ASSERT(pEventInfo != NULL);

	pVcb = pDeviceObject->DeviceExtension;
	
    if (GETIDENTIFIERTYPE(pVcb) != VCB)
    {
        KdPrint(("[PTFS]::PTFSCompleteIrp IdentifierType is not VCB\n"));
        return STATUS_INVALID_PARAMETER;
	}

	if (IsUnmountPendingVcb(pVcb)) 
    {
        KdPrint(("[PTFS]::PTFSCompleteIrp Volume is not mounted\n"));
        return STATUS_NO_SUCH_DEVICE;
	}

	ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);

	KeAcquireSpinLock(&pVcb->pDcb->PendingIrpList.ListSpinLock, &oldIrql);
	listHead = &pVcb->pDcb->PendingIrpList.ListHead;

	for (thisEntry = listHead->Flink; thisEntry != listHead; thisEntry = nextEntry) 
    {
        PIO_STACK_LOCATION pIrpSp = NULL;
        PIRP pTmpIrp = NULL;

		nextEntry = thisEntry->Flink;
		pIrpEntry = CONTAINING_RECORD(thisEntry, IRP_ENTRY, ListEntry);
		if (pIrpEntry->ulSerialNumber != pEventInfo->SerialNumber)
			continue;

		RemoveEntryList(thisEntry);

		pTmpIrp = pIrpEntry->pIrp;

		if (pTmpIrp == NULL)
        {
			ASSERT(pIrpEntry->bCancelRoutineFreeMemory == FALSE);
			FreeIrpEntry(pIrpEntry);
			pIrpEntry = NULL;
			break;
		}

		if (IoSetCancelRoutine(pTmpIrp, NULL) == NULL)
        {
			InitializeListHead(&pIrpEntry->ListEntry);
			pIrpEntry->bCancelRoutineFreeMemory = TRUE;
			break;
		}

		pIrpSp = pIrpEntry->pIrpSp;

		ASSERT(pIrpSp != NULL);

		pIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_IRP_ENTRY] = NULL;
		KeReleaseSpinLock(&pVcb->pDcb->PendingIrpList.ListSpinLock, oldIrql);

		if (IsUnmountPendingVcb(pVcb))
        {
            KdPrint(("[PTFS]::PTFSCompleteIrp Volume is not mounted(second)\n"));
            return STATUS_NO_SUCH_DEVICE;
		}

		if (pEventInfo->Status == STATUS_PENDING)
            KdPrint(("[PTFS]::PTFSCompleteIrp !!WARNING!! Do not return STATUS_PENDING\n"));

		switch (pIrpSp->MajorFunction) 
        {
        case IRP_MJ_CREATE:
            PTFSCompleteCreate(pIrpEntry, pEventInfo);
            break;
        case IRP_MJ_DIRECTORY_CONTROL:
            PTFSCompleteDirectoryControl(pIrpEntry, pEventInfo);
			break;
		case IRP_MJ_READ:
            PTFSCompleteRead(pIrpEntry, pEventInfo);
			break;
		case IRP_MJ_WRITE:
            PTFSCompleteWrite(pIrpEntry, pEventInfo);
			break;
		case IRP_MJ_QUERY_INFORMATION:
            PTFSCompleteQueryInformation(pIrpEntry, pEventInfo);
			break;
        case IRP_MJ_SET_INFORMATION:
            PTFSCompleteSetInformation(pIrpEntry, pEventInfo);
            break;
		case IRP_MJ_QUERY_VOLUME_INFORMATION:
            PTFSCompleteQueryVolumeInformation(pIrpEntry, pEventInfo, pDeviceObject);
			break;
		case IRP_MJ_CLEANUP:
            PTFSCompleteCleanup(pIrpEntry, pEventInfo);
			break;
		case IRP_MJ_LOCK_CONTROL:
            PTFSCompleteLock(pIrpEntry, pEventInfo);
			break;
		case IRP_MJ_FLUSH_BUFFERS:
            PTFSCompleteFlush(pIrpEntry, pEventInfo);
			break;
		case IRP_MJ_QUERY_SECURITY:
            PTFSCompleteQuerySecurity(pIrpEntry, pEventInfo);
			break;
		case IRP_MJ_SET_SECURITY:
            PTFSCompleteSetSecurity(pIrpEntry, pEventInfo);
			break;
		default:
            KdPrint(("[PTFS]::PTFSCompleteIrp Unknown IRP Major[%d]\n", pIrpSp->MajorFunction));
            break;
		}

		FreeIrpEntry(pIrpEntry);
		pIrpEntry = NULL;

		return STATUS_SUCCESS;
	}

	KeReleaseSpinLock(&pVcb->pDcb->PendingIrpList.ListSpinLock, oldIrql);
	return STATUS_SUCCESS;
}

VOID
RemoveSessionDevices(
	IN PPTFS_GLOBAL pPTFSGlobal,
	IN ULONG sessionId
)
{
	PDEVICE_ENTRY pFoundEntry = NULL;
	BOOLEAN bIsDone = FALSE;

	KdPrint(("[PTFS]::RemoveSessionDevices Start\n"));

	if (sessionId == -1)
		return;

	do 
    {
		pFoundEntry = FindDeviceForDeleteBySessionId(pPTFSGlobal, sessionId);
		if (pFoundEntry != NULL) 
        {
			DeleteMountPointSymbolicLink(&pFoundEntry->unstrMountPoint);
			pFoundEntry->ulSessionId = (ULONG)-1;
			pFoundEntry->unstrMountPoint.Buffer = NULL;
		}
		else 
			bIsDone = TRUE;

	} while (!bIsDone);

    KdPrint(("[PTFS]::RemoveSessionDevices End\n"));
}

NTSTATUS
PTFSEventStart(
	IN PDEVICE_OBJECT pDeviceObject,
	IN OUT PIRP pIrp
)
{
	ULONG outBufferLen;
	PIO_STACK_LOCATION pIrpSp = NULL;
	PEVENT_START pEventStart = NULL;
	PEVENT_DRIVER_INFO pEventDriverInfo = NULL;
	PPTFS_GLOBAL pPTFSGlobal = NULL;
	PPTFSDCB pDcb = NULL;
	NTSTATUS status;
	DEVICE_TYPE deviceType;
	ULONG deviceCharacteristics = 0;
	WCHAR* baseGuidString = NULL;
	GUID baseGuid = PTFS_BASE_GUID;
	UNICODE_STRING unicodeGuid;
	ULONG deviceNamePos;
	BOOLEAN useMountManager = FALSE;
	BOOLEAN mountGlobally = TRUE;
	BOOLEAN fileLockUserMode = FALSE;
	BOOLEAN oplocksDisabled = FALSE;
	BOOLEAN fcbGcEnabled = FALSE;
	ULONG sessionId = (ULONG)-1;
	BOOLEAN startFailure = FALSE;
	BOOLEAN isMountPointDriveLetter = FALSE;
	PTFS_CONTROL ptfsControl;
	PMOUNT_ENTRY pMountEntry = NULL;

	KdPrint(("[PTFS]::PTFSEventStart Start\n"));

	pPTFSGlobal = pDeviceObject->DeviceExtension;
	if (GETIDENTIFIERTYPE(pPTFSGlobal) != PGL) 
	{
		KdPrint(("[PTFS]::PTFSEventStart IdentifierType is not DGL\n"));
		return STATUS_INVALID_PARAMETER;
	}

	GET_IRP_BUFFER(pIrp, pEventStart)
	pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
	outBufferLen = pIrpSp->Parameters.DeviceIoControl.OutputBufferLength;

	if (outBufferLen != sizeof(EVENT_DRIVER_INFO) || !pEventStart) 
	{
		KdPrint(("[PTFS]::PTFSEventStart Buffer IN/OUT received do not match the expected size.\n"));
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	pEventStart = PTFSAllocate(sizeof(EVENT_START));
	baseGuidString = PTFSAllocateZero(64 * sizeof(WCHAR));

	if (pEventStart == NULL || baseGuidString == NULL) 
	{
		if (pEventStart) 
			PTFSFree(pEventStart);
		if (baseGuidString) 
			PTFSFree(baseGuidString);

		KdPrint(("[PTFS]::PTFSEventStart Failed to allocate buffers in event start.\n"));
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	RtlCopyMemory(pEventStart, pIrp->AssociatedIrp.SystemBuffer,sizeof(EVENT_START));

	if (pEventStart->UserVersion != PTFS_DRIVER_VERSION)
	{
		KdPrint(("[PTFS]::PTFSEventStart Invalid Driver Version\n"));
		startFailure = TRUE;
	}

	if (SearchStringWChar(pEventStart->MountPoint,sizeof(pEventStart->MountPoint), '\0') == -1 ||
		SearchStringWChar(pEventStart->UNCName,sizeof(pEventStart->UNCName), '\0') == -1) 
	{
		KdPrint(("[PTFS]::PTFSEventStart MountPoint / UNCName provided are not null\n"));
		startFailure = TRUE;
	}

	pEventDriverInfo = pIrp->AssociatedIrp.SystemBuffer;
	if (startFailure) 
	{
		pEventDriverInfo->DriverVersion = PTFS_DRIVER_VERSION;
		pEventDriverInfo->Status = PTFS_START_FAILED;
		pIrp->IoStatus.Status = STATUS_SUCCESS;
		pIrp->IoStatus.Information = sizeof(EVENT_DRIVER_INFO);
		PTFSFree(pEventStart);
		PTFSFree(baseGuidString);
		return STATUS_SUCCESS;
	}

	switch (pEventStart->DeviceType) 
	{
	case PTFS_DISK_FILE_SYSTEM:
		deviceType = FILE_DEVICE_DISK_FILE_SYSTEM;
		break;
	case PTFS_NETWORK_FILE_SYSTEM:
		deviceType = FILE_DEVICE_NETWORK_FILE_SYSTEM;
		deviceCharacteristics |= FILE_REMOTE_DEVICE;
		break;
	default:
		KdPrint(("[PTFS]::PTFSEventStart Unknown device type[%d]\n", pEventStart->DeviceType));
		deviceType = FILE_DEVICE_DISK_FILE_SYSTEM;
	}

	if (pEventStart->Flags & PTFS_EVENT_REMOVABLE)
		deviceCharacteristics |= FILE_REMOVABLE_MEDIA;

	if (pEventStart->Flags & PTFS_EVENT_WRITE_PROTECT)
		deviceCharacteristics |= FILE_READ_ONLY_DEVICE;

	if (pEventStart->Flags & PTFS_EVENT_MOUNT_MANAGER) 
		useMountManager = TRUE;

	if (pEventStart->Flags & PTFS_EVENT_CURRENT_SESSION) 
	{
		KdPrint(("[PTFS]::PTFSEventStart Mounting on current session only\n"));
		mountGlobally = FALSE;
		sessionId = GetCurrentSessionId(pIrp);
	}

	if (pEventStart->Flags & PTFS_EVENT_FILELOCK_USER_MODE)
	{
		KdPrint(("[PTFS]::PTFSEventStart FileLock in User Mode\n"));
		fileLockUserMode = TRUE;
	}

	if (pEventStart->Flags & PTFS_EVENT_DISABLE_OPLOCKS) 
	{
		KdPrint(("[PTFS]::PTFSEventStart OpLocks disabled\n"));
		oplocksDisabled = TRUE;
	}

	if (pEventStart->Flags & PTFS_EVENT_ENABLE_FCB_GC) 
	{
		KdPrint(("[PTFS]::PTFSEventStart FCB GC enabled\n"));
		fcbGcEnabled = TRUE;
	}

	KeEnterCriticalRegion();
	ExAcquireResourceExclusiveLite(&pPTFSGlobal->Resource, TRUE);

	RtlZeroMemory(&ptfsControl, sizeof(PTFS_CONTROL));
	RtlStringCchCopyW(ptfsControl.MountPoint, MAXIMUM_FILENAME_LENGTH,L"\\DosDevices\\");

	if (wcslen(pEventStart->MountPoint) == 1) 
	{
		ptfsControl.MountPoint[12] = towupper(pEventStart->MountPoint[0]);
		ptfsControl.MountPoint[13] = L':';
		ptfsControl.MountPoint[14] = L'\0';
	}
	else 
		RtlStringCchCatW(ptfsControl.MountPoint, MAXIMUM_FILENAME_LENGTH,pEventStart->MountPoint);

	ptfsControl.SessionId = sessionId;

	KdPrint(("[PTFS]::PTFSEventStart Checking for MountPoint %ls\n", ptfsControl.MountPoint));
	pMountEntry = FindMountEntry(pPTFSGlobal, &ptfsControl, FALSE);

	if (pMountEntry != NULL) 
	{
		KdPrint(("[PTFS]::PTFSEventStart Mount point exists %ls\n", ptfsControl.MountPoint));
		pEventDriverInfo->DriverVersion = PTFS_DRIVER_VERSION;
		pEventDriverInfo->Status = PTFS_START_FAILED;
		pIrp->IoStatus.Status = STATUS_SUCCESS;
		pIrp->IoStatus.Information = sizeof(EVENT_DRIVER_INFO);
		ExReleaseResourceLite(&pPTFSGlobal->Resource);
		KeLeaveCriticalRegion();
		PTFSFree(pEventStart);
		PTFSFree(baseGuidString);
		return STATUS_SUCCESS;
	}

	baseGuid.Data2 = (USHORT)(pPTFSGlobal->ulMountId & 0xFFFF) ^ baseGuid.Data2;
	baseGuid.Data3 = (USHORT)(pPTFSGlobal->ulMountId >> 16) ^ baseGuid.Data3;

	status = RtlStringFromGUID(&baseGuid, &unicodeGuid);
	if (!NT_SUCCESS(status)) 
	{
		KdPrint(("[PTFS]::PTFSEventStart Failed to RtlStringFromGUID[0x%x]\n", status));
		ExReleaseResourceLite(&pPTFSGlobal->Resource);
		KeLeaveCriticalRegion();
		PTFSFree(pEventStart);
		PTFSFree(baseGuidString);
		return status;
	}

	RtlStringCchCopyW(baseGuidString, 64, unicodeGuid.Buffer);
	RtlFreeUnicodeString(&unicodeGuid);

	InterlockedIncrement((LONG*)&pPTFSGlobal->ulMountId);

	status = PTFSCreateDiskDevice(pDeviceObject->DriverObject, pPTFSGlobal->ulMountId, pEventStart->MountPoint,
		pEventStart->UNCName, sessionId, baseGuidString, pPTFSGlobal, deviceType,
		deviceCharacteristics, mountGlobally, useMountManager, &pDcb);

	if (!NT_SUCCESS(status)) 
	{
		KdPrint(("[PTFS]::PTFSEventStart Failed to CreateDiskDevice[0x%x]\n", status));
		ExReleaseResourceLite(&pPTFSGlobal->Resource);
		KeLeaveCriticalRegion();
		PTFSFree(pEventStart);
		PTFSFree(baseGuidString);
		return status;
	}

	isMountPointDriveLetter = IsMountPointDriveLetter(pDcb->punstrMountPoint);

	pDcb->bOplocksDisabled = oplocksDisabled;
	pDcb->usFileLockInUserMode = fileLockUserMode;
	pDcb->ulFcbGarbageCollectionIntervalMs = fcbGcEnabled ? 2000 : 0;
	pEventDriverInfo->DeviceNumber = pPTFSGlobal->ulMountId;
	pEventDriverInfo->MountId = pPTFSGlobal->ulMountId;
	pEventDriverInfo->Status = PTFS_MOUNTED;
	pEventDriverInfo->DriverVersion = PTFS_DRIVER_VERSION;

	deviceNamePos = pDcb->punstrSymbolicLinkName->Length / sizeof(WCHAR) - 1;
	deviceNamePos = SearchWcharinUnicodeStringWithUlong(pDcb->punstrSymbolicLinkName, L'\\', deviceNamePos, 0);

	RtlStringCchCopyW(pEventDriverInfo->DeviceName, sizeof(pEventDriverInfo->DeviceName) / sizeof(WCHAR), &(pDcb->punstrSymbolicLinkName->Buffer[deviceNamePos]));

	pDcb->ulIrpTimeout = PTFS_IRP_PENDING_TIMEOUT;

	if (pEventStart->IrpTimeout > 0) 
	{
		if (pEventStart->IrpTimeout > PTFS_IRP_PENDING_TIMEOUT_RESET_MAX) 
			pEventStart->IrpTimeout = PTFS_IRP_PENDING_TIMEOUT_RESET_MAX;

		if (pEventStart->IrpTimeout < PTFS_IRP_PENDING_TIMEOUT) 
			pEventStart->IrpTimeout = PTFS_IRP_PENDING_TIMEOUT;

		pDcb->ulIrpTimeout = pEventStart->IrpTimeout;
	}

	KdPrint(("[PTFS]::PTFSEventStart Event start using mount ID: %d; device name: %s.\n", pDcb->ulMountId, pEventDriverInfo->DeviceName));

	pDcb->usUseAltStream = 0;
	
	if (pEventStart->Flags & PTFS_EVENT_ALTERNATIVE_STREAM_ON) 
	{
		KdPrint(("[PTFS]::PTFSEventStart ALT_STREAM_ON\n"));
		pDcb->usUseAltStream = 1;
	}

	StartEventNotificationThread(pDcb);
	ExReleaseResourceLite(&pPTFSGlobal->Resource);
	KeLeaveCriticalRegion();

	IoVerifyVolume(pDcb->pDeviceObject, FALSE);

	if (useMountManager) 
	{
		if (!isMountPointDriveLetter && pDcb->punstrPersistentSymbolicLinkName) 
		{
			ULONG setReparseInputlength = 0;
			PCHAR setReparseInput = CreateSetReparsePointRequest(pDcb->punstrPersistentSymbolicLinkName, &setReparseInputlength);
			
			if (setReparseInput) 
			{
				status = SendDirectoryFsctl(pDeviceObject, pDcb->punstrMountPoint, FSCTL_SET_REPARSE_POINT, setReparseInput, setReparseInputlength);
				PTFSFree(setReparseInput);

				if (NT_SUCCESS(status))
					NotifyDirectoryMountPointCreated(pDcb);
				else 
					KdPrint(("[PTFS]::PTFSEventStart Failed to set reparse point on MountPoint %wZ\n", pDcb->punstrMountPoint));
			}
		}
	}

	pIrp->IoStatus.Status = STATUS_SUCCESS;
	pIrp->IoStatus.Information = sizeof(EVENT_DRIVER_INFO);

	PTFSFree(pEventStart);
	PTFSFree(baseGuidString);

	KdPrint(("[PTFS]::PTFSEventStart End\n"));
	return pIrp->IoStatus.Status;
}

NTSTATUS
PTFSEventWrite(
	IN PDEVICE_OBJECT pDeviceObject,
	IN OUT PIRP pIrp
)
{
	KIRQL oldIrql;
	PLIST_ENTRY thisEntry, nextEntry, listHead;
	PIRP_ENTRY pIrpEntry = NULL;
	PPTFSVCB pVcb = NULL;
	PEVENT_INFORMATION pEventInfo = NULL;
	PIRP pWriteIrp = NULL;

	GET_IRP_BUFFER_OR_RETURN(pIrp, pEventInfo)

	KdPrint(("[PTFS]::PTFSEventWrite start EventInfo[0x%x]\n", pEventInfo->SerialNumber));

	pVcb = pDeviceObject->DeviceExtension;

	if (GETIDENTIFIERTYPE(pVcb) != VCB) 
	{
		KdPrint(("[PTFS]::PTFSEventWrite IdentifierType is not VCB\n"));
		return STATUS_INVALID_PARAMETER;
	}

	ASSERT(KeGetCurrentIrql() <= DISPATCH_LEVEL);

	KeAcquireSpinLock(&pVcb->pDcb->PendingIrpList.ListSpinLock, &oldIrql);

	listHead = &pVcb->pDcb->PendingIrpList.ListHead;

	for (thisEntry = listHead->Flink; thisEntry != listHead;thisEntry = nextEntry) 
	{
		PIO_STACK_LOCATION pWriteIrpSp = NULL , pEventIrpSp = NULL;
		PEVENT_CONTEXT pEventContext;
		ULONG info = 0;
		NTSTATUS status;

		nextEntry = thisEntry->Flink;
		pIrpEntry = CONTAINING_RECORD(thisEntry, IRP_ENTRY, ListEntry);

		if (pIrpEntry->ulSerialNumber != pEventInfo->SerialNumber) 
			continue;

		pWriteIrp = pIrpEntry->pIrp;

		if (pWriteIrp == NULL) 
		{
			ASSERT(pIrpEntry->bCancelRoutineFreeMemory == FALSE);
			FreeIrpEntry(pIrpEntry);
			continue;
		}

		if (IoSetCancelRoutine(pWriteIrp, PTFSIrpCancelRoutine) == NULL) 
		{
			InitializeListHead(&pIrpEntry->ListEntry);
			pIrpEntry->bCancelRoutineFreeMemory = TRUE;
			continue;
		}

		pWriteIrpSp = pIrpEntry->pIrpSp;
		pEventIrpSp = IoGetCurrentIrpStackLocation(pIrp);

		ASSERT(pWriteIrpSp != NULL);
		ASSERT(pEventIrpSp != NULL);

		pEventContext =(PEVENT_CONTEXT)pWriteIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_EVENT];
		ASSERT(pEventContext != NULL);

		if (pEventIrpSp->Parameters.DeviceIoControl.OutputBufferLength < pEventContext->Length) 
		{
			KdPrint(("[PTFS]::PTFSEventWrite STATUS_INSUFFICIENT_RESOURCE\n"));
			status = STATUS_INSUFFICIENT_RESOURCES;
		}
		else 
		{
			PVOID pBuffer = NULL;
			if (pIrp->MdlAddress)
				pBuffer = MmGetSystemAddressForMdlNormalSafe(pIrp->MdlAddress);
			else
				pBuffer = pIrp->AssociatedIrp.SystemBuffer;

			ASSERT(pBuffer != NULL);
			RtlCopyMemory(pBuffer, pEventContext, pEventContext->Length);
			info = pEventContext->Length;
			status = STATUS_SUCCESS;
		}

		FreeEventContext(pEventContext);
		pWriteIrp->Tail.Overlay.DriverContext[DRIVER_CONTEXT_EVENT] = 0;
		KeReleaseSpinLock(&pVcb->pDcb->PendingIrpList.ListSpinLock, oldIrql);

		pIrp->IoStatus.Status = status;
		pIrp->IoStatus.Information = info;
		return pIrp->IoStatus.Status;
	}

	KeReleaseSpinLock(&pVcb->pDcb->PendingIrpList.ListSpinLock, oldIrql);
	KdPrint(("[PTFS]::PTFSEventWrite Cannot found corresponding IRP. User should already canceled the operation. Return STATUS_CANCELLED.\n"));
	return STATUS_CANCELLED;
}
