#include "directory.h"
#include "init.h"
#include "util.h"
#include "event.h"
#include "except.h"
#include "notification.h"
#include "irp_buffer_helper.h"

#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSDispatchDirectoryControl)
#endif

NTSTATUS
NotifyChangeDirectory(
    IN PDEVICE_OBJECT pDeviceObject, 
    IN PIRP pIrp
) 
{
    PIO_STACK_LOCATION pIrpSp = NULL;
    PFILE_OBJECT pFileObject = NULL;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;
    PPTFSVCB pVcb = NULL;

    KdPrint(("[PTFS]::NotifyChangeDirectory Start\n"));

    pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
    pFileObject = pIrpSp->FileObject;

    pVcb = pDeviceObject->DeviceExtension;
    if (GETIDENTIFIERTYPE(pVcb) != VCB) 
    {
        KdPrint(("[PTFS]::NotifyChangeDirectory Invalid identifier Type\n"));
        return STATUS_INVALID_PARAMETER;
    }

    pCcb = pFileObject->FsContext2;
    ASSERT(pCcb != NULL);

    pFcb = pCcb->pFcb;
    ASSERT(pFcb != NULL);

    if (!PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY)) 
    {
        KdPrint(("[PTFS]::NotifyChangeDirectory not directory\n"));
        return STATUS_INVALID_PARAMETER;
    }

    KdPrint(("[PTFS]::NotifyChangeDirectory FileName[%wZ]\n", &pFcb->unstrFileName));

    PTFSFcbLock(pFcb, TRUE);
    FsRtlNotifyFullChangeDirectory(pVcb->NotifySync, &pVcb->DirNotifyList, pCcb, (PSTRING)&pFcb->unstrFileName,
        (pIrpSp->Flags & SL_WATCH_TREE) ? TRUE : FALSE, FALSE,pIrpSp->Parameters.NotifyDirectory.CompletionFilter, pIrp, NULL, NULL);
    PTFSFcbUnlock(pFcb);

    return STATUS_PENDING;
}

NTSTATUS
QueryDirectory(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
	PIO_STACK_LOCATION pIrpSp = NULL;
	PFILE_OBJECT pFileObject = NULL;
	PPTFSVCB pVcb = NULL;
	PPTFSCCB pCcb = NULL;
	PPTFSFCB pFcb = NULL;
	NTSTATUS status;
	ULONG eventLength;
	PEVENT_CONTEXT pEventContext = NULL;
	ULONG index;
	BOOLEAN initial;
	ULONG flags = 0;

	KdPrint(("[PTFS]::QueryDirectory Start\n"));

	pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
	pFileObject = pIrpSp->FileObject;

	pVcb = pDeviceObject->DeviceExtension;
	if (GETIDENTIFIERTYPE(pVcb) != VCB)
	{
		KdPrint(("[PTFS]::QueryDirectory Invalid identifier Type\n"));
		return STATUS_INVALID_PARAMETER;
	}

	pCcb = pFileObject->FsContext2;
	if (pCcb == NULL)
	{
		KdPrint(("[PTFS]::QueryDirectory null CCB\n"));
		return STATUS_INVALID_PARAMETER;
	}

	ASSERT(pCcb != NULL);

	if (pIrpSp->Flags & SL_INDEX_SPECIFIED)
		KdPrint(("[PTFS]::QueryDirectory SL_INDEX_SPECIFIED index[%d]\n", pIrpSp->Parameters.QueryDirectory.FileIndex));
	if (pIrpSp->Flags & SL_RETURN_SINGLE_ENTRY)
		KdPrint(("[PTFS]::QueryDirectory return single entry\n"));
	if (pIrpSp->Flags & SL_RESTART_SCAN)
		KdPrint(("[PTFS]::QueryDirectory restart scan\n"));

	if (pIrpSp->Parameters.QueryDirectory.FileName)
		KdPrint(("[PTFS]::QueryDirectory pattern[%wZ]\n", pIrpSp->Parameters.QueryDirectory.FileName));

	switch (pIrpSp->Parameters.QueryDirectory.FileInformationClass)
	{
	case FileDirectoryInformation:
		KdPrint(("[PTFS]::QueryDirectory FileDirectoryInformation\n"));
		break;
	case FileIdFullDirectoryInformation:
		KdPrint(("[PTFS]::QueryDirectory FileIdFullDirectoryInformation\n"));
		break;
	case FileFullDirectoryInformation:
		KdPrint(("[PTFS]::QueryDirectory FileFullDirectoryInformation\n"));
		break;
	case FileNamesInformation:
		KdPrint(("[PTFS]::QueryDirectory FileNamesInformation\n"));
		break;
	case FileBothDirectoryInformation:
		KdPrint(("[PTFS]::QueryDirectory FileBothDirectoryInformation\n"));
		break;
	case FileIdBothDirectoryInformation:
		KdPrint(("[PTFS]::QueryDirectory FileIdBothDirectoryInformation\n"));
		break;
	default:
		KdPrint(("[PTFS]::QueryDirectory unknown FileInfoClass[%d]\n", pIrpSp->Parameters.QueryDirectory.FileInformationClass));
		break;
	}

	pFcb = pCcb->pFcb;
	ASSERT(pFcb != NULL);

	if (pIrp->MdlAddress == NULL)
	{
		status = AllocateMdl(pIrp, pIrpSp->Parameters.QueryDirectory.Length);
		if (!NT_SUCCESS(status))
		{
			KdPrint(("[PTFS]::QueryDirectory failed to AllocateMdl staus[0x%x]\n", status));
			return status;
		}
		flags = PTFS_MDL_ALLOCATED;
	}

	PTFSFcbLock(pFcb, TRUE);

	eventLength = sizeof(EVENT_CONTEXT) + pFcb->unstrFileName.Length;
	initial = (BOOLEAN)(pCcb->pwszSearchPattern == NULL && !(PTFSCCBFlagsIsSet(pCcb, PTFS_DIR_MATCH_ALL)));

	// this is an initial query
	if (initial)
	{
		KdPrint(("[PTFS]::QueryDirectory initial query\n"));

		if (pIrpSp->Parameters.QueryDirectory.FileName)
		{
			if (pCcb->pwszSearchPattern)
			{
				PTFSFree(pCcb->pwszSearchPattern);
				pCcb->pwszSearchPattern = NULL;
			}

			pCcb->ulSearchPatternLength = pIrpSp->Parameters.QueryDirectory.FileName->Length;
			pCcb->pwszSearchPattern = PTFSAllocateZero(pCcb->ulSearchPatternLength + sizeof(WCHAR));
			if (pCcb->pwszSearchPattern == NULL)
			{
				KdPrint(("[PTFS]::QueryDirectory Failed to Allocate(search)\n"));
				PTFSFcbUnlock(pFcb);
				return STATUS_INSUFFICIENT_RESOURCES;
			}

			RtlCopyMemory(pCcb->pwszSearchPattern, pIrpSp->Parameters.QueryDirectory.FileName->Buffer, pCcb->ulSearchPatternLength);
		}
		else
			PTFSCCBFlagsSetBit(pCcb, PTFS_DIR_MATCH_ALL);
	}

	if (pCcb->pwszSearchPattern)
		eventLength += pCcb->ulSearchPatternLength;

	pEventContext = AllocateEventContext(pVcb->pDcb, pIrp, eventLength, pCcb);

	if (pEventContext == NULL)
	{
		KdPrint(("[PTFS]::QueryDirectory Failed to Allocate(Event context)\n"));
		PTFSFcbUnlock(pFcb);
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	pEventContext->Context = pCcb->UserContext;
	index = 0;

	if (pIrpSp->Flags & SL_INDEX_SPECIFIED)
	{
		index = pIrpSp->Parameters.QueryDirectory.FileIndex;
		KdPrint(("[PTFS]::QueryDirectory using FileIndex[%d]\n", index));
	}
	else if (FlagOn(pIrpSp->Flags, SL_RESTART_SCAN))
	{
		KdPrint(("[PTFS]::QueryDirectory SL_RESTART_SCAN\n"));
		index = 0;
	}
	else
	{
		index = (ULONG)pCcb->Context;
		KdPrint(("[PTFS]::QueryDirectory Ccb->Context[%d]\n", index));
	}

	pEventContext->Operation.Directory.FileInformationClass = pIrpSp->Parameters.QueryDirectory.FileInformationClass;
	pEventContext->Operation.Directory.BufferLength = pIrpSp->Parameters.QueryDirectory.Length;
	pEventContext->Operation.Directory.FileIndex = index;

	pEventContext->Operation.Directory.DirectoryNameLength = pFcb->unstrFileName.Length;
	RtlCopyMemory(pEventContext->Operation.Directory.DirectoryName, pFcb->unstrFileName.Buffer, pFcb->unstrFileName.Length);
	PTFSFcbUnlock(pFcb);

	if (pCcb->ulSearchPatternLength && pCcb->pwszSearchPattern)
	{
		PVOID pSearchBuffer = NULL;

		pEventContext->Operation.Directory.SearchPatternLength = pCcb->ulSearchPatternLength;
		pEventContext->Operation.Directory.SearchPatternOffset = pEventContext->Operation.Directory.DirectoryNameLength;
		pSearchBuffer = (PVOID)((SIZE_T)&pEventContext->Operation.Directory.SearchPatternBase[0] + (SIZE_T)pEventContext->Operation.Directory.SearchPatternOffset);

		RtlCopyMemory(pSearchBuffer, pCcb->pwszSearchPattern, pCcb->ulSearchPatternLength);

		KdPrint(("[PTFS]::QueryDirectory Ccb SearchPattern[%ws]\n", pCcb->pwszSearchPattern));
	}

	status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, flags);
	return status;
}

NTSTATUS
PTFSDispatchDirectoryControl(
    IN PDEVICE_OBJECT pDeviceObject, 
    IN PIRP pIrp
) 
{
    NTSTATUS status = STATUS_NOT_IMPLEMENTED;
    PFILE_OBJECT pFileObject = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    PPTFSVCB pVcb = NULL;

    __try 
    {
        KdPrint(("[PTFS]::PTFSDispatchDirectoryControl Start\n"));

        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
        pFileObject = pIrpSp->FileObject;

        if (pFileObject == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchDirectoryControl FileObject is NULL\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        pVcb = pDeviceObject->DeviceExtension;
        if (GETIDENTIFIERTYPE(pVcb) != VCB || !PTFSCheckCCB(pVcb->pDcb, pFileObject->FsContext2)) 
        {
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        KdPrint(("[PTFS]::PTFSDispatchDirectoryControl ProcessId[%lu]\n", IoGetRequestorProcessId(pIrp)));

        if (pIrpSp->MinorFunction == IRP_MN_QUERY_DIRECTORY) 
            status = QueryDirectory(pDeviceObject, pIrp);
        else if (pIrpSp->MinorFunction == IRP_MN_NOTIFY_CHANGE_DIRECTORY)
            status = NotifyChangeDirectory(pDeviceObject, pIrp);
        else 
        {
            KdPrint(("[PTFS]::PTFSDispatchDirectoryControl invalid minor function\n"));
            status = STATUS_INVALID_PARAMETER;
        }

    }
    __finally 
    {
        PTFSCompleteIrpRequest(pIrp, status, 0);
        KdPrint(("[PTFS]::PTFSDispatchDirectoryControl end\n"));
    }

    return status;
}

VOID
PTFSCompleteDirectoryControl(
	IN PIRP_ENTRY pIrpEntry,
	IN PEVENT_INFORMATION pEventInfo
)
{
    PIRP pIrp = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status = STATUS_SUCCESS;
    ULONG info = 0;
    ULONG bufferLen = 0;
    PVOID pBuffer = NULL;

    KdPrint(("[PTFS]::PTFSCompleteDirectoryControl Start\n"));

    pIrp = pIrpEntry->pIrp;
    pIrpSp = pIrpEntry->pIrpSp;

    if (pIrp->MdlAddress) 
        pBuffer = MmGetSystemAddressForMdlNormalSafe(pIrp->MdlAddress);
    else
        pBuffer = pIrp->UserBuffer;

    bufferLen = pIrpSp->Parameters.QueryDirectory.Length;

    if (bufferLen == 0 || pBuffer == NULL || bufferLen < pEventInfo->BufferLength)
    {
        info = 0;
        status = STATUS_INSUFFICIENT_RESOURCES;
    }
    else 
    {
        PPTFSCCB pCcb = pIrpEntry->pFileObject->FsContext2;
        ASSERT(pBuffer != NULL);
        RtlZeroMemory(pBuffer, bufferLen);

        RtlCopyMemory(pBuffer, pEventInfo->Buffer, pEventInfo->BufferLength);

        KdPrint(("[PTFS]::PTFSCompleteDirectoryControl Directory.Index[%lu], BufferLength[%lu], Status[%lu]\n",
            pEventInfo->Operation.Directory.Index, pEventInfo->BufferLength, pEventInfo->Status));

        pCcb->Context = pEventInfo->Operation.Directory.Index;
        pCcb->UserContext = pEventInfo->Context;
        status = pEventInfo->Status;
        info = pEventInfo->BufferLength;
    }

    if (pIrpEntry->ulFlags & PTFS_MDL_ALLOCATED) 
    {
        FreeMdl(pIrp);
        pIrpEntry->ulFlags &= ~PTFS_MDL_ALLOCATED;
    }

    PTFSCompleteIrpRequest(pIrp, status, info);
    KdPrint(("[PTFS]::PTFSCompleteDirectoryControl End\n"));
}
