#include "Device.h"
#include "init.h"
#include "util.h"
#include "fcb.h"
#include "event.h"
#include "notification.h"
#include "timeout.h"
#include "irp_buffer_helper.h"
#include "access.h"

#include <ntdddisk.h>
#include <ntddvol.h>
#include <ntddstor.h>
#include <ntstrsafe.h>
#include <mountdev.h>
#include <storduid.h>

#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSDispatchDeviceControl)
#endif


NTSTATUS
GlobalDeviceControl(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
	PIO_STACK_LOCATION pIrpSp = NULL;
	PPTFS_GLOBAL pPTFSGlobal = NULL;
	NTSTATUS status = STATUS_NOT_IMPLEMENTED;

	KdPrint(("[PTFS]::GlobalDeviceControl Start\n"));

	pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
	pPTFSGlobal = pDeviceObject->DeviceExtension;

	switch (pIrpSp->Parameters.DeviceIoControl.IoControlCode)
	{
	case IOCTL_EVENT_START:
		KdPrint(("[PTFS]::GlobalDeviceControl IOCTL_EVENT_START\n"));
		status = PTFSEventStart(pDeviceObject, pIrp);
		break;

	case IOCTL_SET_DEBUG_MODE:
	{
		PULONG pDebug = NULL;
		GET_IRP_BUFFER_OR_BREAK(pIrp, pDebug)
			g_Debug = *pDebug;
		status = STATUS_SUCCESS;
		KdPrint(("[PTFS]::GlobalDeviceControl IOCTL_SET_DEBUG_MODE Mode[%d]\n", g_Debug));
	}
	break;

	case IOCTL_EVENT_RELEASE:
		KdPrint(("[PTFS]::GlobalDeviceControl IOCTL_EVENT_RELEASE\n"));
		status = PTFSGlobalEventRelease(pDeviceObject, pIrp);
		break;

	case IOCTL_MOUNTPOINT_CLEANUP:
		KdPrint(("[PTFS]::GlobalDeviceControl IOCTL_MOUNTPOINT_CLEANUP\n"));
		RemoveSessionDevices(pPTFSGlobal, GetCurrentSessionId(pIrp));
		status = STATUS_SUCCESS;
		break;

	case IOCTL_EVENT_MOUNTPOINT_LIST:
		if (GETIDENTIFIERTYPE(pPTFSGlobal) != PGL)
		{
			KdPrint(("[PTFS]::GlobalDeviceControl Invalid IOCTL_EVENT_MOUNTPOINT_LIST\n"));
			return STATUS_INVALID_PARAMETER;
		}

		KdPrint(("[PTFS]::GlobalDeviceControl IOCTL_EVENT_MOUNTPOINT_LIST\n"));
		status = PTFSGetMountPointList(pDeviceObject, pIrp, pPTFSGlobal);
		break;

	case IOCTL_GET_VERSION:
	{
		ULONG* version;
		if (!PREPARE_OUTPUT(pIrp, version, FALSE))
		{
			KdPrint(("[PTFS]::GlobalDeviceControl Invalid Output buffer IOCTL_GET_VERSION \n"));
			break;
		}
		KdPrint(("[PTFS]::GlobalDeviceControl IOCTL_GET_VERSION \n"));
		*version = (ULONG)PTFS_DRIVER_VERSION;
		status = STATUS_SUCCESS;
	}
	break;

	default:
		KdPrint(("[PTFS]::GlobalDeviceControl Unknown IoctlCode BaseCode[0x%x], FunctionCode[0x%x]\n", 
            DEVICE_TYPE_FROM_CTL_CODE(pIrpSp->Parameters.DeviceIoControl.IoControlCode),
            (pIrpSp->Parameters.DeviceIoControl.IoControlCode & (~0xffffc003)) >> 2));
		status = STATUS_INVALID_PARAMETER;
		break;
	}

	KdPrint(("[PTFS]::GlobalDeviceControl End\n"));
	return status;
}

VOID
PopulateDiskGeometry(
    OUT PDISK_GEOMETRY pDiskGeometry
)
{
    pDiskGeometry->Cylinders.QuadPart = PTFS_DEFAULT_DISK_SIZE / PTFS_DEFAULT_SECTOR_SIZE / 32 / 2;
    pDiskGeometry->MediaType = FixedMedia;
    pDiskGeometry->TracksPerCylinder = 2;
    pDiskGeometry->SectorsPerTrack = 32;
    pDiskGeometry->BytesPerSector = PTFS_DEFAULT_SECTOR_SIZE;
}

VOID 
PopulatePartitionInfo(
    OUT PPARTITION_INFORMATION pPartiInfo
) 
{
    pPartiInfo->RewritePartition = FALSE;
    pPartiInfo->RecognizedPartition = FALSE;
    pPartiInfo->PartitionType = PARTITION_ENTRY_UNUSED;
    pPartiInfo->BootIndicator = FALSE;
    pPartiInfo->HiddenSectors = 0;
    pPartiInfo->StartingOffset.QuadPart = 0;
    pPartiInfo->PartitionLength.QuadPart = PTFS_DEFAULT_DISK_SIZE;
    pPartiInfo->PartitionNumber = 0;
}

VOID 
PopulatePartitionInfoEx(
    OUT PPARTITION_INFORMATION_EX pPartiInfoEx
) 
{
    pPartiInfoEx->PartitionStyle = PARTITION_STYLE_MBR;
    pPartiInfoEx->RewritePartition = FALSE;
    pPartiInfoEx->Mbr.RecognizedPartition = FALSE;
    pPartiInfoEx->Mbr.PartitionType = PARTITION_ENTRY_UNUSED;
    pPartiInfoEx->Mbr.BootIndicator = FALSE;
    pPartiInfoEx->Mbr.HiddenSectors = 0;
    pPartiInfoEx->StartingOffset.QuadPart = 0;
    pPartiInfoEx->PartitionLength.QuadPart = PTFS_DEFAULT_DISK_SIZE;
    pPartiInfoEx->PartitionNumber = 0;
}


NTSTATUS
DiskDeviceControl(
    IN PDEVICE_OBJECT pDeviceObject, 
    IN PIRP pIrp
) 
{
    PIO_STACK_LOCATION pIrpSp = NULL;
    PPTFSDCB pDcb = NULL;
    PPTFSVCB pVcb = NULL;
    NTSTATUS status = STATUS_NOT_IMPLEMENTED;

    KdPrint(("[PTFS]::DiskDeviceControl Start\n"));

    pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
    pDcb = pDeviceObject->DeviceExtension;

    if (GETIDENTIFIERTYPE(pDcb) != DCB) 
    {
        KdPrint(("[PTFS]::DiskDeviceControl Invalid identifier Type\n"));
        return STATUS_INVALID_PARAMETER;
    }

    if (IsDeletePending(pDeviceObject)) 
    {
        KdPrint(("[PTFS]::DiskDeviceControl pending for delete\n"));
        return STATUS_DEVICE_REMOVED;
    }

    pVcb = pDcb->pVcb;
    if (IsUnmountPendingVcb(pVcb)) 
    {
        KdPrint(("[PTFS]::DiskDeviceControl volueme is unmounted\n"));
        return STATUS_NO_SUCH_DEVICE;
    }

    KdPrint(("[PTFS]::DiskDeviceControl Device name[%wZ]\n", pDcb->punstrDiskDeviceName));
    switch (pIrpSp->Parameters.DeviceIoControl.IoControlCode) 
    {
    case IOCTL_DISK_GET_DRIVE_GEOMETRY: 
    {
        PDISK_GEOMETRY pDiskGeometry = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_DRIVE_GEOMETRY\n"));
        if (!PREPARE_OUTPUT(pIrp, pDiskGeometry,FALSE))
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_DRIVE_GEOMETRY Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }
        PopulateDiskGeometry(pDiskGeometry);
        status = STATUS_SUCCESS;
    } break;

    case IOCTL_DISK_GET_LENGTH_INFO: 
    {
        PGET_LENGTH_INFORMATION pGetLengthInfo = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_LENGTH_INFO\n"));
        if (!PREPARE_OUTPUT(pIrp, pGetLengthInfo, FALSE))
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_LENGTH_INFO Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }

        pGetLengthInfo->Length.QuadPart = PTFS_DEFAULT_DISK_SIZE;
        status = STATUS_SUCCESS;
    } break;

    case IOCTL_DISK_GET_DRIVE_LAYOUT: 
    {
        PDRIVE_LAYOUT_INFORMATION pLayout = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_DRIVE_LAYOUT\n"));
        if (!PREPARE_OUTPUT(pIrp, pLayout, FALSE))
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_DRIVE_LAYOUT Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }
        pLayout->PartitionCount = 1;
        pLayout->Signature = 1;
        PopulatePartitionInfo(pLayout->PartitionEntry);
        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_DISK_GET_DRIVE_LAYOUT_EX: 
    {
        PDRIVE_LAYOUT_INFORMATION_EX pLayoutEx = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_DRIVE_LAYOUT_EX\n"));
        if (!PREPARE_OUTPUT(pIrp, pLayoutEx, FALSE)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_DRIVE_LAYOUT_EX Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }
        pLayoutEx->PartitionStyle = PARTITION_STYLE_MBR;
        pLayoutEx->PartitionCount = 1;
        pLayoutEx->Mbr.Signature = 1;
        PopulatePartitionInfoEx(pLayoutEx->PartitionEntry);
        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_DISK_GET_PARTITION_INFO: 
    {
        PPARTITION_INFORMATION pPartiInfo = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_PARTITION_INFO\n"));
        if (!PREPARE_OUTPUT(pIrp, pPartiInfo,FALSE))
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_PARTITION_INFO Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }
        PopulatePartitionInfo(pPartiInfo);
        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_DISK_GET_PARTITION_INFO_EX:
    {
        PPARTITION_INFORMATION_EX pPartiInfoEx = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_PARTITION_INFO_EX\n"));
        if (!PREPARE_OUTPUT(pIrp, pPartiInfoEx, FALSE))
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_GET_PARTITION_INFO_EX Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }
        PopulatePartitionInfoEx(pPartiInfoEx);
        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_DISK_IS_WRITABLE:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_IS_WRITABLE\n"));
        status = IS_DEVICE_READ_ONLY(pDeviceObject) ? STATUS_MEDIA_WRITE_PROTECTED : STATUS_SUCCESS;
        break;

    case IOCTL_DISK_MEDIA_REMOVAL:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_MEDIA_REMOVAL\n"));
        status = STATUS_SUCCESS;
        break;

    case IOCTL_STORAGE_MEDIA_REMOVAL:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_MEDIA_REMOVAL\n"));
        status = STATUS_SUCCESS;
        break;

    case IOCTL_DISK_SET_PARTITION_INFO:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_SET_PARTITION_INFO\n"));
        break;

    case IOCTL_DISK_VERIFY:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_DISK_VERIFY\n"));
        break;

    case IOCTL_STORAGE_GET_HOTPLUG_INFO: 
    {
        PSTORAGE_HOTPLUG_INFO pHotplugInfo = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_GET_HOTPLUG_INFO\n"));
        if (!PREPARE_OUTPUT(pIrp, pHotplugInfo, FALSE))
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_GET_HOTPLUG_INFO Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }
        pHotplugInfo->Size = sizeof(STORAGE_HOTPLUG_INFO);
        pHotplugInfo->MediaRemovable = 1;
        pHotplugInfo->MediaHotplug = 1;
        pHotplugInfo->DeviceHotplug = 1;
        pHotplugInfo->WriteCacheEnableOverride = 0;
        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_VOLUME_GET_GPT_ATTRIBUTES: 
    {
        PVOLUME_GET_GPT_ATTRIBUTES_INFORMATION pGptAttrInfo = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_GET_GPT_ATTRIBUTES\n"));
        if (!PREPARE_OUTPUT(pIrp, pGptAttrInfo,FALSE)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_GET_GPT_ATTRIBUTES Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }
        if (IS_DEVICE_READ_ONLY(pDeviceObject)) 
            pGptAttrInfo->GptAttributes = GPT_BASIC_DATA_ATTRIBUTE_READ_ONLY;

        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_STORAGE_CHECK_VERIFY:
    case IOCTL_DISK_CHECK_VERIFY:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_CHECK_VERIFY\n"));
        status = STATUS_SUCCESS;
        break;

    case IOCTL_STORAGE_CHECK_VERIFY2:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_CHECK_VERIFY2\n"));
        status = STATUS_SUCCESS;
        break;

    case IOCTL_STORAGE_QUERY_PROPERTY:
    {
        PSTORAGE_PROPERTY_QUERY pQuery = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_QUERY_PROPERTY\n"));
        GET_IRP_BUFFER_OR_BREAK(pIrp, pQuery)

	    if (pQuery->QueryType == PropertyExistsQuery)
        {
			if (pQuery->PropertyId == StorageDeviceUniqueIdProperty)
			{
				PSTORAGE_DEVICE_UNIQUE_IDENTIFIER pStorage = NULL;
				KdPrint(("[PTFS]::DiskDeviceControl PropertyExistsQuery StorageDeviceUniqueIdProperty\n"));
				GET_IRP_BUFFER_OR_BREAK(pIrp, pStorage)
				status = STATUS_SUCCESS;
			}
			else if (pQuery->PropertyId == StorageDeviceWriteCacheProperty)
            {
                KdPrint(("[PTFS]::DiskDeviceControl PropertyExistsQuery StorageDeviceWriteCacheProperty\n"));
				status = STATUS_NOT_IMPLEMENTED;
			}
			else 
            {
                KdPrint(("[PTFS]::DiskDeviceControl PropertyExistsQuery Unknown[%d]\n", pQuery->PropertyId));
                status = STATUS_NOT_IMPLEMENTED;
			}
		}
		else if (pQuery->QueryType == PropertyStandardQuery)
		{
			if (pQuery->PropertyId == StorageDeviceProperty)
			{
				PSTORAGE_DEVICE_DESCRIPTOR pStorageDesc = NULL;
				KdPrint(("[PTFS]::DiskDeviceControl PropertyStandardQuery StorageDeviceProperty\n"));
				GET_IRP_BUFFER_OR_BREAK(pIrp, pStorageDesc)
				status = STATUS_SUCCESS;
			}
			else if (pQuery->PropertyId == StorageAdapterProperty)
			{
				KdPrint(("[PTFS]::DiskDeviceControl PropertyStandardQuery StorageAdapterProperty\n"));
				status = STATUS_NOT_IMPLEMENTED;
			}
			else
			{
				KdPrint(("[PTFS]::DiskDeviceControl PropertyStandardQuery Unknown[%d]\n", pQuery->PropertyId));
				status = STATUS_ACCESS_DENIED;
			}
		}
		else 
        {
            KdPrint(("[PTFS]::DiskDeviceControl Unknown query type[%d]\n", pQuery->QueryType));
			status = STATUS_ACCESS_DENIED;
		}
	}
	break;

	case IOCTL_MOUNTDEV_QUERY_DEVICE_NAME:
	{
		PMOUNTDEV_NAME pMountdevName = NULL;
		KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_DEVICE_NAME\n"));
		if (!PREPARE_OUTPUT(pIrp, pMountdevName, TRUE))
		{
			KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_DEVICE_NAME Invalid Out Buffer\n"));
			status = STATUS_BUFFER_TOO_SMALL;
			break;
		}
		pMountdevName->NameLength = pDcb->punstrDiskDeviceName->Length;
		if (AppendVarSizeOutputString(pIrp, &pMountdevName->Name, pDcb->punstrDiskDeviceName, FALSE, FALSE))
		{
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_DEVICE_NAME DeviceName[%ws]\n", pMountdevName->Name));
            status = STATUS_SUCCESS;
		}
		else
		{
			KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_DEVICE_NAME Buffer Overflow\n"));
			status = STATUS_BUFFER_OVERFLOW;
		}
	}
	break;

    case IOCTL_MOUNTDEV_QUERY_UNIQUE_ID: 
    {
        PMOUNTDEV_UNIQUE_ID pUniqueId = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_UNIQUE_ID\n"));
        if (!PREPARE_OUTPUT(pIrp, pUniqueId, TRUE))
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_UNIQUE_ID Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }

        pUniqueId->UniqueIdLength = pDcb->punstrDiskDeviceName->Length;
        if (AppendVarSizeOutputString(pIrp, &pUniqueId->UniqueId, pDcb->punstrDiskDeviceName,FALSE, FALSE)) 
        {
            status = STATUS_SUCCESS;
        }
        else 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_UNIQUE_ID Buffer Overflow\n"));
            status = STATUS_BUFFER_OVERFLOW;
        }
    } break;

    case IOCTL_MOUNTDEV_QUERY_SUGGESTED_LINK_NAME: 
    {
        PMOUNTDEV_SUGGESTED_LINK_NAME pLinkName = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_SUGGESTED_LINK_NAME\n"));
        if (!PREPARE_OUTPUT(pIrp, pLinkName,TRUE)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_SUGGESTED_LINK_NAME Invalid Out Buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }
        if (pDcb->punstrMountPoint == NULL || pDcb->punstrMountPoint->Length == 0) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_SUGGESTED_LINK_NAME Not suggesting link name\n"));
            status = STATUS_NOT_FOUND;
            break;
        }
        if (!IsMountPointDriveLetter(pDcb->punstrMountPoint)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_SUGGESTED_LINK_NAME non-drive-letter mount point[%wZ]\n", pDcb->punstrMountPoint));
            status = STATUS_NOT_FOUND;
            break;
        }

        pLinkName->UseOnlyIfThereAreNoOtherLinks = FALSE;
        pLinkName->NameLength = pDcb->punstrMountPoint->Length;
        if (!AppendVarSizeOutputString(pIrp, &pLinkName->Name, pDcb->punstrMountPoint, FALSE, FALSE)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_SUGGESTED_LINK_NAME Buffer Overflow\n"));
            status = STATUS_BUFFER_OVERFLOW;
            break;
        }

        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_SUGGESTED_LINK_NAME [%wZ]\n", pDcb->punstrMountPoint));
        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_MOUNTDEV_LINK_CREATED: 
    {
        PMOUNTDEV_NAME pMountdevName = NULL;
        PMOUNT_ENTRY pMountEntry = NULL;
        UNICODE_STRING mountdevNameString;

        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_CREATED\n"));

        GET_IRP_MOUNTDEV_NAME_OR_BREAK(pIrp, pMountdevName)
        mountdevNameString = PTFSWrapUnicodeString(pMountdevName->Name, pMountdevName->NameLength);
        status = STATUS_SUCCESS;

        if (pMountdevName->NameLength == 0) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_CREATED Link created with empty name\n"));
            break;
        }

        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_CREATED [%wZ], Mount DevName[%wZ]\n", pDcb->punstrMountPoint, mountdevNameString));

        if (IsUnmountPending(pDeviceObject)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_CREATED unmount is pending\n"));
            break;
        }

        if (!pDcb->punstrPersistentSymbolicLinkName && StartsWithVolumeGuidPrefix(&mountdevNameString)) 
        {
            pDcb->punstrPersistentSymbolicLinkName = PTFSAllocDuplicateString(&mountdevNameString);
            break;
        }
        if (!StartsWithDosDevicesPrefix(&mountdevNameString)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_CREATED not DosDevices\n"));
            break;
        }

        if (pDcb->punstrMountPoint && RtlEqualUnicodeString(pDcb->punstrMountPoint, &mountdevNameString, FALSE)) 
        {
            pDcb->bMountPointDetermined = TRUE;
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_CREATED Link name match\n"));
            break;
        }

        if (pDcb->punstrMountPoint) 
            PTFSFree(pDcb->punstrMountPoint);

        pDcb->punstrMountPoint = PTFSAllocDuplicateString(&mountdevNameString);
        if (pDcb->punstrMountPoint == NULL)
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_CREATED Failed to Allocate\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            break;
        }
        pDcb->bMountPointDetermined = TRUE;

        pMountEntry = FindMountEntryByName(pDcb->pPTFSGlobal, pDcb->punstrDiskDeviceName, pDcb->punstrUNCName,TRUE);

        if (pMountEntry != NULL) 
            RtlStringCchCopyUnicodeString(pMountEntry->MountControl.MountPoint, MAXIMUM_FILENAME_LENGTH, pDcb->punstrMountPoint);
        else 
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_CREATED Cannot find associated MountEntry\n"));
    } 
    break;

    case IOCTL_MOUNTDEV_LINK_DELETED: 
	{
		PMOUNTDEV_NAME pMountdevName = NULL;
		UNICODE_STRING mountdevNameString;

		GET_IRP_MOUNTDEV_NAME_OR_BREAK(pIrp, pMountdevName)
		mountdevNameString = PTFSWrapUnicodeString(pMountdevName->Name, pMountdevName->NameLength);
		status = STATUS_SUCCESS;

        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_DELETED deleted Name[%wZ]\n", &mountdevNameString));

		if (!pDcb->usUseMountManager) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_DELETED Mount manager is disabled for this device or PTFS initiated\n"));
			break;
		}
		if (!pDcb->punstrMountPoint || pDcb->punstrMountPoint->Length == 0) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_DELETED Deleting the device even though it never had the mount point set.\n"));
			status = PTFSEventRelease(pVcb->pDeviceObject, pIrp);
			break;
		}
		if (!RtlEqualUnicodeString(pDcb->punstrMountPoint, &mountdevNameString, FALSE)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_LINK_DELETED Ignoring deletion because device has different mount point[%wZ]\n", pDcb->punstrMountPoint));
			break;
		}
		status = PTFSEventRelease(pVcb->pDeviceObject, pIrp);
	} 
    break;

    case IOCTL_MOUNTDEV_QUERY_STABLE_GUID:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_MOUNTDEV_QUERY_STABLE_GUID\n"));
        break;
    case IOCTL_VOLUME_ONLINE:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_ONLINE\n"));
        status = STATUS_SUCCESS;
        break;
    case IOCTL_VOLUME_OFFLINE:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_OFFLINE\n"));
        status = STATUS_SUCCESS;
        break;
    case IOCTL_VOLUME_READ_PLEX:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_READ_PLEX\n"));
        break;
    case IOCTL_VOLUME_PHYSICAL_TO_LOGICAL:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_PHYSICAL_TO_LOGICAL\n"));
        break;
    case IOCTL_VOLUME_IS_CLUSTERED:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_IS_CLUSTERED\n"));
        break;
    case IOCTL_VOLUME_PREPARE_FOR_CRITICAL_IO:
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_PREPARE_FOR_CRITICAL_IO\n"));
        break;

    case IOCTL_VOLUME_GET_VOLUME_DISK_EXTENTS: 
    {
        PVOLUME_DISK_EXTENTS pVolDiskExt = NULL;
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_GET_VOLUME_DISK_EXTENTS\n"));
        if (!PREPARE_OUTPUT(pIrp, pVolDiskExt, FALSE)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_VOLUME_GET_VOLUME_DISK_EXTENTS Invalid Out buffer\n"));
            status = STATUS_INVALID_PARAMETER;
            break;
        }
        pVolDiskExt->NumberOfDiskExtents = 1;
        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_STORAGE_EJECT_MEDIA: 
    {
        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_EJECT_MEDIA\n"));
        PTFSUnmount(pDcb);
        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_REDIR_QUERY_PATH_EX:
    case IOCTL_REDIR_QUERY_PATH: 
    {
        PQUERY_PATH_RESPONSE pathResp = NULL;
        BOOLEAN prefixOk = FALSE;
        unsigned int i = 1;

        if (pDcb->punstrUNCName != NULL && pDcb->punstrUNCName->Length > 0) 
        {
            WCHAR* lpPath = NULL;
            ULONG ulPath = 0;

            if (pIrp->RequestorMode != KernelMode) 
                break;

            if (pIrpSp->Parameters.DeviceIoControl.IoControlCode == IOCTL_REDIR_QUERY_PATH) 
            {
                PQUERY_PATH_REQUEST pathReq = NULL;
                KdPrint(("[PTFS]::DiskDeviceControl IOCTL_REDIR_QUERY_PATH\n"));

                GET_IRP_BUFFER_OR_BREAK(pIrp, pathReq);

                KdPrint(("[PTFS]::DiskDeviceControl IOCTL_REDIR_QUERY_PATH Length[%d], Secu[%p], FilePath[%ws]\n",
                    pathReq->PathNameLength, pathReq->SecurityContext,  pathReq->FilePathName));

                lpPath = pathReq->FilePathName;
                ulPath = pathReq->PathNameLength / sizeof(WCHAR);
                if (pathReq->PathNameLength >= pDcb->punstrUNCName->Length / sizeof(WCHAR)) 
                    prefixOk = (_wcsnicmp(pathReq->FilePathName, pDcb->punstrUNCName->Buffer,pDcb->punstrUNCName->Length / sizeof(WCHAR)) == 0);
            }
            else 
            {
                PQUERY_PATH_REQUEST_EX pathReqEx = NULL;

                KdPrint(("[PTFS]::DiskDeviceControl IOCTL_REDIR_QUERY_PATH_EX\n"));
                GET_IRP_BUFFER_OR_BREAK(pIrp, pathReqEx);

                KdPrint(("[PTFS]::DiskDeviceControl IOCTL_REDIR_QUERY_PATH_EX Length[%d], Secu[%p], FilePath[%ws]\n",
                    pathReqEx->PathName.Length, pathReqEx->pSecurityContext, pathReqEx->PathName.Buffer));

                lpPath = pathReqEx->PathName.Buffer;
                ulPath = pathReqEx->PathName.Length / sizeof(WCHAR);
                if (pathReqEx->PathName.Length >= pDcb->punstrUNCName->Length) 
                    prefixOk = (_wcsnicmp(pathReqEx->PathName.Buffer, pDcb->punstrUNCName->Buffer, pDcb->punstrUNCName->Length / sizeof(WCHAR)) == 0);
            }

            for (i=1;i < ulPath && i * sizeof(WCHAR) < pDcb->punstrUNCName->Length && !prefixOk;++i) 
            {
                if (_wcsnicmp(&lpPath[i], &pDcb->punstrUNCName->Buffer[i], 1) != 0) 
                    break;

                if ((i + 1) * sizeof(WCHAR) < pDcb->punstrUNCName->Length) 
                    prefixOk = (pDcb->punstrUNCName->Buffer[i + 1] == L'\\');
            }

            if (!prefixOk)
            {
                status = STATUS_BAD_NETWORK_PATH;
                break;
            }

            for (; i < ulPath && i * sizeof(WCHAR) < pDcb->punstrUNCName->Length && prefixOk;++i) 
            {
                if (_wcsnicmp(&lpPath[i], &pDcb->punstrUNCName->Buffer[i], 1) != 0)
                    prefixOk = FALSE;
            }

            if (!prefixOk)
            {
                status = STATUS_BAD_NETWORK_NAME;
                break;
            }

            if (!PREPARE_OUTPUT(pIrp, pathResp, FALSE)) 
            {
                status = STATUS_BUFFER_TOO_SMALL;
                break;
            }

            pathResp->LengthAccepted = pDcb->punstrUNCName->Length;
            status = STATUS_SUCCESS;
        }
    } 
    break;

    case IOCTL_STORAGE_GET_MEDIA_TYPES_EX: 
    {
        PGET_MEDIA_TYPES pMediaTypes = NULL;
        PDEVICE_MEDIA_INFO pMediaInfo = NULL;
        PDISK_GEOMETRY pDiskGeometry = NULL;

        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_GET_MEDIA_TYPES_EX\n"));

        if (!PREPARE_OUTPUT(pIrp, pMediaTypes, FALSE)) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_GET_MEDIA_TYPES_EX Invalid Out buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }

        pMediaInfo = &pMediaTypes->MediaInfo[0];
        pMediaTypes->DeviceType = FILE_DEVICE_VIRTUAL_DISK;
        pMediaTypes->MediaInfoCount = 1;

        pDiskGeometry = PTFSAllocateZero(sizeof(DISK_GEOMETRY));
        if (pDiskGeometry == NULL) 
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_GET_MEDIA_TYPES_EX failed to allocate\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            break;
        }

        PopulateDiskGeometry(pDiskGeometry);
        pMediaInfo->DeviceSpecific.DiskInfo.MediaType = pDiskGeometry->MediaType;
        pMediaInfo->DeviceSpecific.DiskInfo.NumberMediaSides = 1;
        pMediaInfo->DeviceSpecific.DiskInfo.MediaCharacteristics = (MEDIA_CURRENTLY_MOUNTED | MEDIA_READ_WRITE);
        pMediaInfo->DeviceSpecific.DiskInfo.Cylinders.QuadPart = pDiskGeometry->Cylinders.QuadPart;
        pMediaInfo->DeviceSpecific.DiskInfo.TracksPerCylinder = pDiskGeometry->TracksPerCylinder;
        pMediaInfo->DeviceSpecific.DiskInfo.SectorsPerTrack = pDiskGeometry->SectorsPerTrack;
        pMediaInfo->DeviceSpecific.DiskInfo.BytesPerSector = pDiskGeometry->BytesPerSector;
        PTFSFree(pDiskGeometry);
        status = STATUS_SUCCESS;
    } 
    break;

    case IOCTL_STORAGE_GET_DEVICE_NUMBER:
    {
        PSTORAGE_DEVICE_NUMBER pDeviceNumber = NULL;

        KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_GET_DEVICE_NUMBER\n"));

        if (!PREPARE_OUTPUT(pIrp, pDeviceNumber, TRUE))
        {
            KdPrint(("[PTFS]::DiskDeviceControl IOCTL_STORAGE_GET_DEVICE_NUMBER Invalid out buffer\n"));
            status = STATUS_BUFFER_TOO_SMALL;
            break;
        }

        pDeviceNumber->DeviceType = FILE_DEVICE_VIRTUAL_DISK;
        if (pVcb)
            pDeviceNumber->DeviceType = pVcb->pDeviceObject->DeviceType;
        pDeviceNumber->DeviceNumber = 0;
        pDeviceNumber->PartitionNumber = (ULONG)-1;

        status = STATUS_SUCCESS;
    } 
    break;

    default:
        KdPrint(("[PTFS]::DiskDeviceControl Unknown IoctlCode BaseCode[0x%x], FunctionCode[0x%x]\n",
            DEVICE_TYPE_FROM_CTL_CODE(pIrpSp->Parameters.DeviceIoControl.IoControlCode),
            (pIrpSp->Parameters.DeviceIoControl.IoControlCode & (~0xffffc003)) >> 2));
        status = STATUS_INVALID_DEVICE_REQUEST;
        break;
    }

    KdPrint(("[PTFS]::DiskDeviceControl End\n"));
    return status;
}

NTSTATUS
DiskDeviceControlWithLock(
    IN PDEVICE_OBJECT pDeviceObject, 
    IN PIRP pIrp
) 
{
    PPTFSDCB pDcb = NULL;
    NTSTATUS status = STATUS_NOT_IMPLEMENTED;

    pDcb = pDeviceObject->DeviceExtension;

    KdPrint(("[PTFS]::DiskDeviceControlWithLock Start\n"));

    if (GETIDENTIFIERTYPE(pDcb) != DCB) 
    {
        KdPrint(("[PTFS]::DiskDeviceControlWithLock Invalid identifier Type\n"));
        return STATUS_INVALID_PARAMETER;
    }

    status = IoAcquireRemoveLock(&pDcb->RemoveLock, pIrp);

    if (!NT_SUCCESS(status)) 
    {
        KdPrint(("[PTFS]::DiskDeviceControlWithLock Failed to IoAcquireRemoveLock status[0x%x]\n", status));
        return STATUS_INSUFFICIENT_RESOURCES;
    }

    if (IsDeletePending(pDeviceObject)) 
    {
        KdPrint(("[PTFS]::DiskDeviceControlWithLock Device is deleted\n"));
        IoReleaseRemoveLock(&pDcb->RemoveLock, pIrp);
        return STATUS_NO_SUCH_DEVICE;
    }

    status = DiskDeviceControl(pDeviceObject, pIrp);
    IoReleaseRemoveLock(&pDcb->RemoveLock, pIrp);

    KdPrint(("[PTFS]::DiskDeviceControlWithLock End\n"));
    return status;
}

BOOLEAN
IsVolumeOpen(
    IN PPTFSVCB pVcb, 
    IN PFILE_OBJECT pFileObject
) 
{
    return pFileObject != NULL && pFileObject->FsContext == &pVcb->VolumeFileHeader;
}

NTSTATUS 
GetVolumeMetrics(
    IN PIRP pIrp, 
    IN PPTFSVCB pVcb
) 
{
    VOLUME_METRICS* pVolMetrics = NULL;

    if (!PREPARE_OUTPUT(pIrp, pVolMetrics, TRUE))
    {
        return STATUS_BUFFER_TOO_SMALL;
    }
    PTFSVcbLock(pVcb, TRUE);
    *pVolMetrics = pVcb->VolumeMetrics;
    PTFSVcbUnlock(pVcb);
    return STATUS_SUCCESS;
}


NTSTATUS
PTFSDispatchDeviceControl(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
    PPTFSVCB pVcb = NULL;
    PPTFSDCB pDcb = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status = STATUS_NOT_IMPLEMENTED;
    ULONG controlCode = 0;

    __try 
    {
        pIrp->IoStatus.Information = 0;
        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
        controlCode = pIrpSp->Parameters.DeviceIoControl.IoControlCode;

        if (controlCode != IOCTL_EVENT_WAIT && controlCode != IOCTL_EVENT_INFO && controlCode != IOCTL_KEEPALIVE)
        {
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl ProcessId[%lu], IoControlCode[%lx]\n", IoGetRequestorProcessId(pIrp), controlCode));
        }

        if (pDeviceObject->DriverObject == NULL || pDeviceObject->ReferenceCount < 0) 
        {
            status = STATUS_DEVICE_DOES_NOT_EXIST;
            __leave;
        }

        pVcb = pDeviceObject->DeviceExtension;

        if (GETIDENTIFIERTYPE(pVcb) == PGL) 
        {
            status = GlobalDeviceControl(pDeviceObject, pIrp);
            __leave;
        }
        else if (GETIDENTIFIERTYPE(pVcb) == DCB) 
        {
            status = DiskDeviceControlWithLock(pDeviceObject, pIrp);
            __leave;
        }
        else if (GETIDENTIFIERTYPE(pVcb) != VCB) 
        {
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        pDcb = pVcb->pDcb;

        switch (pIrpSp->Parameters.DeviceIoControl.IoControlCode) 
        {
        case IOCTL_EVENT_WAIT:
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl IOCTL_EVENT_WAIT\n"));
            status = RegisterPendingIrpForEvent(pDeviceObject, pIrp);
            break;

        case IOCTL_EVENT_INFO:
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl IOCTL_EVENT_INFO\n"));
            status = PTFSCompleteIrp(pDeviceObject, pIrp);
            break;

        case IOCTL_EVENT_RELEASE:
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl IOCTL_EVENT_RELEASE\n"));
            status = PTFSEventRelease(pDeviceObject, pIrp);
            break;

        case IOCTL_EVENT_WRITE:
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl IOCTL_EVENT_WRITE\n"));
            status = PTFSEventWrite(pDeviceObject, pIrp);
            break;

        case IOCTL_GET_VOLUME_METRICS:
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl IOCTL_GET_VOLUME_METRICS\n"));
            status = GetVolumeMetrics(pIrp, pVcb);
            break;

        case IOCTL_KEEPALIVE:
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl IOCTL_KEEPALIVE\n"));
            if (IsFlagOn(pVcb->ulFlags, VCB_MOUNTED))
            {
                ExEnterCriticalRegionAndAcquireResourceExclusive(&pDcb->Resource);
                UpdateTimeout(&pDcb->liTickCount, PTFS_KEEPALIVE_TIMEOUT_DEFAULT);
                ExReleaseResourceAndLeaveCriticalRegion(&pDcb->Resource);
                status = STATUS_SUCCESS;
            }
            else 
            {
                KdPrint(("[PTFS]::PTFSDispatchDeviceControl device is not mounted\n"));
                status = STATUS_INSUFFICIENT_RESOURCES;
            }
            break;

        case IOCTL_RESET_TIMEOUT:
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl IOCTL_RESET_TIMEOUT\n"));
            status = ResetPendingIrpTimeout(pDeviceObject, pIrp);
            break;

        case IOCTL_GET_ACCESS_TOKEN:
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl IOCTL_GET_ACCESS_TOKEN\n"));
            status = GetAccessToken(pDeviceObject, pIrp);
            break;

		default:
		{
			ULONG baseCode = DEVICE_TYPE_FROM_CTL_CODE(pIrpSp->Parameters.DeviceIoControl.IoControlCode);
			status = STATUS_NOT_IMPLEMENTED;
			if (baseCode == IOCTL_STORAGE_BASE || baseCode == IOCTL_DISK_BASE ||
				baseCode == FILE_DEVICE_NETWORK_FILE_SYSTEM || baseCode == MOUNTDEVCONTROLTYPE)
			{
				status = DiskDeviceControlWithLock(pDcb->pDeviceObject, pIrp);
			}

			if (status == STATUS_NOT_IMPLEMENTED)
			{
				KdPrint(("[PTFS]::PTFSDispatchDeviceControl Unknown Device Ioctl Code[0x%x]\n", pIrpSp->Parameters.DeviceIoControl.IoControlCode));
			}

			if (status == STATUS_NOT_IMPLEMENTED && !IsVolumeOpen(pVcb, pIrpSp->FileObject))
				status = STATUS_INVALID_PARAMETER;
		}
		break;
		} // switch IoControlCode
    }
    __finally 
    {
        if (status != STATUS_PENDING) 
        {
            if (IsDeletePending(pDeviceObject)) 
            {
                KdPrint(("[PTFS]::PTFSDispatchDeviceControl pDeviceObject is invalid\n"));
                status = STATUS_DEVICE_REMOVED;
            }

            PTFSCompleteIrpRequest(pIrp, status, pIrp->IoStatus.Information);
        }

        if (controlCode != IOCTL_EVENT_WAIT && controlCode != IOCTL_EVENT_INFO && controlCode != IOCTL_KEEPALIVE) 
        {
            KdPrint(("[PTFS]::PTFSDispatchDeviceControl End Status[0x%x]\n", status));
        }
    }

    return status;
}
