#include "Create.h"
#include "init.h"
#include "util.h"
#include <ntstrsafe.h>
#include "Fcb.h"
#include "event.h"
#include "notification.h"


#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSDispatchCreate)
#endif

static const UNICODE_STRING g_systemVolumeInformationFileName = RTL_CONSTANT_STRING(L"\\System Volume Information");

void 
FixFileNameForReparseMountPoint(
    IN const UNICODE_STRING* punstrMountPoint,
    IN PIRP pIrp
) 
{
    PECP_LIST pEcpList = NULL;
    struct SYMLINK_ECP_CONTEXT* pEcpContext = NULL;
    // IopSymlinkECPGuid "73d5118a-88ba-439f-92f4-46d38952d250";
    static const GUID iopSymlinkECPGuid = { 0x73d5118a,0x88ba,0x439f,{0x92, 0xf4, 0x46, 0xd3, 0x89, 0x52, 0xd2, 0x50}};
    USHORT unparsedNameLength = 0;
    PUNICODE_STRING punstrFileName = NULL;
    USHORT fileNameLength = 0, ecpNameLength = 0;
    PWSTR unparsedNameInFileObject = NULL, unparsedNameInEcp = NULL;
    UNICODE_STRING unparsedNameInFileObjectUS, unparsedNameInEcpUS;

    if (!g_FixFileNameForReparseMountPoint) 
        return;
 
    if (IsMountPointDriveLetter(punstrMountPoint))
        return;

    if (!NT_SUCCESS(FsRtlGetEcpListFromIrp(pIrp, &pEcpList)) || !pEcpList) 
        return;
    
    if (!NT_SUCCESS(FsRtlFindExtraCreateParameter(pEcpList, &iopSymlinkECPGuid,(void**)&pEcpContext, 0))) 
        return;
    
    if (FsRtlIsEcpFromUserMode(pEcpContext) ||!pEcpContext->FlagsMountPoint.MountPoint.MountPoint) 
        return;

    unparsedNameLength = pEcpContext->UnparsedNameLength;
    if (unparsedNameLength == 0) 
        return;

    punstrFileName = &IoGetCurrentIrpStackLocation(pIrp)->FileObject->FileName;
    fileNameLength = punstrFileName->Length;
    ecpNameLength = pEcpContext->Name.Length;
    if (unparsedNameLength > ecpNameLength ||unparsedNameLength > fileNameLength)
        return;

    unparsedNameInFileObject = (PWSTR)RtlOffsetToPointer(punstrFileName->Buffer, fileNameLength - unparsedNameLength);
    unparsedNameInFileObjectUS = PTFSWrapUnicodeString(unparsedNameInFileObject, unparsedNameLength);

    unparsedNameInEcp = (PWSTR)RtlOffsetToPointer(pEcpContext->Name.Buffer, ecpNameLength - unparsedNameLength);
    unparsedNameInEcpUS = PTFSWrapUnicodeString(unparsedNameInEcp, unparsedNameLength);

    if (RtlEqualUnicodeString(&unparsedNameInFileObjectUS, &unparsedNameInEcpUS,TRUE)) 
    {
        RtlCopyMemory(unparsedNameInFileObject, unparsedNameInEcp,unparsedNameLength);
    }
}

BOOLEAN 
IsPTFSProcessFiles(
    IN UNICODE_STRING FileName
) 
{
    return (FileName.Length > 0 &&
        (RtlEqualUnicodeString(&FileName, &g_KeepAliveFileName, FALSE) ||
            RtlEqualUnicodeString(&FileName, &g_NotificationFileName, FALSE)));
}

VOID 
SetFileObjectForVCB(
    IN PFILE_OBJECT pFileObject,
    IN PPTFSVCB pVcb
) 
{
    pFileObject->SectionObjectPointer = &pVcb->SectionObjectPointers;
    pFileObject->FsContext = &pVcb->VolumeFileHeader;
}


NTSTATUS 
GetParentDir(
    IN const WCHAR* pwszFileName, 
    OUT WCHAR** ppwszParentDir,
    OUT ULONG* pulParentDirLength) 
{
    LONG len = (LONG)wcslen(pwszFileName);
    BOOLEAN bTrailingSlash = FALSE;
    LONG i = 0;

    *ppwszParentDir = NULL;
    *pulParentDirLength = 0;

    if (len < 1)
        return STATUS_INVALID_PARAMETER;
 
    if (!wcscmp(pwszFileName, L"\\"))
        return STATUS_ACCESS_DENIED;

    bTrailingSlash = pwszFileName[len - 1] == '\\';

    *ppwszParentDir = (WCHAR*)PTFSAllocateZero((len + 1) * sizeof(WCHAR));
    if (!*ppwszParentDir)
        return STATUS_INSUFFICIENT_RESOURCES;

    RtlStringCchCopyW(*ppwszParentDir, len, pwszFileName);

    for (i = len - 1; i >= 0; i--) 
    {
        if ((*ppwszParentDir)[i] == '\\') 
        {
            if (i == len - 1 && bTrailingSlash)
                continue;
            
            (*ppwszParentDir)[i] = 0;
            break;
        }
    }

    if (i <= 0) 
    {
        i = 1;
        (*ppwszParentDir)[0] = '\\';
        (*ppwszParentDir)[1] = 0;
    }

    *pulParentDirLength = i * sizeof(WCHAR);
    if (bTrailingSlash && i > 1) 
    {
        (*ppwszParentDir)[i] = '\\';
        (*ppwszParentDir)[i + 1] = 0;
        *pulParentDirLength += sizeof(WCHAR);
    }
    return STATUS_SUCCESS;
}


NTSTATUS
CheckShareAccess(
    IN PFILE_OBJECT pFileObject, 
    IN PPTFSFCB pFcbOrDcb,
    IN ACCESS_MASK DesiredAccess, 
    IN ULONG ShareAccess)
{
    NTSTATUS status;
    PAGED_CODE();

    if ((pFcbOrDcb->Identifier.FsdIdType == FCB) &&
        !FlagOn(ShareAccess, FILE_SHARE_DELETE) &&
        PTFSFCBFlagsIsSet(pFcbOrDcb, PTFS_DELETE_ON_CLOSE))
        return STATUS_DELETE_PENDING;
 
    if ((pFcbOrDcb->Identifier.FsdIdType == FCB) &&
        !FlagOn(ShareAccess, FILE_SHARE_WRITE) &&
        FlagOn(DesiredAccess, FILE_EXECUTE | FILE_READ_DATA | FILE_WRITE_DATA |FILE_APPEND_DATA | DELETE | MAXIMUM_ALLOWED) &&
        MmDoesFileHaveUserWritableReferences(&pFcbOrDcb->SectionObjectPointers)) 
    {
        KdPrint(("[PTFS]::CheckShareAccess FCB has no write shared access\n"));
        return STATUS_SHARING_VIOLATION;
    }

    status = IoCheckShareAccess(DesiredAccess, ShareAccess, pFileObject, &pFcbOrDcb->ShareAccess, FALSE);
    return status;
}

PPTFSCCB 
AllocateCCB(
    IN PPTFSDCB pDcb, 
    IN PPTFSFCB pFcb) 
{
    PPTFSCCB pCcb = ExAllocateFromLookasideListEx(&g_PTFSCCBLookasideList);

    if (pCcb == NULL)
        return NULL;

    ASSERT(pCcb != NULL);
    ASSERT(pFcb != NULL);

    KdPrint(("[PTFS]::AllocateCCB\n"));

    RtlZeroMemory(pCcb, sizeof(PTFSCCB));

    pCcb->Identifier.FsdIdType = CCB;
    pCcb->Identifier.ulSize = sizeof(PTFSCCB);

    pCcb->pFcb = pFcb;
    ExInitializeResourceLite(&pCcb->Resource);
    InitializeListHead(&pCcb->NextCCB);

    InsertTailList(&pFcb->NextCCB, &pCcb->NextCCB);

    pCcb->ulMountId = pDcb->ulMountId;
    pCcb->hProcessId = PsGetCurrentProcessId();

    InterlockedIncrement(&pFcb->pVcb->CcbAllocated);
    return pCcb;
}

NTSTATUS
FreeCCB(
    IN PPTFSCCB pCcb) 
{
    PPTFSFCB pFcb = NULL;

    ASSERT(pCcb != NULL);
    
    pFcb = pCcb->pFcb;

    if (!pFcb)
        return STATUS_SUCCESS;
 
    PTFSFcbLock(pFcb, FALSE);
    KdPrint(("[PTFS]::FreeCCB\n"));

    if (IsListEmpty(&pCcb->NextCCB)) 
    {
        KdPrint(("[PTFS]::FreeCCB WARNING NextCCB is empty\n"));
        PTFSFcbUnlock(pFcb);
        return STATUS_SUCCESS;
    }
    else 
    {
        RemoveEntryList(&pCcb->NextCCB);
        InitializeListHead(&pCcb->NextCCB);
    }

    PTFSFcbUnlock(pFcb);

    ExDeleteResourceLite(&pCcb->Resource);
    if (pCcb->pwszSearchPattern)
        PTFSFree(pCcb->pwszSearchPattern);

    ExFreeToLookasideListEx(&g_PTFSCCBLookasideList, pCcb);
    InterlockedIncrement(&pFcb->pVcb->CcbFreed);
    return STATUS_SUCCESS;
}

VOID 
RetryCreateAfterOplockBreak(
    IN PVOID pContext, 
    IN PIRP pIrp) 
{
    if (NT_SUCCESS(pIrp->IoStatus.Status))
        RegisterPendingRetryIrp((PDEVICE_OBJECT)pContext, pIrp);
    else 
        RegisterAsyncCreateFailure((PDEVICE_OBJECT)pContext, pIrp, pIrp->IoStatus.Status);
}

VOID
MaybeBackOutAtomicOplockRequest(
    IN PPTFSCCB pCcb, 
    IN PIRP pIrp) 
{
    if (pCcb->bAtomicOplockRequestPending)
    {
        FsRtlCheckOplockEx(PTFSGetFcbOplock(pCcb->pFcb), pIrp, OPLOCK_FLAG_BACK_OUT_ATOMIC_OPLOCK, NULL, NULL, NULL);
        pCcb->bAtomicOplockRequestPending = FALSE;
    }
}

NTSTATUS
PTFSDispatchCreate(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status = STATUS_INVALID_PARAMETER;
    PFILE_OBJECT pFileObject = NULL, pRelatedFileObject = NULL;
    PEVENT_CONTEXT pEventContext = NULL;
    ULONG ulInfo = 0, ulFileNameLength = 0, ulEventLength = 0;
    PPTFSVCB pVcb = NULL;
    PPTFSDCB pDcb = NULL;
    PPTFSFCB pFcb = NULL, pRelatedFcb = NULL;
    PPTFSCCB pCcb = NULL;
    PWCHAR pwszFileName = NULL, pwszParentDir = NULL;
    ULONG ulParentDirLength = 0;
    BOOLEAN bNeedBackSlashAfterRelatedFile = FALSE;
    BOOLEAN bAlternateDataStreamOfRootDir = FALSE;
    ULONG ulSecurityDescriptorSize = 0;
    ULONG ulAlignedEventContextSize = 0;
    ULONG ulAlignedObjectNameSize = PointerAlignSize(sizeof(UNICODE_STRING_INTERMEDIATE));
    ULONG ulAlignedObjectTypeNameSize = PointerAlignSize(sizeof(UNICODE_STRING_INTERMEDIATE));
    PUNICODE_STRING_INTERMEDIATE pIntermediateUnicodeStr = NULL;
    PUNICODE_STRING punstrRelatedFileName = NULL;
    PSECURITY_DESCRIPTOR pNewFileSecurityDescriptor = NULL;
    BOOLEAN bOpenRequiringOplock = FALSE;
    BOOLEAN bUnwindShareAccess = FALSE;
    BOOLEAN bEventContextConsumed = FALSE;
    DWORD dwDisposition = 0;
    BOOLEAN bFcbLocked = FALSE, bAllocateCcb = TRUE;

    PAGED_CODE();

    __try 
    {
        KdPrint(("[PTFS]::PTFSDispatchCreate Start\n"));
        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);

        if (pIrpSp->FileObject == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate Null FileObject\n"));
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        pFileObject = pIrpSp->FileObject;
        pRelatedFileObject = pFileObject->RelatedFileObject;
        dwDisposition = (pIrpSp->Parameters.Create.Options >> 24) & 0x000000ff;

        KdPrint(("[PTFS]::PTFSDispatchCreate Create: ProcessId[%lu],FileName[%wZ]\n", IoGetRequestorProcessId(pIrp), &pFileObject->FileName));
        pVcb = pDeviceObject->DeviceExtension;
       
        if (pVcb == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate Null Device Extension\n"));
            status = STATUS_SUCCESS;
            __leave;
        }


        if (GETIDENTIFIERTYPE(pVcb) != VCB) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate not Voulme control block Id[%d]\n", GETIDENTIFIERTYPE(pVcb)));
            status = STATUS_SUCCESS;
            __leave;
        }

        if (IsUnmountPendingVcb(pVcb)) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate not mounted Voulme\n"));
            status = STATUS_NO_SUCH_DEVICE;
            __leave;
        }

        pDcb = pVcb->pDcb;

        FixFileNameForReparseMountPoint(pDcb->punstrMountPoint, pIrp);

        if (pDcb->VolumeDeviceType != FILE_DEVICE_NETWORK_FILE_SYSTEM)
        {
            if (pRelatedFileObject != NULL) 
                pFileObject->Vpb = pRelatedFileObject->Vpb;
            else
                pFileObject->Vpb = pDcb->pDeviceObject->Vpb;
        }

        if (!pVcb->bHasEventWait) 
        {
            if (IsPTFSProcessFiles(pFileObject->FileName))
            {
                KdPrint(("[PTFS]::PTFSDispatchCreate Process File\n"));
            }
            else 
            {
                if (pFileObject->FileName.Length > 0) 
                {
                    if (StartsWith(&pFileObject->FileName, &g_systemVolumeInformationFileName))
                    {
                        KdPrint(("[PTFS]::PTFSDispatchCreate System Volume FileName[%wZ]\n", &pFileObject->FileName));
                        status = STATUS_NO_SUCH_FILE;
                        __leave;
                    }
                }
                status = STATUS_SUCCESS;
                __leave;
            }
        }

        KdPrint(("[PTFS]::PTFSDispatchCreate IrpSp->Flags[%d]\n", pIrpSp->Flags));

        if ((pFileObject->FileName.Length > sizeof(WCHAR)) &&
            (pFileObject->FileName.Buffer[1] == L'\\') && (pFileObject->FileName.Buffer[0] == L'\\')) 
        {
            pFileObject->FileName.Length -= sizeof(WCHAR);
            RtlMoveMemory(&pFileObject->FileName.Buffer[0], &pFileObject->FileName.Buffer[1], pFileObject->FileName.Length);
        }

        if (pRelatedFileObject != NULL && pRelatedFileObject->FsContext2) 
        {
            PPTFSCCB pRelatedCcb = (PPTFSCCB)pRelatedFileObject->FsContext2;
            if (pRelatedCcb->pFcb) 
            {
                pRelatedFcb = pRelatedCcb->pFcb;
                
                PTFSFcbLock(pRelatedFcb, TRUE);

                if (pRelatedFcb->unstrFileName.Length > 0 && pRelatedFcb->unstrFileName.Buffer != NULL) 
                {
                    punstrRelatedFileName = PTFSAllocate(sizeof(UNICODE_STRING));
                    if (punstrRelatedFileName == NULL) 
                    {
                        KdPrint(("[PTFS]::PTFSDispatchCreate Failed allocatePool punstrRelatedFileName\n"));
                        status = STATUS_INSUFFICIENT_RESOURCES;
                        PTFSFcbUnlock(pRelatedFcb);
                        __leave;
                    }

                    punstrRelatedFileName->Buffer = PTFSAllocate(pRelatedFcb->unstrFileName.MaximumLength);
                    if (punstrRelatedFileName->Buffer == NULL) 
                    {
                        KdPrint(("[PTFS]::PTFSDispatchCreate Failed allocatePool punstrRelatedFileName Buffer\n"));
                        PTFSFree(punstrRelatedFileName);
                        punstrRelatedFileName = NULL;
                        status = STATUS_INSUFFICIENT_RESOURCES;
                        PTFSFcbUnlock(pRelatedFcb);
                        __leave;
                    }

                    punstrRelatedFileName->MaximumLength = pRelatedFcb->unstrFileName.MaximumLength;
                    punstrRelatedFileName->Length = pRelatedFcb->unstrFileName.Length;
                    RtlUnicodeStringCopy(punstrRelatedFileName, &pRelatedFcb->unstrFileName);
                }

                PTFSFcbUnlock(pRelatedFcb);
            }
        }

        if (punstrRelatedFileName == NULL && pFileObject->FileName.Length == 0) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate request for FS device\n"));
            if (pIrpSp->Parameters.Create.Options & FILE_DIRECTORY_FILE)
                status = STATUS_NOT_A_DIRECTORY;
            else 
            {
                SetFileObjectForVCB(pFileObject, pVcb);
                ulInfo = FILE_OPENED;
                status = STATUS_SUCCESS;
            }
            __leave;
        }

        if (pFileObject->FileName.Length > sizeof(WCHAR) && 
            pFileObject->FileName.Buffer[pFileObject->FileName.Length / sizeof(WCHAR) - 1] == L'\\') 
        {
            pFileObject->FileName.Length -= sizeof(WCHAR);
        }

        ulFileNameLength = pFileObject->FileName.Length;
        if (punstrRelatedFileName != NULL) 
        {
            ulFileNameLength += punstrRelatedFileName->Length;
            if (pFileObject->FileName.Length > 0 && pFileObject->FileName.Buffer[0] == '\\') 
            {
                KdPrint(("[PTFS]::PTFSDispatchCreate relative path\n"));
                status = STATUS_INVALID_PARAMETER;
                __leave;
            }

            if (punstrRelatedFileName->Length > 0 && pFileObject->FileName.Length > 0 &&
                punstrRelatedFileName->Buffer[punstrRelatedFileName->Length / sizeof(WCHAR) -1] != '\\' && pFileObject->FileName.Buffer[0] != ':') 
            {
                bNeedBackSlashAfterRelatedFile = TRUE;
                ulFileNameLength += sizeof(WCHAR);
            }

            if (punstrRelatedFileName->Length / sizeof(WCHAR) == 1 && pFileObject->FileName.Length > 0 &&
                punstrRelatedFileName->Buffer[0] == '\\' && pFileObject->FileName.Buffer[0] == ':') 
            {
                bAlternateDataStreamOfRootDir = TRUE;
            }
        }

        if (!pDcb->usUseAltStream && SearchUnicodeStringWChar(&pFileObject->FileName, L':') != -1) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate alternate stream\n"));
            status = STATUS_INVALID_PARAMETER;
            ulInfo = 0;
            __leave;
        }

        pwszFileName = PTFSAllocateZero(ulFileNameLength + sizeof(WCHAR));
        if (pwszFileName == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate Failed to allocate for pwszFileName\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        if (punstrRelatedFileName != NULL && !bAlternateDataStreamOfRootDir) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate RelatedFileName[:%wZ]\n", punstrRelatedFileName));

            RtlCopyMemory(pwszFileName, punstrRelatedFileName->Buffer, punstrRelatedFileName->Length);

            if (bNeedBackSlashAfterRelatedFile) 
                ((PWCHAR)pwszFileName)[punstrRelatedFileName->Length / sizeof(WCHAR)] = '\\';

            RtlCopyMemory((PCHAR)pwszFileName + punstrRelatedFileName->Length + (bNeedBackSlashAfterRelatedFile ? sizeof(WCHAR) : 0), pFileObject->FileName.Buffer, pFileObject->FileName.Length);
        }
        else
            RtlCopyMemory(pwszFileName, pFileObject->FileName.Buffer, pFileObject->FileName.Length);
 

        if (IS_DEVICE_READ_ONLY(pDeviceObject) &&
            ((dwDisposition == FILE_SUPERSEDE) || (dwDisposition == FILE_CREATE) ||
                (dwDisposition == FILE_OVERWRITE) ||
                (dwDisposition == FILE_OVERWRITE_IF) ||
                (pIrpSp->Parameters.Create.Options & FILE_DELETE_ON_CLOSE))) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate Media is write protected\n"));
            status = STATUS_MEDIA_WRITE_PROTECTED;
            PTFSFree(pwszFileName);
            __leave;
        }

        if (pFileObject->FsContext2 != NULL) 
        {
            pCcb = pFileObject->FsContext2;
            
            if (GETIDENTIFIERTYPE(pCcb) == CCB && (PTFSCCBFlagsIsSet(pCcb, PTFS_RETRY_CREATE))) 
            {
                PTFSCCBFlagsClearBit(pCcb, PTFS_RETRY_CREATE);
                pFcb = pCcb->pFcb;
                bAllocateCcb = FALSE;
                PTFSFree(pwszFileName);
                pwszFileName = NULL;
            }
        }

        if (bAllocateCcb) 
        {
            if (pIrpSp->Flags & SL_OPEN_TARGET_DIRECTORY) 
            {
                status = GetParentDir(pwszFileName, &pwszParentDir, &ulParentDirLength);
                if (status != STATUS_SUCCESS) 
                {
                    PTFSFree(pwszFileName);
                    pwszFileName = NULL;
                    __leave;
                }

                pFcb = GetFCB(pVcb, pwszParentDir, ulParentDirLength, FlagOn(pIrpSp->Flags, SL_CASE_SENSITIVE));
            }
            else 
            {
                pFcb = GetFCB(pVcb, pwszFileName, ulFileNameLength, FlagOn(pIrpSp->Flags, SL_CASE_SENSITIVE));
            }

            if (pFcb == NULL)
            {
                status = STATUS_INSUFFICIENT_RESOURCES;
                __leave;
            }

            KdPrint(("[PTFS]::PTFSDispatchCreate FileName:%wZ got pFcb %p\n", &pFileObject->FileName, pFcb));
            if (pFcb->lFileCount > 1 && dwDisposition == FILE_CREATE) 
            {
                status = STATUS_OBJECT_NAME_COLLISION;
                __leave;
            }
        }

        if (FlagOn(pIrpSp->Parameters.Create.Options, FILE_DIRECTORY_FILE) &&
            FlagOn(pIrpSp->Parameters.Create.FileAttributes, FILE_ATTRIBUTE_TEMPORARY) &&
            (FILE_CREATE == dwDisposition || FILE_OPEN_IF == dwDisposition)) 
        {
            status = STATUS_INVALID_PARAMETER;
            __leave;
        }

        bFcbLocked = TRUE;
        PTFSFcbLock(pFcb, TRUE);

        if (pIrpSp->Flags & SL_OPEN_PAGING_FILE) 
        {
            status = STATUS_ACCESS_DENIED;
            __leave;
        }

        if (bAllocateCcb) 
        {
            pCcb = AllocateCCB(pDcb, pFcb);
        }

        if (pCcb == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate Failed to allocate CCB\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        pFileObject->FsContext = &pFcb->AdvancedFCBHeader;
        pFileObject->FsContext2 = pCcb;
        pFileObject->PrivateCacheMap = NULL;
        pFileObject->SectionObjectPointer = &pFcb->SectionObjectPointers;
        
        if (pFcb->bBlockUserModeDispatch) 
        {
            ulInfo = FILE_OPENED;
            status = STATUS_SUCCESS;
            __leave;
        }

        ulAlignedEventContextSize = PointerAlignSize(sizeof(EVENT_CONTEXT));

        if (pIrpSp->Parameters.Create.SecurityContext->AccessState) 
        {
            if (pIrpSp->Parameters.Create.SecurityContext->AccessState->SecurityDescriptor) 
            {
                if (STATUS_SUCCESS == SeAssignSecurity(NULL,pIrpSp->Parameters.Create.SecurityContext->AccessState->SecurityDescriptor,
                    &pNewFileSecurityDescriptor,
                    (pIrpSp->Parameters.Create.Options & FILE_DIRECTORY_FILE) || (pIrpSp->Flags & SL_OPEN_TARGET_DIRECTORY),
                    &pIrpSp->Parameters.Create.SecurityContext->AccessState->SubjectSecurityContext, IoGetFileObjectGenericMapping(), PagedPool)) 
                {
                    ulSecurityDescriptorSize = PointerAlignSize(RtlLengthSecurityDescriptor(pNewFileSecurityDescriptor));
                }
                else 
                {
                    pNewFileSecurityDescriptor = NULL;
                }
            }

            if (pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectName.Length > 0) 
            {
                ulAlignedObjectNameSize = PointerAlignSize(sizeof(UNICODE_STRING_INTERMEDIATE) + pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectName.Length + sizeof(WCHAR));
            }
 
            if (pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectTypeName.Length > 0) 
            {
                ulAlignedObjectTypeNameSize = PointerAlignSize(sizeof(UNICODE_STRING_INTERMEDIATE) + pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectTypeName.Length + sizeof(WCHAR));
            }
      }

        ulEventLength = ulAlignedEventContextSize + ulSecurityDescriptorSize;
        ulEventLength += ulAlignedObjectNameSize;
        ulEventLength += ulAlignedObjectTypeNameSize;
        ulEventLength += (pwszParentDir ? ulFileNameLength : pFcb->unstrFileName.Length) + sizeof(WCHAR);

        pEventContext = AllocateEventContext(pVcb->pDcb, pIrp, ulEventLength, pCcb);

        if (pEventContext == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate Failed to allocate pEventContext\n"));
            status = STATUS_INSUFFICIENT_RESOURCES;
            __leave;
        }

        RtlZeroMemory((char*)pEventContext + ulAlignedEventContextSize,ulEventLength - ulAlignedEventContextSize);

        if (pIrpSp->Parameters.Create.SecurityContext->AccessState) 
        {
            pEventContext->Operation.Create.SecurityContext.AccessState.SecurityEvaluated = pIrpSp->Parameters.Create.SecurityContext->AccessState->SecurityEvaluated;
            pEventContext->Operation.Create.SecurityContext.AccessState.GenerateAudit = pIrpSp->Parameters.Create.SecurityContext->AccessState->GenerateAudit;
            pEventContext->Operation.Create.SecurityContext.AccessState.GenerateOnClose = pIrpSp->Parameters.Create.SecurityContext->AccessState->GenerateOnClose;
            pEventContext->Operation.Create.SecurityContext.AccessState.AuditPrivileges = pIrpSp->Parameters.Create.SecurityContext->AccessState->AuditPrivileges;
            pEventContext->Operation.Create.SecurityContext.AccessState.Flags = pIrpSp->Parameters.Create.SecurityContext->AccessState->Flags;
            pEventContext->Operation.Create.SecurityContext.AccessState.RemainingDesiredAccess = pIrpSp->Parameters.Create.SecurityContext->AccessState->RemainingDesiredAccess;
            pEventContext->Operation.Create.SecurityContext.AccessState.PreviouslyGrantedAccess = pIrpSp->Parameters.Create.SecurityContext->AccessState->PreviouslyGrantedAccess;
            pEventContext->Operation.Create.SecurityContext.AccessState.OriginalDesiredAccess = pIrpSp->Parameters.Create.SecurityContext->AccessState->OriginalDesiredAccess;

            if (ulSecurityDescriptorSize > 0) 
            {
                pEventContext->Operation.Create.SecurityContext.AccessState.SecurityDescriptorOffset = (ULONG)(((char*)pEventContext + ulAlignedEventContextSize) -(char*)&pEventContext->Operation.Create.SecurityContext.AccessState);
            }

            pEventContext->Operation.Create.SecurityContext.AccessState.UnicodeStringObjectNameOffset = (ULONG)(((char*)pEventContext + ulAlignedEventContextSize + ulSecurityDescriptorSize) -(char*)&pEventContext->Operation.Create.SecurityContext.AccessState);
            pEventContext->Operation.Create.SecurityContext.AccessState.UnicodeStringObjectTypeOffset = pEventContext->Operation.Create.SecurityContext.AccessState.UnicodeStringObjectNameOffset + ulAlignedObjectNameSize;
        }

        pEventContext->Operation.Create.SecurityContext.DesiredAccess = pIrpSp->Parameters.Create.SecurityContext->DesiredAccess;
        pEventContext->Operation.Create.FileAttributes = pIrpSp->Parameters.Create.FileAttributes;
        pEventContext->Operation.Create.CreateOptions = pIrpSp->Parameters.Create.Options;

        if (IS_DEVICE_READ_ONLY(pDeviceObject) && dwDisposition == FILE_OPEN_IF) 
        {
            pEventContext->Operation.Create.CreateOptions &= ((FILE_OPEN << 24) | 0x00FFFFFF);
        }

        pEventContext->Operation.Create.ShareAccess = pIrpSp->Parameters.Create.ShareAccess;
        pEventContext->Operation.Create.FileNameLength = pwszParentDir ? ulFileNameLength : pFcb->unstrFileName.Length;
        pEventContext->Operation.Create.FileNameOffset = (ULONG)(((char*)pEventContext + ulAlignedEventContextSize + ulSecurityDescriptorSize + ulAlignedObjectNameSize + ulAlignedObjectTypeNameSize) - (char*)&pEventContext->Operation.Create);

        if (pNewFileSecurityDescriptor != NULL) 
        {
            RtlCopyMemory((char*)pEventContext + ulAlignedEventContextSize, pNewFileSecurityDescriptor,RtlLengthSecurityDescriptor(pNewFileSecurityDescriptor));
            SeDeassignSecurity(&pNewFileSecurityDescriptor);
            pNewFileSecurityDescriptor = NULL;
        }

        if (pIrpSp->Parameters.Create.SecurityContext->AccessState) 
        {
            pIntermediateUnicodeStr = (PUNICODE_STRING_INTERMEDIATE)((char*)&pEventContext->Operation.Create.SecurityContext.AccessState +pEventContext->Operation.Create.SecurityContext.AccessState.UnicodeStringObjectNameOffset);
            pIntermediateUnicodeStr->Length = pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectName.Length;
            pIntermediateUnicodeStr->MaximumLength = (USHORT)ulAlignedObjectNameSize;

            if (pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectName.Length > 0) 
            {
                RtlCopyMemory(pIntermediateUnicodeStr->Buffer, pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectName.Buffer, pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectName.Length);
                *(WCHAR*)((char*)pIntermediateUnicodeStr->Buffer + pIntermediateUnicodeStr->Length) = 0;
            }
            else 
            {
                pIntermediateUnicodeStr->Buffer[0] = 0;
            }

            // Object type name
            pIntermediateUnicodeStr = (PUNICODE_STRING_INTERMEDIATE)((char*)pIntermediateUnicodeStr + ulAlignedObjectNameSize);
            pIntermediateUnicodeStr->Length = pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectTypeName.Length;
            pIntermediateUnicodeStr->MaximumLength = (USHORT)ulAlignedObjectTypeNameSize;

            if (pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectTypeName.Length > 0) 
            {
                RtlCopyMemory(pIntermediateUnicodeStr->Buffer,pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectTypeName.Buffer,pIrpSp->Parameters.Create.SecurityContext->AccessState->ObjectTypeName.Length);
                *(WCHAR*)((char*)pIntermediateUnicodeStr->Buffer +pIntermediateUnicodeStr->Length) = 0;
            }
            else 
            {
                pIntermediateUnicodeStr->Buffer[0] = 0;
            }
        }

        pEventContext->Context = 0;
        pEventContext->FileFlags |= PTFSFCBFlagsGet(pFcb);

        {
            char * pDesBuffer = (char*)&pEventContext->Operation.Create;
            pDesBuffer += pEventContext->Operation.Create.FileNameOffset;
            RtlCopyMemory(pDesBuffer, pwszParentDir?pwszFileName: pFcb->unstrFileName.Buffer, pwszParentDir?ulFileNameLength: pFcb->unstrFileName.Length);
        }

        *(PWCHAR)((char*)&pEventContext->Operation.Create + pEventContext->Operation.Create.FileNameOffset + (pwszParentDir ? ulFileNameLength : pFcb->unstrFileName.Length)) = 0;

        bOpenRequiringOplock = BooleanFlagOn(pIrpSp->Parameters.Create.Options,FILE_OPEN_REQUIRING_OPLOCK);
        
        if (pFcb->lFileCount > 1) 
        {
            status = CheckShareAccess(pFileObject, pFcb, pEventContext->Operation.Create.SecurityContext.DesiredAccess, pEventContext->Operation.Create.ShareAccess);

            if (!NT_SUCCESS(status)) 
            {
                NTSTATUS OplockBreakStatus = STATUS_SUCCESS;

                KdPrint(("[PTFS]::PTFSDispatchCreate Failed to CheckShareAccess Status[0x%x]\n", status));

                if ((status == STATUS_SHARING_VIOLATION) && !FlagOn(pIrpSp->Parameters.Create.Options,FILE_COMPLETE_IF_OPLOCKED)) 
                {
                    POPLOCK oplock = PTFSGetFcbOplock(pFcb);

                    OplockBreakStatus = FsRtlOplockBreakH(oplock, pIrp, 0, pDeviceObject, RetryCreateAfterOplockBreak, PTFSPrePostIrp);

                    if (OplockBreakStatus == STATUS_PENDING) 
                    {
                        KdPrint(("[PTFS]::PTFSDispatchCreate FsRtlOplockBreakH returned STATUS_PENDING\n"));
                        status = STATUS_PENDING;
                        __leave;
                    }
                    else if (!NT_SUCCESS(OplockBreakStatus)) 
                    {
                        KdPrint(("[PTFS]::PTFSDispatchCreate FsRtlOplockBreakH returned status[0x%x]\n", OplockBreakStatus));
                        status = OplockBreakStatus;
                        __leave;
                    }
                    else 
                    {
                        status = CheckShareAccess(pFileObject, pFcb, pEventContext->Operation.Create.SecurityContext.DesiredAccess, pEventContext->Operation.Create.ShareAccess);
                        KdPrint(("[PTFS]::PTFSDispatchCreate checked share access again, status[0x%x]\n", status));
                        NT_ASSERT(OplockBreakStatus == STATUS_SUCCESS);
                        if (status != STATUS_SUCCESS)
                            __leave;
                    } 
                }
                else 
                {
                    if (status == STATUS_SHARING_VIOLATION && FlagOn(pIrpSp->Parameters.Create.Options, FILE_COMPLETE_IF_OPLOCKED)) 
                    {
                        KdPrint(("[PTFS]::PTFSDispatchCreate sharing violation FILE_COMPLETE_IF_OPLOCKED\n"));
                    }

                    KdPrint(("[PTFS]::PTFSDispatchCreate failed sharing/oplock status[0x%x]\n", status));
                    __leave;
                }
            }
            IoUpdateShareAccess(pFileObject, &pFcb->ShareAccess);
        }
        else 
        {
            IoSetShareAccess(pEventContext->Operation.Create.SecurityContext.DesiredAccess, pEventContext->Operation.Create.ShareAccess, pFileObject, &pFcb->ShareAccess);
        }

        bUnwindShareAccess = TRUE;

        if (pFcb->lFileCount > 1)
        {
            status = FsRtlCheckOplock(PTFSGetFcbOplock(pFcb), pIrp, pDeviceObject, RetryCreateAfterOplockBreak, PTFSPrePostIrp);
            if (status == STATUS_PENDING) 
            {
                KdPrint(("[PTFS]::PTFSDispatchCreate FsRtlCheckOplock returned STATUS_PENDING pFcb[%p], FileCount[%d]\n", pFcb, pFcb->lFileCount));
                __leave;
            }
        }

        status =FsRtlCheckOplockEx(PTFSGetFcbOplock(pFcb), pIrp,OPLOCK_FLAG_OPLOCK_KEY_CHECK_ONLY, NULL, NULL, NULL);

        if (!NT_SUCCESS(status)) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate FsRtlCheckOplockEx return status[0x%x]\n", status));
            __leave;
        }

        if (bOpenRequiringOplock) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCreate OpenRequiringOplock\n"));

            if ((status == STATUS_SUCCESS)) 
            {
                status = FsRtlOplockFsctrl(PTFSGetFcbOplock(pFcb), pIrp, pFcb->lFileCount);
            }

            if ((status != STATUS_SUCCESS) && (status != STATUS_OPLOCK_BREAK_IN_PROGRESS)) 
            {
                KdPrint(("[PTFS]::PTFSDispatchCreate Failed FsRtlOplockFsctrl status[0x%x], FCB[%p], FileCount[%d]\n", status, pFcb, pFcb->lFileCount));
                __leave;
            }
            else if (status == STATUS_OPLOCK_BREAK_IN_PROGRESS) 
            {
                KdPrint(("[PTFS]::PTFSDispatchCreate STATUS_OPLOCK_BREAK_IN_PROGRESS\n"));
            }

            if (status == STATUS_SUCCESS) 
            {
                pCcb->bAtomicOplockRequestPending = TRUE;
            }
        }

        status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, 0);
        bEventContextConsumed = TRUE;
    }
    __finally 
    {
        KdPrint(("[PTFS]::PTFSDispatchCreate FileName[%wZ], status[0x%x]\n", &pFileObject->FileName, status));

        if (!NT_SUCCESS(status)) 
        {
            if (pCcb != NULL) 
                MaybeBackOutAtomicOplockRequest(pCcb, pIrp);

            if (bUnwindShareAccess) 
                IoRemoveShareAccess(pFileObject, &pFcb->ShareAccess);
        }

        if (bFcbLocked)
            PTFSFcbUnlock(pFcb);

        if (punstrRelatedFileName) 
        {
            PTFSFree(punstrRelatedFileName->Buffer);
            PTFSFree(punstrRelatedFileName);
        }

        if (!NT_SUCCESS(status)) 
        {
            if (!bEventContextConsumed && pEventContext) 
                FreeEventContext(pEventContext);

            if (pCcb)
                FreeCCB(pCcb);

            if (pFcb)
                FreeFCB(pVcb, pFcb);

            pFileObject->FsContext = NULL;
            pFileObject->FsContext2 = NULL;
        }

        if (pwszParentDir && pwszFileName) 
            PTFSFree(pwszFileName);

        PTFSCompleteIrpRequest(pIrp, status, ulInfo);
        KdPrint(("[PTFS]::PTFSDispatchCreate End\n"));
    }
    return status;
}

VOID
PTFSCompleteCreate(
    IN PIRP_ENTRY pIrpEntry,
    IN PEVENT_INFORMATION pEventInfo
)
{
    PIRP pIrp = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status;
    ULONG info = 0;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;
    PPTFSVCB pVcb = NULL;

    pIrp = pIrpEntry->pIrp;
    pIrpSp = pIrpEntry->pIrpSp;

    KdPrint(("[PTFS]::PTFSCompleteCreate Start\n"));

    pCcb = pIrpEntry->pFileObject->FsContext2;
    ASSERT(pCcb != NULL);

    pFcb = pCcb->pFcb;
    ASSERT(pFcb != NULL);

    pVcb = pIrpSp->DeviceObject->DeviceExtension;
    ASSERT(pVcb != NULL);
    PTFSFcbLock(pFcb, FALSE);

    KdPrint(("[PTFS]::PTFSCompleteCreate FileName[%wZ]\n", &pFcb->unstrFileName));

    pCcb->UserContext = pEventInfo->Context;
    status = pEventInfo->Status;
    info = pEventInfo->Operation.Create.Information;

    switch (info) 
    {
    case FILE_OPENED:
        KdPrint(("[PTFS]::PTFSCompleteCreate FILE_OPENED\n"));
        break;
    case FILE_CREATED:
        KdPrint(("[PTFS]::PTFSCompleteCreate FILE_CREATED\n"));
        break;
    case FILE_OVERWRITTEN:
        KdPrint(("[PTFS]::PTFSCompleteCreate FILE_OVERWRITTEN\n"));
        break;
    case FILE_DOES_NOT_EXIST:
        KdPrint(("[PTFS]::PTFSCompleteCreate FILE_DOES_NOT_EXIST\n"));
        break;
    case FILE_EXISTS:
        KdPrint(("[PTFS]::PTFSCompleteCreate FILE_EXISTS\n"));
        break;
    case FILE_SUPERSEDED:
        KdPrint(("[PTFS]::PTFSCompleteCreate FILE_SUPERSEDED\n"));
        break;
    default:
        KdPrint(("[PTFS]::PTFSCompleteCreate Opened info[%d]\n", info));
        break;
    }

    if ((info == FILE_DOES_NOT_EXIST) && (IS_DEVICE_READ_ONLY(pIrpSp->DeviceObject))) 
    {
        DWORD disposition = (pIrpSp->Parameters.Create.Options >> 24) & 0x000000ff;
        if (disposition == FILE_OPEN_IF) 
        {
            KdPrint(("[PTFS]::PTFSCompleteCreate Media is write protected\n"));
            status = STATUS_MEDIA_WRITE_PROTECTED;
        }
    }

    if (NT_SUCCESS(status) && ((pIrpSp->Parameters.Create.Options & FILE_DIRECTORY_FILE) || (pEventInfo->Operation.Create.Flags & PTFS_FILE_DIRECTORY)))
    {
        if (pIrpSp->Parameters.Create.Options & FILE_DIRECTORY_FILE) 
            KdPrint(("[PTFS]::PTFSCompleteCreate FILE_DIRECTORY_FILE[%p]\n", pFcb));
        else 
            KdPrint(("[PTFS]::PTFSCompleteCreate PTFS_DIRECTORY_FILE[%p]\n", pFcb));

        PTFSFCBFlagsSetBit(pFcb, PTFS_FILE_DIRECTORY);
    }

    if (NT_SUCCESS(status))
        PTFSCCBFlagsSetBit(pCcb, PTFS_FILE_OPENED);

    if (NT_SUCCESS(status) && pIrpSp->Parameters.Create.Options & FILE_DELETE_ON_CLOSE) 
    {
        PTFSFCBFlagsSetBit(pFcb, PTFS_DELETE_ON_CLOSE);
        PTFSCCBFlagsSetBit(pCcb, PTFS_DELETE_ON_CLOSE);
        KdPrint(("[PTFS]::PTFSCompleteCreate FILE_DELETE_ON_CLOSE\n"));
    }

    if (NT_SUCCESS(status)) 
    {
        if (info == FILE_CREATED) 
        {
            if (PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY)) 
                NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_DIR_NAME, FILE_ACTION_ADDED);
            else 
                NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_FILE_NAME, FILE_ACTION_ADDED);
        }
        pCcb->bAtomicOplockRequestPending = FALSE;
    }
    else 
    {
        KdPrint(("[PTFS]::PTFSCompleteCreate failed CCB[%p], status[0x%x]\n", pCcb, status));
        MaybeBackOutAtomicOplockRequest(pCcb, pIrp);
        FreeCCB(pCcb);
        IoRemoveShareAccess(pIrpSp->FileObject, &pFcb->ShareAccess);
        PTFSFcbUnlock(pFcb);
        FreeFCB(pVcb, pFcb);
        pFcb = NULL;
        pIrpEntry->pFileObject->FsContext2 = NULL;
    }

    if (pFcb)
        PTFSFcbUnlock(pFcb);

    PTFSCompleteIrpRequest(pIrp, status, info);
    KdPrint(("[PTFS]::PTFSCompleteCreate End\n"));
}
