#include "Cleanup.h"
#include "init.h"
#include "util.h"
#include "notification.h"
#include "event.h"
#include "fileInfo.h"
#include "timeout.h"

#ifdef ALLOC_PRAGMA
#pragma alloc_text (PAGE, PTFSDispatchCleanup)
#endif


NTSTATUS
PTFSDispatchCleanup(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
    PPTFSVCB pVcb = NULL;
    PIO_STACK_LOCATION pIrpSp = NULL;
    NTSTATUS status = STATUS_INVALID_PARAMETER;
    PFILE_OBJECT pFileObject;
    PPTFSCCB pCcb = NULL;
    PPTFSFCB pFcb = NULL;
    PEVENT_CONTEXT pEventContext = NULL;
    ULONG eventLength = 0;

    __try 
    {
        KdPrint(("[PTFS]::PTFSDispatchCleanup Start\n"));

        pIrpSp = IoGetCurrentIrpStackLocation(pIrp);
        pFileObject = pIrpSp->FileObject;

        KdPrint(("[PTFS]::PTFSDispatchCleanup ProcessId[%lu]\n", IoGetRequestorProcessId(pIrp)));

        if (pFileObject == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCleanup FileObject Null\n"));
            status = STATUS_SUCCESS;
            __leave;
        }

        pVcb = pDeviceObject->DeviceExtension;
        if (pVcb == NULL) 
        {
            KdPrint(("[PTFS]::PTFSDispatchCleanup device extension Null\n"));
            status = STATUS_SUCCESS;
            __leave;
        }

        if (GETIDENTIFIERTYPE(pVcb) != VCB || !PTFSCheckCCB(pVcb->pDcb, pFileObject->FsContext2)) 
        {
            status = STATUS_SUCCESS;
            __leave;
        }

        pCcb = pFileObject->FsContext2;
        ASSERT(pCcb != NULL);

        pFcb = pCcb->pFcb;
        ASSERT(pFcb != NULL);

        if (pFcb->bIsKeepalive) 
        {
            BOOLEAN shouldUnmount = FALSE;
            PTFSFcbLock(pFcb,FALSE);
            shouldUnmount = pCcb->bIsKeepaliveActive;
            if (shouldUnmount)
                pCcb->bIsKeepaliveActive = FALSE;

            PTFSFcbUnlock(pFcb);

            if (shouldUnmount) 
            {
                if (IsUnmountPendingVcb(pVcb)) 
                    KdPrint(("[PTFS]::PTFSDispatchCleanup Ignoring keepalive close because unmount is already in progress\n"));
                else 
                {
                    KdPrint(("[PTFS]::PTFSDispatchCleanup Unmounting due to keepalive close.\n"));
                    PTFSUnmount(pVcb->pDcb);
                }
            }
        }

        if (pFcb->bBlockUserModeDispatch) 
        {
            status = STATUS_SUCCESS;
            __leave;
        }

        FlushFcb(pFcb, pFileObject);

        PTFSFcbLock(pFcb, FALSE);

        eventLength = sizeof(EVENT_CONTEXT) + pFcb->unstrFileName.Length;
        pEventContext = AllocateEventContext(pVcb->pDcb, pIrp, eventLength, pCcb);
        if (pEventContext == NULL) 
        {
            status = STATUS_INSUFFICIENT_RESOURCES;
            PTFSFcbUnlock(pFcb);
            __leave;
        }

        pFileObject->Flags |= FO_CLEANUP_COMPLETE;
        pEventContext->Context = pCcb->UserContext;
        pEventContext->FileFlags |= PTFSCCBFlagsGet(pCcb);

        pEventContext->Operation.Cleanup.FileNameLength = pFcb->unstrFileName.Length;
        RtlCopyMemory(pEventContext->Operation.Cleanup.FileName, pFcb->unstrFileName.Buffer, pFcb->unstrFileName.Length);

        status = PTFSCheckOplock(pFcb, pIrp, pEventContext, PTFSOplockComplete, PTFSPrePostIrp);
        PTFSFcbUnlock(pFcb);

        if (status != STATUS_SUCCESS) 
        {
            if (status == STATUS_PENDING) 
                KdPrint(("[PTFS]::PTFSDispatchCleanup FsRtlCheckOplock returned STATUS_PENDING\n"));
            else 
                FreeEventContext(pEventContext);

            __leave;
        }

        status = RegisterPendingIrp(pDeviceObject, pIrp, pEventContext, 0);

    }
    __finally 
    {
        PTFSCompleteIrpRequest(pIrp, status, 0);
        KdPrint(("[PTFS]::PTFSDispatchCleanup End\n"));
    }

    return status;
}

VOID
PTFSCompleteCleanup(
	IN PIRP_ENTRY pIrpEntry,
	IN PEVENT_INFORMATION pEventInfo
)
{
	PIRP pIrp = NULL;
	PIO_STACK_LOCATION pIrpSp = NULL;
	NTSTATUS status = STATUS_SUCCESS;
	PPTFSCCB pCcb = NULL;
	PPTFSFCB pFcb = NULL;
	PPTFSVCB pVcb = NULL;
	PFILE_OBJECT pFileObject = NULL;

	KdPrint(("[PTFS]::PTFSCompleteCleanup Start\n"));

	pIrp = pIrpEntry->pIrp;
	pIrpSp = pIrpEntry->pIrpSp;

	pFileObject = pIrpEntry->pFileObject;
	ASSERT(pFileObject != NULL);

	pCcb = pFileObject->FsContext2;
	ASSERT(pCcb != NULL);

	pCcb->UserContext = pEventInfo->Context;
	pFcb = pCcb->pFcb;
	ASSERT(pFcb != NULL);

	pVcb = pFcb->pVcb;
	status = pEventInfo->Status;

	PTFSFcbLock(pFcb, FALSE);
	IoRemoveShareAccess(pIrpSp->FileObject, &pFcb->ShareAccess);

	if (PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_CHANGE_LAST_WRITE))
		NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_LAST_WRITE, FILE_ACTION_MODIFIED);

	if (PTFSFCBFlagsIsSet(pFcb, PTFS_DELETE_ON_CLOSE))
	{
		if (PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY))
			NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_DIR_NAME, FILE_ACTION_REMOVED);
		else
			NotifyReportChange(pFcb, FILE_NOTIFY_CHANGE_FILE_NAME, FILE_ACTION_REMOVED);
	}
	PTFSFcbUnlock(pFcb);

	(VOID)FsRtlFastUnlockAll(&pFcb->FileLock, pFileObject, IoGetRequestorProcess(pIrp), NULL);

	if (PTFSFCBFlagsIsSet(pFcb, PTFS_FILE_DIRECTORY))
		FsRtlNotifyCleanup(pVcb->NotifySync, &pVcb->DirNotifyList, pCcb);

	PTFSCompleteIrpRequest(pIrp, status, 0);
	KdPrint(("[PTFS]::PTFSCompleteCleanup End\n"));
}
