#include "PTFileSystem.h"
#include "init.h"
#include "util.h"
#include "except.h"
#include "dispatch.h"
#include <wdm.h>

NTSTATUS
DriverEntry(
	IN PDRIVER_OBJECT  pDriverObject,
	IN PUNICODE_STRING RegistryPath
);

VOID
PTFSUnload(
	IN PDRIVER_OBJECT pDriverObject
);

#ifdef ALLOC_PRAGMA
#pragma alloc_text (INIT, DriverEntry)
#pragma alloc_text (PAGE, PTFSUnload)
#endif

FAST_IO_DISPATCH					g_FastIoDispatch;
NPAGED_LOOKASIDE_LIST			g_PTFSIrpEntryLookasideList; 

LOOKASIDE_LIST_EX					g_PTFSCCBLookasideList;
LOOKASIDE_LIST_EX					g_PTFSFCBLookasideList;
LOOKASIDE_LIST_EX					g_PTFSEResourceLookasideList;
BOOLEAN										g_FixFileNameForReparseMountPoint = FALSE;

ULONG											g_PTFSMdlSafePriority = 0;
ULONG											g_Debug = 0;

PFN_FsRtlCheckLockForOplockRequest* g_pfn_FsRtlCheckLockForOplockRequest = NULL;
PFN_FsRtlAreThereWaitingFileLocks* g_pfn_FsRtlAreThereWaitingFileLocks = NULL;

FAST_IO_CHECK_IF_POSSIBLE	PTFSFastIoCheckIfPossible;
BOOLEAN
PTFSFastIoCheckIfPossible(
	IN PFILE_OBJECT pFileObject,
	IN PLARGE_INTEGER pliFileOffset, 
	IN ULONG ulLength,
	IN BOOLEAN bWait, 
	IN ULONG ulLockKey,
	IN BOOLEAN bCheckForReadOperation,
	OUT PIO_STATUS_BLOCK pIoStatus,
	IN PDEVICE_OBJECT pDeviceObject
) {
	UNREFERENCED_PARAMETER(pFileObject);
	UNREFERENCED_PARAMETER(pliFileOffset);
	UNREFERENCED_PARAMETER(ulLength);
	UNREFERENCED_PARAMETER(bWait);
	UNREFERENCED_PARAMETER(ulLockKey);
	UNREFERENCED_PARAMETER(bCheckForReadOperation);
	UNREFERENCED_PARAMETER(pIoStatus);
	UNREFERENCED_PARAMETER(pDeviceObject);

	KdPrint(("[PTFS]::PTFSFastIoCheckIfPossible"));
	return FALSE;
}

FAST_IO_ACQUIRE_FILE PTFSAcquireForCreateSection;

VOID 
PTFSAcquireForCreateSection(
	IN PFILE_OBJECT pFileObject
) 
{
	PPTFSCCB pCcb = (PPTFSCCB)pFileObject->FsContext2;
	if (pCcb != NULL)
	{
		PTFSFcbLock(pCcb->pFcb, FALSE);
		KeLeaveCriticalRegion();
	}

	KdPrint(("[PTFS]::PTFSAcquireForCreateSection"));
}

FAST_IO_RELEASE_FILE PTFSReleaseForCreateSection;

VOID 
PTFSReleaseForCreateSection(
	IN PFILE_OBJECT pFileObject
) 
{
	PPTFSCCB pCcb = (PPTFSCCB)pFileObject->FsContext2;
	
	if (pCcb != NULL)
	{
		KeEnterCriticalRegion();
		PTFSFcbUnlock(pCcb->pFcb);
	}

	KdPrint(("[PTFS]::PTFSReleaseForCreateSection"));
}


NTSTATUS
PTFSFilterCallbackAcquireForCreateSection(
	IN PFS_FILTER_CALLBACK_DATA	pCallbackData,
	OUT PVOID* ppCompletionContext
) 
{
	PPTFSCCB pCcb = NULL;

	UNREFERENCED_PARAMETER(ppCompletionContext);

	KdPrint(("[PTFS]::PTFSFilterCallbackAcquireForCreateSection"));

	pCcb = pCallbackData->FileObject->FsContext2;

	if (pCcb != NULL)
	{
		PTFSFcbLock(pCcb->pFcb, FALSE);
		KeLeaveCriticalRegion();
	}

	if (pCcb == NULL || pCallbackData->Parameters.AcquireForSectionSynchronization.SyncType !=SyncTypeCreateSection)
		return STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY;
	else if (pCcb->pFcb->ShareAccess.Writers == 0)
		return STATUS_FILE_LOCKED_WITH_ONLY_READERS;
	else 
		return STATUS_FILE_LOCKED_WITH_WRITERS;
}

BOOLEAN
CreateLookaside(
	IN LOOKASIDE_LIST_EX* pCache, 
	IN size_t cbElement) 
{
	NTSTATUS Status = ExInitializeLookasideListEx(pCache, NULL, NULL, NonPagedPool, 0, cbElement, TAG, 0);

	if (!NT_SUCCESS(Status)) 
	{
		KdPrint(("[PTFS]::CreateLookaside Failed to ExInitializeLookasideListEx Status (0x%x)", Status));
		return FALSE;
	}

	return TRUE;
}

NTSTATUS
PTFSDispatchRoutine(
	IN PDEVICE_OBJECT pDeviceObject,
	IN PIRP pIrp
)
{
	BOOLEAN bAtIrqlPassiveLevel = FALSE;
	BOOLEAN bIsTopLevelIrp = FALSE;
	NTSTATUS Status = STATUS_UNSUCCESSFUL;

	__try {

		__try {

			bAtIrqlPassiveLevel = (KeGetCurrentIrql() == PASSIVE_LEVEL);

			if (bAtIrqlPassiveLevel) 
				FsRtlEnterFileSystem();

			if (!IoGetTopLevelIrp()) 
			{
				bIsTopLevelIrp = TRUE;
				IoSetTopLevelIrp(pIrp);
			}

			Status = DispatchRoutine(pDeviceObject, pIrp);

		}
		__except (PTFSExceptionFilter(pIrp, GetExceptionInformation())) 
		{
			Status = PTFSExceptionHandler(pDeviceObject, pIrp, GetExceptionCode());
		}
	}
	__finally 
	{
		if (bIsTopLevelIrp)
			IoSetTopLevelIrp(NULL);

		if (bAtIrqlPassiveLevel)
			FsRtlExitFileSystem();
	}

	return Status;
}

VOID 
CleanupGlobalDiskDevice(
	IN PPTFS_GLOBAL pPTFSGlobal) 
{
	WCHAR symbolicLinkBuf[] = PTFS_GLOBAL_SYMBOLIC_LINK_NAME;
	UNICODE_STRING symbolicLinkName;

	KeSetEvent(&pPTFSGlobal->KillDeleteDeviceEvent, 0, FALSE);

	RtlInitUnicodeString(&symbolicLinkName, symbolicLinkBuf);
	IoDeleteSymbolicLink(&symbolicLinkName);

	IoUnregisterFileSystem(pPTFSGlobal->pFsDiskDeviceObject);

	IoDeleteDevice(pPTFSGlobal->pFsDiskDeviceObject);
	IoDeleteDevice(pPTFSGlobal->pDeviceObject);
	ExDeleteResourceLite(&pPTFSGlobal->Resource);
	ExDeleteResourceLite(&pPTFSGlobal->MountManagerLock);
}

VOID
PTFSUnload(
	IN PDRIVER_OBJECT pDriverObject
)
{
	PDEVICE_OBJECT pDeviceObject = pDriverObject->DeviceObject;
	PPTFS_GLOBAL pPTFSGlobal = NULL;

	KdPrint(("[PTFS]::PTFSUnload Start"));

	PAGED_CODE();

	pPTFSGlobal = pDeviceObject->DeviceExtension;
	if (GETIDENTIFIERTYPE(pPTFSGlobal) == PGL) 
	{
		KdPrint(("[PTFS]::PTFSUnload Delete Global DeviceObject"));
		CleanupGlobalDiskDevice(pPTFSGlobal);
	}

	ExDeleteNPagedLookasideList(&g_PTFSIrpEntryLookasideList);

	ExDeleteLookasideListEx(&g_PTFSCCBLookasideList);
	ExDeleteLookasideListEx(&g_PTFSFCBLookasideList);
	ExDeleteLookasideListEx(&g_PTFSEResourceLookasideList);

	KdPrint(("[PTFS]::PTFSUnload End"));
}

NTSTATUS
InitializeDriver(
	VOID
)
{

	if (!CreateLookaside(&g_PTFSCCBLookasideList, sizeof(PTFSCCB))) 
	{
		KdPrint(("[PTFS]::CreateLookaside Failed CCB"));
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	if (!CreateLookaside(&g_PTFSFCBLookasideList, sizeof(PTFSFCB))) 
	{
		KdPrint(("[PTFS]::CreateLookaside Failed FCB"));
		ExDeleteLookasideListEx(&g_PTFSCCBLookasideList);
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	if (!CreateLookaside(&g_PTFSEResourceLookasideList,sizeof(ERESOURCE))) 
	{
		KdPrint(("[PTFS]::CreateLookaside Failed Resource"));
		ExDeleteLookasideListEx(&g_PTFSCCBLookasideList);
		ExDeleteLookasideListEx(&g_PTFSFCBLookasideList);
		return STATUS_INSUFFICIENT_RESOURCES;
	}

	g_FixFileNameForReparseMountPoint = !RtlIsNtDdiVersionAvailable(0x0A000005);
	return STATUS_SUCCESS;
}

NTSTATUS 
DriverEntry(
	IN PDRIVER_OBJECT  pDriverObject, 
	IN PUNICODE_STRING pRegistryPath
	)
{
	NTSTATUS						status = STATUS_SUCCESS;
	FS_FILTER_CALLBACKS	FilterCallbacks;
	PPTFS_GLOBAL				pPTFSGlobal = NULL;

	UNREFERENCED_PARAMETER(pRegistryPath);

	KdPrint(("[PTFS]::DriverEntry Start"));

	status = PTFSCreateGlobalDiskDevice(pDriverObject, &pPTFSGlobal);

	if (status != STATUS_SUCCESS) 
	{
		KdPrint(("[PTFS]::DriverEntry Failed to PTFSCreateGlobalDiskDevice status[0x%x]", status));
		return status;
	}

	pDriverObject->MajorFunction[IRP_MJ_CREATE] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_CLOSE] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_CLEANUP] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_FILE_SYSTEM_CONTROL] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_DIRECTORY_CONTROL] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_QUERY_INFORMATION] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_SET_INFORMATION] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_QUERY_VOLUME_INFORMATION] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_SET_VOLUME_INFORMATION] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_READ] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_WRITE] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_FLUSH_BUFFERS] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_SHUTDOWN] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_PNP] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_LOCK_CONTROL] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_QUERY_SECURITY] = PTFSDispatchRoutine;
	pDriverObject->MajorFunction[IRP_MJ_SET_SECURITY] = PTFSDispatchRoutine;

	pDriverObject->DriverUnload	= PTFSUnload;

	RtlZeroMemory(&g_FastIoDispatch, sizeof(FAST_IO_DISPATCH));

	g_FastIoDispatch.SizeOfFastIoDispatch = sizeof(FAST_IO_DISPATCH);
	g_FastIoDispatch.FastIoCheckIfPossible = PTFSFastIoCheckIfPossible;
	g_FastIoDispatch.FastIoRead = FsRtlCopyRead;
	g_FastIoDispatch.FastIoWrite = FsRtlCopyWrite;
	g_FastIoDispatch.AcquireFileForNtCreateSection = PTFSAcquireForCreateSection;
	g_FastIoDispatch.ReleaseFileForNtCreateSection = PTFSReleaseForCreateSection;
	g_FastIoDispatch.MdlRead = FsRtlMdlReadDev;
	g_FastIoDispatch.MdlReadComplete = FsRtlMdlReadCompleteDev;
	g_FastIoDispatch.PrepareMdlWrite = FsRtlPrepareMdlWriteDev;
	g_FastIoDispatch.MdlWriteComplete = FsRtlMdlWriteCompleteDev;

	pDriverObject->FastIoDispatch = &g_FastIoDispatch;

//	ExInitializeDriverRuntime(DrvRtPoolNxOptIn);

	if (RtlIsNtDdiVersionAvailable(NTDDI_WIN8))
	{
		UNICODE_STRING SystemRoutineName;
		g_PTFSMdlSafePriority = MdlMappingNoExecute;

		RtlInitUnicodeString(&SystemRoutineName, L"FsRtlCheckLockForOplockRequest");
		g_pfn_FsRtlCheckLockForOplockRequest = (PFN_FsRtlCheckLockForOplockRequest*)MmGetSystemRoutineAddress(&SystemRoutineName);

		RtlInitUnicodeString(&SystemRoutineName, L"FsRtlAreThereWaitingFileLocks");
		g_pfn_FsRtlAreThereWaitingFileLocks = (PFN_FsRtlAreThereWaitingFileLocks*)MmGetSystemRoutineAddress(&SystemRoutineName);
	}

	ExInitializeNPagedLookasideList( &g_PTFSIrpEntryLookasideList, NULL, NULL, 0, sizeof(IRP_ENTRY), TAG, 0);
    RtlZeroMemory(&FilterCallbacks, sizeof(FS_FILTER_CALLBACKS));

	FilterCallbacks.SizeOfFsFilterCallbacks = sizeof(FS_FILTER_CALLBACKS);
	FilterCallbacks.PreAcquireForSectionSynchronization = PTFSFilterCallbackAcquireForCreateSection;

	status = FsRtlRegisterFileSystemFilterCallbacks(pDriverObject, &FilterCallbacks);

	if (!NT_SUCCESS(status)) 
	{
		CleanupGlobalDiskDevice(pPTFSGlobal);
		KdPrint(("[PTFS]::DriverEntry Failed to FsRtlRegisterFileSystemFilterCallbacks status[0x%x]", status));
		return status;
	}

	status = InitializeDriver();

	if (STATUS_SUCCESS != status)
	{
		CleanupGlobalDiskDevice(pPTFSGlobal);
	}

	return status;
}

